package ru.yandex.msearch;

import java.io.IOException;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.NativePositiveLong2IntHashMap;

public class NativeKeysDocIdCachingPrimaryKeyPart extends PrimaryKeyPartBase {
    private static final int MAP_ENTRY_SIZE = 48;
    private final NativeUnboxedKeysSet cache;
    private final AtomicInteger keysSize = new AtomicInteger(0);

    private static class NativeUnboxedKeysSet
        extends NativePositiveLong2IntHashMap
    {
        @Override
        protected int keyHashCode(final long key) {
            return NativeCacheKey.hashCode(key);
        }

        @Override
        protected boolean keysEquals(final long a, final long b) {
            return NativeCacheKey.equals(a, b);
        }

        @Override
        protected void freeKey(final long key) {
            NativeCacheKey.free(key);
        }
    }

    public NativeKeysDocIdCachingPrimaryKeyPart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> queueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            queueIds,
            version,
            config);
        cache = new NativeUnboxedKeysSet();
    }

    @Override
    protected void cleanup() {
        cache.close();
    }

    @Override
    public int indexDocument(final HTMLDocument doc,
        final Analyzer analyzer)
        throws IOException
    {
        //reduntant check
        final PrimaryKey primaryKey = doc.primaryKey();
        final long nativeKey = primaryKey.nativeCacheKey();
        if (cache.contains(nativeKey)) {
            throw new CorruptIndexException(
                "Document for key: " + primaryKey
                + " is already in the part");
        }
        int docId = super.indexDocument(doc, analyzer);
        if (!cache.put(nativeKey, docId)) {
            throw new CorruptIndexException(
                "Document for key: " + primaryKey
                + " is already in the part after first cache check"
                + ". Lock failed?");
        } else {
            //null key for cloning
            primaryKey.setNativeCacheKey(0);
        }
        keysSize.addAndGet(NativeCacheKey.size(nativeKey));
        return docId;
    }

    @Override
    public boolean hasPrimaryKey(final PrimaryKey primaryKey,
        final PrimaryKeySearcher primaryKeySearcher)
        throws IOException
    {
        return cache.contains(primaryKey.nativeCacheKey());
    }

    @Override
    public Document getDocumentByPrimaryKey(final PrimaryKey primaryKey)
        throws IOException
    {
        long docId = cache.get(primaryKey.nativeCacheKey());
        if (docId == cache.NULL_VALUE) {
            throw new CorruptIndexException(
                "Not docId found in cache for key: " + primaryKey);
        }
        Document doc = document((int) docId);
        if (doc == null) {
            throw new CorruptIndexException(
                "Reader returned null for docId: " + docId + " with "
                    + "primaryKey: " + primaryKey);
        }
        if (!primaryKey.extractAndCompare(doc)) {
            throw new CorruptIndexException(
                "reader.document(" + docId + ").primeryKey differs from cached "
                    + "value. Expected: " + primaryKey
                    + ", doc found: " + doc);
        }
        return doc;
    }

    @Override
    public void removePrimaryKey(final PrimaryKey primaryKey) {
        cache.remove(primaryKey.nativeCacheKey());
    }

    @Override
    public boolean deleteDocument(final PrimaryKey primaryKey)
        throws IOException
    {
        long docId = cache.remove(primaryKey.nativeCacheKey());
        if (docId == cache.NULL_VALUE) {
            return false;
        } else {
            int keySize = NativeCacheKey.size(primaryKey.nativeCacheKey());
            keysSize.addAndGet(-keySize);
            deleteDocuments(primaryKey.queryProducer(), (int) docId);
            return true;
        }
    }

    @Override
    public long payloadSize() {
        return super.payloadSize() + keysSize.get() + cache.sizeInBytes();
    }
}
