package ru.yandex.msearch;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.protocol.HttpContext;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.MultiTermQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.Version;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.MergeFuncFactory;
import ru.yandex.msearch.collector.aggregate.AggregatorFactory;
import ru.yandex.msearch.collector.docprocessor.CompositeDocProcessor;
import ru.yandex.msearch.collector.docprocessor.DocProcessor;
import ru.yandex.msearch.collector.docprocessor.ModuleFieldsAggregator;
import ru.yandex.msearch.collector.group.GroupFuncFactory;
import ru.yandex.msearch.collector.group.NullGroupFunc;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.msearch.collector.postfilter.CompositePostFilter;
import ru.yandex.msearch.collector.postfilter.PostFilter;
import ru.yandex.msearch.collector.postfilter.PostFilterFactory;
import ru.yandex.msearch.collector.sort.NullSortFunc;
import ru.yandex.msearch.collector.sort.SortFuncFactory;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.queryParser.MultiFieldQueryParser;
import ru.yandex.queryParser.QueryParserFactory;
import ru.yandex.queryParser.YandexQueryParserFactory;
import ru.yandex.search.YandexScorerFactory;
import ru.yandex.search.prefix.Prefix;

public class NewSearchRequest extends SearchRequestBase {
    private static final CollectionParser<
        String,
        List<String>,
        Exception>
        GET_FIELDS_PARSER =
        new CollectionParser<>(NonEmptyValidator.INSTANCE, ArrayList::new);

    private ProcessorRequestContext processorContext;

    public NewSearchRequest(
        final CgiParams params,
        final HttpContext context,
        final Index index,
        final DatabaseConfig config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory)
        throws HttpException
    {
        super(new HttpRequestContext(context), index, config);

        this.debug = params.getBoolean("debug", false);

        sortDirection = params.getBoolean("asc", false);
        offset = params.getInt("offset", 0);
        length = params.get(
            "length",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        memoryLimit = params.get(
            "memory-limit",
            Long.MAX_VALUE,
            LongMemorySizeParser.INSTANCE);

        Set<String> generatedFields = new LinkedHashSet<>();

        //create scorer factory before docProcessor to allow
        //docprocessing on score fields
        scorerFactory =
            YandexScorerFactory.create(params.getString("scorer", null), config);
        if (scorerFactory != null) {
            generatedFields.addAll(scorerFactory.outFields());
        }

        this.prefixes = params.getAll(
            "prefix",
            Collections.emptySet(),
            prefixParser(),
            new LinkedHashSet<>());

        processorContext =
            new ProcessorRequestContext(prefixes, index, ctx, fieldToIndex, params);
        docProcessor =
            extractDocProcessor(config, generatedFields, params, processorContext);

        if (docProcessor != null) {
            this.dpReadOutFields(
                params.getBoolean("dp-read-out-fields", false));
        }

        group = params.get(
            "group",
            NullGroupFunc.INSTANCE,
            x -> GroupFuncFactory.INSTANCE.apply(x, fieldToIndex));
        for (String field: group.loadFields()) {
            if (!generatedFields.contains(field)) {
                checkStoredField(field);
            }
        }

        aggregator = params.get(
            "aggregate",
            null,
            x -> AggregatorFactory.INSTANCE.apply(x, fieldToIndex));
        if (aggregator != null) {
            for (String field: aggregator.loadFields()) {
                if (!generatedFields.contains(field)) {
                    checkStoredField(field);
                }
            }
            generatedFields.addAll(aggregator.outFields());
        }

        sort = params.get(
            "sort",
            NullSortFunc.INSTANCE,
            x -> SortFuncFactory.INSTANCE.apply(x, fieldToIndex));
        for (String field: sort.loadFields()) {
            if (!generatedFields.contains(field)) {
                checkStoredField(field);
            }
        }

        postFilter = extractPostFilter(
            config,
            generatedFields,
            params,
            fieldToIndex);

        GetFieldsConfig getFieldsConfig =
            extractGetFieldsConfig(config, generatedFields, params);
        getFields = getFieldsConfig.fields();
        skipNulls = getFieldsConfig.skipNulls();

        if (getFields.isEmpty()
            && (aggregator == null || aggregator.outFields().isEmpty())
            && length > 0)
        {
            throw new BadRequestException("no 'get' field specified");
        }

        if (group == NullGroupFunc.INSTANCE) {
            mergeFunc = null;
        } else {
            mergeFunc = MergeFuncFactory.create(
                params.getString("merge_func", null), primaryKey() != null);
        }
        outerGroup = params.get(
            "outer",
            null,
            x -> outerGroupFunctionFactory.apply(x, fieldToIndex));
        if (outerGroup != null) {
            for (String field: outerGroup.loadFields()) {
                if (!generatedFields.contains(field)) {
                    checkStoredField(field);
                }
            }
        }

        collectorName = params.getString("collector", null);

        String pruningGroupField =
            params.getString("pruning-group-field", null);
        if (pruningGroupField != null) {
            this.pruningGroupField = StringHelper.intern(pruningGroupField);
        }

        this.forcePruningGroupField =
            params.getBoolean("force-pruning-group-field", false);

        queries =
            parseQueries(config, processorContext, params, scorerFactory, prefixes);

        userIdField = params.getString("user-id-field", null);
        userIdTerm = params.getString("user-id-term", null);

        reverseTraverse = params.getBoolean("reverse-traverse", !sortDirection);

        updatePrefixActivity =
            params.getBoolean("update-prefix-activity", false);

        syncSearcher =
            params.getBoolean("sync-searcher", config.syncSearcherDefault());

        earlyInterrupt =
            params.getBoolean("early-interrupt", config.earlyInterruptDefault());

    }

    public static List<PrefixedQuery> parseQueries(
        final DatabaseConfig config,
        final ProcessorRequestContext context,
        final CgiParams params,
        final QueryParserFactory queryParserFactory,
        final Collection<Prefix> prefixes)
        throws HttpException
    {
        boolean checkCopyness =
            params.getBoolean("check-copyness", config.checkCopyness());
        String text = params.get("text", NonEmptyValidator.INSTANCE);
        List<String> prefixlessFields = params.getAll("prefixless-field");
        PrefixingAnalyzerWrapper analyzer;
        if (prefixlessFields.isEmpty()) {
            analyzer = context.index().searchAnalyzer();
        } else {
            analyzer = DefaultAnalyzerFactory.SEARCH.apply(config);

            for (String field: prefixlessFields) {
                analyzer.addSkipField(field);
            }
        }
        QueryParser parser =
            createParser(config, params, analyzer, queryParserFactory);

        context.queryParser(parser);

        List<PrefixedQuery> queries =
            new ArrayList<>(Math.max(prefixes.size(), 1));
        if (prefixes.isEmpty()) {
            if (checkCopyness) {
                try {
                    context.index().checkShardCopied((Prefix) null);
                } catch (IOException e) {
                    throw new ServiceUnavailableException(e);
                }
            }
            analyzer.resetLastPrefixedField();
            try {
                Query query = parser.parse(text);
                if (analyzer.lastPrefixedField() != null) {
                    throw new BadRequestException(
                        "No prefix specified, but prefixed field "
                            + analyzer.lastPrefixedField()
                            + " used in query");
                }
                if (config.logParsedRequest()) {
                    if (context.ctx().logger().isLoggable(Level.INFO)) {
                        context.ctx().logger().info(
                            "Non-prefixed request parsed: " + query);
                    }
                }
                queries.add(new PrefixedQuery(query, null));
            } catch (Exception e) {
                throw new BadRequestException(
                    "Failed to parse query '" + text + '\'', e);
            }
        } else {
            for (Prefix prefix: prefixes) {
                if (checkCopyness) {
                    try {
                        context.index().checkShardCopied(prefix);
                    } catch (IOException e) {
                        throw new ServiceUnavailableException(e);
                    }
                }
                analyzer.setPrefix(prefix.toString());
                Query query = null;
                try {
                    query = parser.parse(text);
                    // XXX: Do not remove this log record!
                    // It exposes the bug in WildcardQuery
                    if (config.logParsedRequest()) {
                        if (context.ctx().logger().isLoggable(Level.INFO)) {
                            context.ctx().logger().info(
                                "Request parsed: " + query);
                        }
                    }
                } catch (Exception e) {
                    throw new BadRequestException(
                        "Failed to parse query '" + text + '\'', e);
                }
                queries.add(new PrefixedQuery(query, prefix));
            }
        }
        return queries;
    }

    public static PostFilter extractPostFilter(
        final DatabaseConfig config,
        final Set<String> generatedFields,
        final CgiParams params,
        final FieldToIndex fieldToIndex)
        throws BadRequestException
    {
        List<PostFilter> filters = params.getAll(
            "postfilter",
            Collections.emptyList(),
            x -> PostFilterFactory.INSTANCE.apply(x, fieldToIndex),
            new ArrayList<>());
        for (PostFilter filter: filters) {
            for (String field: filter.loadFields()) {
                if (!generatedFields.contains(field)) {
                    checkStoredField(config, field);
                }
            }
        }
        return CompositePostFilter.create(filters);
    }

    public static DocProcessor extractDocProcessor(
        final DatabaseConfig config,
        final Set<String> generatedFields,
        final CgiParams params,
        final ProcessorRequestContext processorContext)
        throws BadRequestException
    {
        List<DocProcessor> docProcessors = params.getAll(
            "dp",
            Collections.emptyList(),
            x -> config.docProcessorFactory().apply(x, processorContext),
            new ArrayList<>());
        ModuleFieldsAggregator fieldsAggregator = new ModuleFieldsAggregator();
        for (DocProcessor docProcessor: docProcessors) {
            docProcessor.apply(fieldsAggregator);
        }

        for (String field: fieldsAggregator.loadFields()) {
            checkStoredField(config, field);
        }

        generatedFields.addAll(fieldsAggregator.outFields());

        return CompositeDocProcessor.create(docProcessors);
    }

    public static GetFieldsConfig extractGetFieldsConfig(
        final DatabaseConfig config,
        final Set<String> generatedFields,
        final CgiParams params)
        throws BadRequestException
    {
        List<String> get = params.getAll(
            "get",
            Collections.emptyList(),
            GET_FIELDS_PARSER);
        Set<String> getFields = new LinkedHashSet<>();
        boolean hasAsterisk = false;
        for (String value: get) {
            char c = value.charAt(0);
            if (c == '*'
                && (value.length() == 1
                || (value.length() == 2 && value.charAt(1) == '*')))
            {
                hasAsterisk = true;
                for (String field: config.knownFields()) {
                    if (config.fieldConfig(field).store()
                        && !field.startsWith("__"))
                    {
                        getFields.add(StringHelper.intern(field));
                    }
                }
                if (value.length() == 2) {
                    getFields.addAll(generatedFields);
                }
            } else if (c == '-') {
                getFields.remove(value.substring(1));
            } else {
                if (!generatedFields.contains(value)) {
                    checkStoredField(config, value);
                }
                getFields.add(StringHelper.intern(value));
            }
        }
        return new GetFieldsConfig(
            getFields,
            params.getBoolean("skip-nulls", hasAsterisk),
            hasAsterisk);
    }

    private static QueryParser selectParser(
        final DatabaseConfig config,
        final CgiParams params,
        final Analyzer analyzer,
        final QueryParserFactory factory)
        throws HttpException
    {
        String scope = params.getString("scope", null);
        if (scope == null) {
            return factory.create(
                Version.LUCENE_40,
                null,
                analyzer);
        } else {
            String[] fields = scope.split(",");
            for (String field: fields) {
                if (config.fieldConfig(field) == null) {
                    throw new BadRequestException("Unknown field: " + field);
                }
            }
            return new MultiFieldQueryParser(factory, analyzer, fields);
        }
    }

    public static void adjustParser(
        final QueryParser parser,
        final DatabaseConfig config,
        final CgiParams params)
        throws HttpException
    {
        parser.setAllowLeadingWildcard(true);
        parser.setAnalyzeRangeTerms(true);
        parser.setAutoGeneratePhraseQueries(true);
        parser.setLowercaseExpandedTerms(
            params.getBoolean(
                "lowercase-expanded-terms",
                config.lowercaseExpandedTerms()));
        parser.setReplaceEExpandedTerms(
            params.getBoolean(
                "replace-ee-expanded-terms",
                false));
        parser.setDefaultOperator(
            params.getEnum(
                QueryParser.Operator.class,
                "default-operator",
                config.defaultOperator()));
        boolean rangesOverBoolean =
            params.getBoolean("ranges-over-boolean", false);
        if (rangesOverBoolean) {
            parser.setMultiTermRewriteMethod(
                MultiTermQuery.CONSTANT_SCORE_BOOLEAN_QUERY_REWRITE);
        }
    }

    public static QueryParser createParser(
        final DatabaseConfig config,
        final CgiParams params,
        final Analyzer analyzer)
        throws HttpException
    {
        return createParser(
            config, params, analyzer, new YandexQueryParserFactory(config));
    }

    public static QueryParser createParser(
        final DatabaseConfig config,
        final CgiParams params,
        final Analyzer analyzer,
        final QueryParserFactory factory)
        throws HttpException
    {
        QueryParser parser = selectParser(
            config,
            params,
            analyzer,
            factory);
        adjustParser(parser, config, params);
        return parser;
    }
}
