package ru.yandex.msearch;

import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.server.sync.JsonContentProducerWriter;

import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServerException;

import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.uri.CgiParams;

public class NumDocsHandler implements HttpRequestHandler {
    private final DatabaseManager dbManager;

    public NumDocsHandler(final DatabaseManager dbManager) {
        this.dbManager = dbManager;
    }

    @Override
    public void handle(final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        Logger logger =
            (Logger) context.getAttribute(BaseHttpServer.LOGGER);
        logger.info("Incoming numdocs request");

        CgiParams params = new CgiParams(request);
        Index index = dbManager.indexOrException(params, SearchHandler.BRE_GEN);

        final ShardStats[] result = new ShardStats[index.shardsCount()];
        final AtomicLong totalDocs = new AtomicLong();
        try {
            index.traverseShards(new ShardVisitor() {
                @Override
                public void visit(final int shardNo, final Shard shard)
                    throws IOException
                {
                    final ShardStats stats =
                        new ShardStats(shard.numDocsLong());
                    result[shardNo] = stats;
                    shard.traverseParts(new PartVisitor() {
                        @Override
                        public void visit(final AbstractPart part)
                            throws IOException
                        {
                            long numDocs = part.numDocsLong();
                            stats.add(numDocs);
                        }
                    });
                    totalDocs.addAndGet(stats.docs());
                }
            });
        } catch (IOException e) {
            throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR,
                "Failed to collect num docs", e);
        }
        EntityTemplate entity = new EntityTemplate(
            new JsonContentProducerWriter(
                new NumDocsProducer(totalDocs.get(), index, result),
                request));
        entity.setChunked(true);
        entity.setContentType(ContentType.APPLICATION_JSON.withCharset(
            CharsetUtils.acceptedCharset(request)).toString());
        response.setEntity(entity);
        response.setStatusCode(HttpStatus.SC_OK);
    }

    @Override
    public String toString() {
        return
            "Prints amount of documents in each shard and memory index part";
    }

    private static class ShardStats {
        private final List<Long> parts = new ArrayList<>();
        private long docs;

        public ShardStats(final long docs) {
            this.docs = docs;
        }

        public void add(final long numDocs) {
            parts.add(numDocs);
        }

        public long docs() {
            return docs;
        }

        public List<Long> parts() {
            return parts;
        }
    }

    private static class NumDocsProducer implements JsonValue {
        private final long totalDocs;
        private final Index index;
        private final ShardStats[] stats;

        public NumDocsProducer(final long totalDocs,
            final Index index, final ShardStats[] stats)
        {
            this.totalDocs = totalDocs;
            this.index = index;
            this.stats = stats;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("docs");
            writer.value(totalDocs);
            writer.key("shards");
            writer.startObject();
            for (int i = 0; i < index.shardsCount(); ++i) {
                writer.key(Integer.toString(i));
                writer.startObject();
                writer.key("docs");
                writer.value(stats[i].docs());
                writer.key("parts");
                writer.startArray();
                for (Long part: stats[i].parts()) {
                    writer.value(part);
                }
                writer.endArray();
                writer.endObject();
            }
            writer.endObject();
            writer.endObject();
        }
    }
}

