package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.Reader;
import java.io.IOException;
import java.util.HashSet;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;

public class PrefixingAnalyzerWrapper extends Analyzer {
  private String lastPrefixedField = null;
  private String fixedField = null;
  private String prefix = new String();
  private Analyzer origAnalyzer;
  private char separator;
  private HashSet<String> skipFields;
  private PrefixingFilter reusableStream;

  public PrefixingAnalyzerWrapper(Analyzer origAnalyzer) {
    this.origAnalyzer = origAnalyzer;
    this.reusableStream = null;
    skipFields = new HashSet<String>();
  }

  public String lastPrefixedField() {
      return lastPrefixedField;
  }

  public void resetLastPrefixedField() {
      lastPrefixedField = null;
  }

  public boolean prefixed(final String field) {
      if (skipFields.contains(field)) {
          return false;
      } else {
          lastPrefixedField = field;
          return true;
      }
  }

  public void setSeparator( char sep )
  {
    separator = sep;
    if( reusableStream != null ) reusableStream.setSeparator( sep );
  }

  public void setPrefix( String prefix )
  {
    this.prefix = prefix;
    if( reusableStream != null ) reusableStream.setPrefix( prefix );
  }

  public void addSkipField( String field )
  {
    skipFields.add( field );
  }

  public char getSeparator()
  {
    return separator;
  }

  public String getPrefix()
  {
    return prefix;
  }

  public void fixedField(final String fieldName) {
    fixedField = fieldName;
  }

  public String fixedField() {
    return fixedField;
  }

  private String selectField(final String fieldName) {
    if (fixedField == null) {
      return fieldName;
    } else {
      return fixedField;
    }
  }

  public TokenStream tokenStream(String fieldName, Reader reader) throws IOException {
    fieldName = selectField(fieldName);
    if (prefixed(fieldName)) {
        return new PrefixingFilter(origAnalyzer.tokenStream(fieldName,reader),
            prefix, separator);
    } else {
        return origAnalyzer.tokenStream(fieldName, reader);
    }
  }

  public TokenStream reusableTokenStream(String fieldName, Reader reader)
    throws IOException
  {
    fieldName = selectField(fieldName);
    if (prefixed(fieldName)) {
        if (reusableStream == null) {
            reusableStream = new PrefixingFilter(
                origAnalyzer.reusableTokenStream(fieldName,reader),
                prefix,
                separator);
        } else {
            reusableStream.setTokenStream(
                origAnalyzer.reusableTokenStream(fieldName,reader));
            reusableStream.setSeparator(separator);
            reusableStream.setPrefix(prefix);
        }
        return reusableStream;
    } else {
        return origAnalyzer.reusableTokenStream( fieldName, reader );
    }
  }

  /** Return the positionIncrementGap from the analyzer assigned to fieldName */
  public int getPositionIncrementGap(String fieldName) {
    return origAnalyzer.getPositionIncrementGap(fieldName);
  }

  public String toString() {
    return "PrefixingAnalyzerWrapper( " + separator + prefix + ")";
  }
}
