package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.ReusableTokenStream;

/**
 * Removes words that are too long or too short from the stream.
 */
public class PrefixingFilter extends ReusableTokenStream {

    private final int DEFAULT_BUFFER_SIZE = 1024;

    private String prefix;
    private int prefixLength;
    private char separator;

    private char[] buffer;

    private CharTermAttribute termAtt = null;

  /**
   * Build a filter that removes words that are too long or too
   * short from the text.
   */
  public PrefixingFilter(TokenStream in, String prefix, char separator)
  {
    super(in);
    this.prefix = prefix;
    this.separator = separator;
    prefixLength = prefix.length() + 1;
    buffer = new char[DEFAULT_BUFFER_SIZE];
    if( prefixLength > buffer.length ) growBuffer( prefixLength + DEFAULT_BUFFER_SIZE );
    prefix.getChars( 0, prefix.length(), buffer, 0 );
    buffer[prefix.length()] = separator;
  }

  protected void resetAttributes()
  {
//    termAtt = addAttribute(CharTermAttribute.class);
//    termAtt.setEmpty(0);
    termAtt = null;
  }

  public void setPrefix( String prefix )
  {
	this.prefix = prefix;
	prefixLength = prefix.length() + 1;
	if( buffer.length < prefixLength )
	    growBuffer( prefixLength + DEFAULT_BUFFER_SIZE );
	prefix.getChars( 0, prefix.length(), buffer, 0 );
        buffer[prefix.length()] = separator;
  }

  public void setSeparator( char separator )
  {
        this.separator = separator;
        buffer[prefix.length()] = separator;
  }

  private void growBuffer( int len )
  {
    int newLength = DEFAULT_BUFFER_SIZE + buffer.length;
	if( len < buffer.length ) return;
	if( len > newLength ) newLength += len;
	char[] newBuffer = new char[newLength];
        for (int i = 0; i < prefixLength; i++) {
            newBuffer[i] = buffer[i];
        }
        buffer = newBuffer;
  }

  @Override
  public final boolean incrementToken() throws IOException {
    if( super.incrementToken() ) {
        if (termAtt == null) {
            termAtt = addAttribute(CharTermAttribute.class);
        }
	int len = termAtt.length();
	if( len + prefixLength > buffer.length )
	    growBuffer( len + prefixLength + DEFAULT_BUFFER_SIZE );
	System.arraycopy( termAtt.buffer(), 0, buffer, prefixLength, len );
	termAtt.copyBuffer( buffer, 0, len + prefixLength );
	return true;
    }
    // reached EOS -- return false
    return false;
  }
}
