package ru.yandex.msearch;

import java.io.Closeable;
import java.io.UnsupportedEncodingException;

import java.nio.charset.StandardCharsets;

import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.search.TermQuery;

import ru.yandex.function.AbstractStringBuilderable;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;

import ru.yandex.util.string.StringUtils;

import ru.yandex.util.unicode.UnicodeUtil;

public abstract class PrimaryKey
    extends AbstractStringBuilderable
    implements Closeable
{
    private static final int HASH_PRIME = 31;
    private static final int CACHE_KEY_OBJECT_WEIGHT = 24;
    private static final Prefix FAKE_PREFIX = new LongPrefix(0L);

    protected final Prefix prefix;
    protected final Prefix comparePrefix;
    protected final DatabaseConfig config;
    protected QueryProducer queryProducer = null;
    protected CacheKey cacheKey = null;
    protected volatile long nativeCacheKey = 0;

    static class CacheKey {
        private final byte[] keyData;
        private int hashCode = -1;
        public CacheKey(final byte[] keyData) {
            this.keyData = keyData;
        }

        @Override
        public int hashCode() {
            if (hashCode == -1) {
                hashCode = 0;
                for(byte b : keyData) {
                    hashCode = HASH_PRIME * hashCode + b;
                }
            }
            return hashCode;
        }

        @Override
        public boolean equals(Object o) {
            CacheKey other = (CacheKey)o;
            return Arrays.equals(keyData, other.keyData);
        }

        public int weight() {
            return CACHE_KEY_OBJECT_WEIGHT + keyData.length;
        }
    }

    protected PrimaryKey(
        final Prefix prefix,
        final Prefix comparePrefix,
        final DatabaseConfig config)
    {
        this.prefix = prefix;
        this.comparePrefix = comparePrefix;
        this.config = config;
    }

    public Prefix prefix() {
        return prefix;
    }

    protected abstract Query constructQuery();

    public abstract boolean extractAndCompare(final Document doc);

    public Query query() {
        if (queryProducer == null) {
            queryProducer = SlimDownQueryHelper.slimDownQuery(
                constructQuery());
        }
        return queryProducer.produceQuery();
    }

    public QueryProducer queryProducer() {
        if (queryProducer == null) {
            queryProducer = SlimDownQueryHelper.slimDownQuery(
                constructQuery());
        }
        return queryProducer;
    }

    public void clearQuery() {
        if (queryProducer
            instanceof SlimDownQueryHelper.DeserializingQueryProducer)
        {
            ((SlimDownQueryHelper.DeserializingQueryProducer) queryProducer)
                .clearCached();
        }
        queryProducer = null;
    }

    public CacheKey cacheKey() {
        if (cacheKey == null) {
            cacheKey = createCacheKey();
        }
        return cacheKey;
    }

    public long nativeCacheKey() {
        if (nativeCacheKey == 0) {
            nativeCacheKey = createNativeCacheKey();
        }
        return nativeCacheKey;
    }

    public void setNativeCacheKey(final long key) {
        this.nativeCacheKey = key;
    }

    public void freeNativeCacheKey() {
        if (nativeCacheKey != 0) {
            NativeCacheKey.free(nativeCacheKey);
            nativeCacheKey = 0;
        }
    }

    public void close() {
        freeNativeCacheKey();
        clearQuery();
    }

    protected abstract CacheKey createCacheKey();
    protected abstract long createNativeCacheKey();

    public static PrimaryKey create(
        final Map<String, String> key,
        final Prefix prefix,
        final DatabaseConfig config)
    {
        if (key.size() == 1) {
            Map.Entry<String, String> entry = key.entrySet().iterator().next();
            return new BasicPrimaryKey(
                entry.getKey(),
                entry.getValue(),
                prefix,
                config);
        } else {
            return new MapPrimaryKey(key, prefix, config);
        }
    }

    protected Query constructTermQuery(final String name, final String value) {
        if (config.fieldConfigFast(name).prefixed()) {
            StringBuilder sb = new StringBuilder();
            prefix.toStringBuilder(sb);
            sb.append('#');
            sb.append(value);
            return new TermQuery(new Term(name, new String(sb)));
        } else {
            return new TermQuery(new Term(name, value));
        }
    }

    private static Prefix selectComparePrefix(
        final Prefix prefix,
        final DatabaseConfig config)
    {
        if (config.prefixedPrimaryKey()) {
            return prefix;
        } else {
            return FAKE_PREFIX;
        }
    }

    public static class BasicPrimaryKey extends PrimaryKey {
        private final String name;
        private final String value;

        public BasicPrimaryKey(
            final String name,
            final String value,
            final Prefix prefix,
            final DatabaseConfig config)
        {
            super(prefix, selectComparePrefix(prefix, config), config);
            this.name = name;
            this.value = value;
        }

        public String name() {
            return name;
        }

        public String value() {
            return value;
        }

        @Override
        public int hashCode() {
            return comparePrefix.hashCode()
                ^ name.hashCode()
                ^ value.hashCode();
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            prefix.toStringBuilder(sb);
            sb.append('#');
            sb.append('{');
            sb.append(name);
            sb.append('=');
            sb.append(value);
            sb.append('}');
        }

        @Override
        protected CacheKey createCacheKey() {
            int utf8Size = 0;
            int prefixSize = comparePrefix.sizeInBytes();
            int nameSize = UnicodeUtil.utf8Length(name, 0, name.length());
            int valueSize = UnicodeUtil.utf8Length(value, 0, value.length());
            utf8Size += prefixSize + 1;
            utf8Size += nameSize + 1;
            utf8Size += valueSize;

            byte[] utf8 = new byte[utf8Size];
            int pos = 0;
            comparePrefix.writeTo(utf8, pos);
            pos += prefixSize;
            utf8[pos++] = 0;
            UnicodeUtil.toUtf8(name, 0, name.length(), utf8, pos);
            pos += nameSize;
            utf8[pos++] = 0;
            UnicodeUtil.toUtf8(value, 0, value.length(), utf8, pos);
            return new CacheKey(utf8);
        }

        @Override
        protected long createNativeCacheKey() {
            int utf8Size = 0;
            int prefixSize = comparePrefix.sizeInBytes();
            int nameSize = UnicodeUtil.utf8Length(name, 0, name.length());
            int valueSize = UnicodeUtil.utf8Length(value, 0, value.length());
            //+1 for separate prefix \0 name \0 value
            utf8Size += prefixSize + 1;
            utf8Size += nameSize + 1;
            utf8Size += valueSize;
            final long nativeKey = NativeCacheKey.allocate(utf8Size);
            long keyPos = nativeKey;
            comparePrefix.writeTo(nativeKey);
            keyPos += prefixSize + 1;
            UnicodeUtil.toUtf8(name, 0, name.length(), keyPos);
            keyPos += nameSize + 1;
            UnicodeUtil.toUtf8(value, 0, value.length(), keyPos);
            return nativeKey;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof BasicPrimaryKey) {
                BasicPrimaryKey other = (BasicPrimaryKey) o;
                return comparePrefix.equals(other.comparePrefix)
                    && value.equals(other.value)
                    && name.equals(other.name);
            }
            return false;
        }

        @Override
        protected Query constructQuery() {
            return constructTermQuery(name, value);
        }

        @Override
        public boolean extractAndCompare(final Document doc) {
            return value.equals(doc.get(name));
        }
    }

    public static class MapPrimaryKey extends PrimaryKey {
        private final Map<String, String> key;

        public MapPrimaryKey(
            final Map<String, String> key,
            final Prefix prefix,
            final DatabaseConfig config)
        {
            super(prefix, selectComparePrefix(prefix, config), config);
            this.key = key;
        }

        public Map<String, String> key() {
            return key;
        }

        @Override
        public int hashCode() {
            return comparePrefix.hashCode() ^ key.hashCode();
        }

        @Override
        public void toStringBuilder(final StringBuilder sb) {
            prefix.toStringBuilder(sb);
            sb.append('#');
            sb.append(key);
        }

        private String compareString() {
            return StringUtils.concat(
                comparePrefix.toString(),
                '#',
                key.toString());
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof MapPrimaryKey) {
                MapPrimaryKey other = (MapPrimaryKey) o;
                return comparePrefix.equals(other.comparePrefix)
                    && key.equals(other.key);
            }
            return false;
        }

        @Override
        protected CacheKey createCacheKey() {
            return new CacheKey(
                compareString().getBytes(StandardCharsets.UTF_8));
        }

        @Override
        protected long createNativeCacheKey() {
            final byte[] data =
                compareString().getBytes(StandardCharsets.UTF_8);
            final long nativeKey = NativeCacheKey.allocate(data);
            return nativeKey;
        }

        @Override
        protected Query constructQuery() {
            BooleanQuery booleanQuery = new BooleanQuery(true);
            for (Map.Entry<String, String> entry: key.entrySet()) {
                booleanQuery.add(
                    constructTermQuery(entry.getKey(), entry.getValue()),
                    BooleanClause.Occur.MUST);
            }
            return booleanQuery;
        }

        @Override
        public boolean extractAndCompare(final Document doc) {
            for (Map.Entry<String, String> entry: key.entrySet()) {
                 if (!entry.getValue().equals(doc.get(entry.getKey()))) {
                    return false;
                 }
            }
            return true;
        }
    }
}

