package ru.yandex.msearch;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;

import ru.yandex.msearch.collector.BaseUpdatingCollector;
import ru.yandex.msearch.collector.KeepPrefixUpdatingCollector;
import ru.yandex.msearch.collector.MapCollector;
import ru.yandex.msearch.collector.PlainCollector;
import ru.yandex.msearch.collector.UpdatingCollector;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.json.fieldfunction.ConditionsAccessor;
import ru.yandex.search.json.fieldfunction.FieldFunction;
import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.prefix.Prefix;

//import org.apache.lucene.queryParser.ParseException;

public abstract class PrimaryKeyPartBase extends PrimaryKeyPart {
    private class Deleter implements PartVisitor {
        private final QueryProducer query;
        private final Prefix prefix;

        public Deleter(final QueryProducer query, final Prefix prefix) {
            this.query = query;
            this.prefix = prefix;
        }

        @Override
        public void visit(final AbstractPart part) throws IOException {
            if (part instanceof PrimaryKeyPart) {
                Searcher searcher = part.getSearcher();
                try
                {
                    final PrimaryKeyPart primaryKeyPart = (PrimaryKeyPart) part;
                    MapCollector col = new MapCollector(config.primaryKey());
                    searcher.searcher().search(
                        query.produceQuery(),
                        col,
                        true);
                    for (Map<String, String> primaryKey
                        : col.documents().values())
                    {
                        final PrimaryKey key =
                            PrimaryKey.create(primaryKey, prefix, config);
                        try {
                            indexAccessor().primaryKeyAtomicOp(key,
                                new IndexAccessor.AtomicOp() {
                                    @Override
                                    public void run() throws IOException {
                                        primaryKeyPart.deleteDocument(key);
                                    }
                                });
                        } finally {
                            key.close();
                        }
                    }
                } finally {
                    searcher.free();
                }
            }
        }
    }

    private class Updater implements PartVisitor {
        private final QueryProducer query;
        private final Analyzer indexAnalyzer;
        private final Prefix prefix;
        private final long phantomQueueId;
        private final String queueName;
        private final boolean nonPrefixUpdate;
        private final ConditionsAccessor condAccessor;
        private final Map<String, FieldFunction> fields;
        private final Logger logger;
        public int totalFound;

        private final boolean orderIndependentUpdate;

        public Updater(
            final QueryProducer query,
            final PrefixingAnalyzerWrapper indexAnalyzer,
            final Prefix prefix,
            final long phantomQueueId,
            final String queueName,
            final boolean nonPrefixUpdate,
            final ConditionsAccessor condAccessor,
            final Map<String, FieldFunction> fields,
            final boolean orderIndependentUpdate,
            final Logger logger)
        {
            this.query = query;
            this.indexAnalyzer = indexAnalyzer;
            this.prefix = prefix;
            this.phantomQueueId = phantomQueueId;
            this.queueName = queueName;
            this.condAccessor = condAccessor;
            this.fields = fields;
            this.logger = logger;
            this.nonPrefixUpdate = nonPrefixUpdate;
            this.orderIndependentUpdate = orderIndependentUpdate;
        }

        @Override
        public void visit(final AbstractPart part) throws IOException {
            if (part instanceof PrimaryKeyPart) {
                Searcher searcher = part.getSearcher();
                try
                {
                    final PrimaryKeyPart primaryKeyPart = (PrimaryKeyPart) part;

                    BaseUpdatingCollector col;
                    if (nonPrefixUpdate) {
                        col =
                            new KeepPrefixUpdatingCollector(
                                fields,
                                phantomQueueId,
                                queueName,
                                condAccessor,
                                config,
                                orderIndependentUpdate,
                                logger);
                    } else {
                        col =
                            new UpdatingCollector(
                                fields,
                                prefix,
                                phantomQueueId,
                                queueName,
                                condAccessor,
                                orderIndependentUpdate,
                                config,
                                logger);
                    }
                    searcher.searcher().search(
                        query.produceQuery(),
                        col,
                        true);
                    List<HTMLDocument> found = col.documents();
                    totalFound += found.size();
                    for (final HTMLDocument doc: found) {
                        try {
                            final PrimaryKey key = doc.primaryKey();
                            final Analyzer docAnalyzer;
                            if (nonPrefixUpdate) {
                                docAnalyzer = analyzerProvider().indexAnalyzer(doc.prefix());
                            } else {
                                docAnalyzer = indexAnalyzer;
                            }
                            indexAccessor().primaryKeyAtomicOp(key,
                                new IndexAccessor.AtomicOp() {
                                    @Override
                                    public void run() throws IOException {
                                        if (primaryKeyPart.deleteDocument(key)) {
                                            indexDocument(doc, docAnalyzer);
                                        }
                                    }
                                });
                        } finally {
                            doc.close();
                        }
                    }
                }
                finally
                {
                    searcher.free();
                }
            }
        }
    }

    private static class QueryMatcher implements PartVisitor {
        private final Query query;
        public boolean matches;

        public QueryMatcher(final Query query)
        {
            this.query = query;
            matches = false;
        }

        @Override
        public void visit(final AbstractPart part) throws IOException {
            if (matches) return;
            if (part instanceof PrimaryKeyPart) {
                final PrimaryKeyPart primaryKeyPart = (PrimaryKeyPart) part;
                Document[] docs = primaryKeyPart.search(query, 1);
                if (docs.length > 0) {
                    matches = true;
                }
            }
        }
    }

    public PrimaryKeyPartBase(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> queueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            queueIds,
            version,
            config);
    }

    private ConditionsAccessor createCondAccessor(
        final Map<String,String> conditions,
        final IndexSearcher searcher,
        final Analyzer analyzer)
        throws ParseException, IOException
    {
        if (conditions == null) return null;
        QueryParser parser = new QueryParser(Version.LUCENE_40,"",analyzer);
        parser.setAllowLeadingWildcard( true );
        parser.setAnalyzeRangeTerms( true );
        parser.setAutoGeneratePhraseQueries( true );
        final Map<String, Boolean> results = new HashMap<String, Boolean>();
        for (Map.Entry<String,String> entry : conditions.entrySet()) {
            final String name = entry.getKey();
            final String queryText = entry.getValue();
            boolean matches = false;
            try {
                Query query = parser.parse(queryText);
                QueryMatcher matcher = new QueryMatcher(query);
                traverseParts(matcher);
                matches = matcher.matches;
                if (!matches) {
                    PlainCollector col = new PlainCollector(1);
                    searcher.search(query, col, true);
                    col.flush();
                    if (col.docs().size() > 0) {
                        matches = true;
                    }
                }
            } catch (org.apache.lucene.queryParser.ParseException e) {
                throw new ParseException(e.getMessage(), -1);
            }
            if (matches) results.put(name, matches);
        }
        return new ConditionsAccessor() {
            @Override
            public boolean getBoolean(String name) {
                Boolean m = results.get(name);
                return m != null;
            }

            @Override
            public int getInt(String name) {
                return getBoolean(name) ? 1 : 0;
            }
        };
    }

    public PrimaryKeySearcher findPrimaryKey(
        final PrimaryKey primaryKey,
        final IndexSearcher searcher,
        final boolean allowDuplicates)
        throws IOException
    {
        PrimaryKeySearcher primaryKeySearcher =
            new PrimaryKeySearcher(primaryKey);
        traverseParts(primaryKeySearcher);
        if (primaryKeySearcher.part() == null) {
            if (searcher == null) {
                return null;
            } else {
                final List<Document> docs;
                final Query query = primaryKey.query();
                if (query instanceof TermQuery) {
//                if (false) {
                    docs = fastDiskSearch((TermQuery) query, searcher);
                } else {
                    PlainCollector col = new PlainCollector(1000);
                    searcher.search(query, col, true);
                    col.flush();
                    docs = col.docs();
                }
                if (docs.size() > 1) {
                    if (!allowDuplicates) {
                        throw new DuplicatePrimaryKeyException(primaryKey);
                    } else {
                        if (logger.isLoggable(Level.SEVERE)) {
                            logger.severe("Document " + primaryKey
                                + " has duplicate entries leaving the last one");
                        }
                    }
                }
                if (docs.size() == 0) {
                    return null;
                } else {
                    primaryKeySearcher.doc(docs.get(docs.size()-1));
                    return primaryKeySearcher;
                }
            }
        } else {
            return primaryKeySearcher;
        }
    }

    public static List<Document> fastDiskSearch(
        final TermQuery query,
        final IndexSearcher searcher)
        throws IOException
    {
        final Term keyTerm = query.getTerm();
        final String field = keyTerm.field();
        final BytesRef needle = keyTerm.bytes();

        IndexReader.AtomicReaderContext[] atomicReaders =
            searcher.getIndexReader().getTopReaderContext().leaves();

        ArrayList<Document> docs = null;
        BytesRef term = null;
        int skipped = 0;
        int scanned = 0;
        for (IndexReader.AtomicReaderContext arc : atomicReaders) {
            IndexReader reader = arc.reader;

            Terms terms = reader.fields().terms(field);
            if (terms == null) {
                continue;
            }

            TermsEnum te = terms.getThreadTermsEnum(true);
            if (te == null) {
                continue;
            }
            try {
                DocsEnum de = null;
                int sr = te.seekExactEx(needle, true);
                if (sr == -1) {
                    skipped++;
                } else {
                    scanned++;
                }
                if (sr < 1) {
                    continue;
                }
                Bits deletedDocs = reader.getDeletedDocs();
                term = te.term();
                if (term == null) {
                    continue;
                }
                de = te.docs(deletedDocs, de);
                int docId;
                while ((docId = de.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                    if (docs == null) {
                        docs = new ArrayList<>(1);
                    }
                    docs.add(reader.document(docId));
                }
            } finally {
                te.close();
            }
        }
//        if (logger.isLoggable(Level.FINEST)) {
//            logger.finest("key: <" + needle.utf8ToString()
//                + ">: scans: " + scanned
//                + ", skips: " + skipped);
//        }
        if (docs == null) {
            return Collections.emptyList();
        }
        return docs;
    }

    private void add(
        final HTMLDocument doc,
        final Analyzer analyzer,
        final IndexSearcher searcher)
        throws IOException
    {
        PrimaryKey primaryKey = doc.primaryKey();
        try {
            PrimaryKeySearcher primaryKeySearcher =
                findPrimaryKey(primaryKey, searcher, false);
            if (primaryKeySearcher == null) {
                indexDocument(doc, analyzer);
            } else {
                if( doc.failOnDuplicateKey() ) {
                    throw new ConflictPrimaryKeyException( primaryKey );
                } else {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Document " + primaryKey
                            + " is already in index. Add rejected");
                    }
                }
            }
        } catch (DuplicatePrimaryKeyException e) {
            if (config.autoFixIndex()) {
                delete(doc, searcher);
                add(doc, analyzer, null);
            } else {
                throw e;
            }
        }
    }

    @Override
    public void add(final DocumentsMessage message, final Map<String,String> conditions)
        throws IOException, ParseException
    {
        final Prefix prefix = message.prefix();
        indexAccessor().prefixReadAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    final Analyzer searchAnalyzer =
                        analyzerProvider().searchAnalyzer(message.prefix());

                    if (message.docProcessor() != null) {
                        message.docProcessor().init(analyzerProvider());
                    }

                    try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                        final ConditionsAccessor condAccessor =
                                createCondAccessor(conditions, searcher.searcher(), searchAnalyzer);

                        for (final HTMLDocument doc: message.documents()) {
                            doc.prepare(condAccessor);
                            if (message.docProcessor() != null) {
                                message.docProcessor().process(
                                    PrimaryKeyPartBase.this,
                                    searcher.searcher(),
                                    condAccessor,
                                    doc);
                            }

                            indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                new IndexAccessor.AtomicOp() {
                                    @Override
                                    public void run() throws IOException {
                                        final Analyzer indexAnalyzer =
                                                analyzerProvider().indexAnalyzer(message.prefix());
                                        add(doc, indexAnalyzer, searcher.searcher());
                                    }
                                });
                        }
                    }
                }
            });
    }

    private void modify(
        final HTMLDocument doc,
        final Analyzer analyzer,
        final IndexSearcher searcher)
        throws IOException
    {
        delete(doc, searcher);
        indexDocument(doc, analyzer);
    }

    @Override
    public void modify(final DocumentsMessage message, final Map<String,String> conditions)
        throws IOException, ParseException
    {
        Prefix prefix = message.prefix();
        indexAccessor().prefixReadAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    final Analyzer searchAnalyzer =
                            analyzerProvider().searchAnalyzer(message.prefix());

                    if (message.docProcessor() != null) {
                        message.docProcessor().init(analyzerProvider());
                    }
                    try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                        final ConditionsAccessor condAccessor =
                                createCondAccessor(conditions, searcher.searcher(), searchAnalyzer);

                        for (final HTMLDocument doc: message.documents()) {
                            doc.prepare(condAccessor);
                            if (message.docProcessor() != null) {
                                message.docProcessor().process(
                                    PrimaryKeyPartBase.this,
                                    searcher.searcher(),
                                    condAccessor,
                                    doc);
                            }
                            indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                new IndexAccessor.AtomicOp() {
                                    @Override
                                    public void run() throws IOException {
                                        final Analyzer indexAnalyzer =
                                                analyzerProvider().indexAnalyzer(message.prefix());

                                        modify(doc, indexAnalyzer, searcher.searcher());
                                    }
                                });
                        }
                    }
                }
            });
    }

    private void delete(final HTMLDocument doc, final IndexSearcher searcher)
        throws IOException
    {
        PrimaryKey primaryKey = doc.primaryKey();
        PrimaryKeySearcher primaryKeySearcher =
            findPrimaryKey(
                primaryKey,
                searcher,
                config.autoFixIndex());
        if (primaryKeySearcher == null) {
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Document " + primaryKey
                    + " was not found. Delete rejected");
            }
        } else {
            if (primaryKeySearcher.part() == null) {
                indexAccessor().deleteDocuments(primaryKey.queryProducer());
            } else {
                primaryKeySearcher.part().deleteDocument(primaryKey);
            }
        }
    }

    @Override
    public void delete(final DocumentsMessage message)
        throws IOException, ParseException
    {
        Prefix prefix = message.prefix();
        indexAccessor().prefixReadAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                        for (final HTMLDocument doc: message.documents()) {
                            doc.prepare(null);
                            indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                new IndexAccessor.AtomicOp() {
                                    @Override
                                    public void run() throws IOException {
                                        delete(doc, searcher.searcher());
                                    }
                                });
                        }
                    }
                }
            });
    }

    @Override
    public void delete(final DeleteMessage message)
        throws IOException, ParseException
    {
        final Prefix prefix = message.prefix();
        indexAccessor().prefixWriteAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    QueryProducer query =
                        message.query(analyzerProvider().searchAnalyzer(prefix));
                    traverseParts(new Deleter(query, prefix));
                    indexAccessor().deleteDocuments(query);
                }
            });
    }

    private HTMLDocument upgradeDocument(final HTMLDocument doc, final ConditionsAccessor condAccessor)
        throws IOException
    {
        if (doc instanceof UpdateOnlyHTMLDocument) {
            try {
                ((UpdateOnlyHTMLDocument)doc).upgrade(condAccessor);
            } catch (FieldFunctionException e) {
                throw new FieldFunctionIOException(
                    "Caught FieldFunctionException while trying to upgrade "
                    + "Update document to Index document",
                    e);
            }
        }
        return doc;
    }

    private HTMLDocument createUpdatedDoc(
        final Document doc,
        final HTMLDocument newValues,
        final long phantomQueueId,
        final String queueName,
        final boolean orderIndependentUpdate,
        final ConditionsAccessor condAccessor)
        throws IOException
    {
        try {
            HTMLDocument newDoc =
                new HTMLDocument(
                    doc,
                    newValues.prefix(),
                    phantomQueueId,
                    queueName,
                    orderIndependentUpdate,
                    config,
                    newValues.logger());
            if (newValues instanceof UpdateOnlyHTMLDocument) {
                try {
                    newDoc.updateWith(
                        ((UpdateOnlyHTMLDocument)newValues).updateFields(),
                        condAccessor);
                } catch (FieldFunctionException e) {
                    throw new FieldFunctionIOException(e);
                }
            } else {
                newDoc.updateWith(newValues);
            }
            return newDoc;
        } catch (ParseException e) {
            CorruptIndexException ex =
                new CorruptIndexException("Failed to update document");
            ex.initCause(e);
            throw ex;
        }
    }

    private void update(
        final HTMLDocument doc,
        final Analyzer analyzer,
        final IndexSearcher searcher,
        final long phantomQueueId,
        final String queueName,
        final ConditionsAccessor condAccessor,
        final boolean addIfNotExists,
        final boolean orderIndependentUpdate)
        throws IOException
    {
        update(
            doc,
            analyzer,
            searcher,
            phantomQueueId,
            queueName,
            condAccessor,
            null,
            addIfNotExists,
            orderIndependentUpdate);
    }

    private void update(
        final HTMLDocument doc,
        final Analyzer analyzer,
        final IndexSearcher searcher,
        final long phantomQueueId,
        final String queueName,
        final ConditionsAccessor condAccessor,
        final Set<String> preserveFields,
        final boolean addIfNotExists,
        final boolean orderIndependentUpdate)
        throws IOException
    {
        PrimaryKey primaryKey = doc.primaryKey();
        PrimaryKeySearcher primaryKeySearcher =
            findPrimaryKey(
                primaryKey,
                searcher,
                config.autoFixIndex());

        if (primaryKeySearcher == null) {
            if (addIfNotExists) {
                indexDocument(upgradeDocument(doc, condAccessor), analyzer);
            } else {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Document " + primaryKey
                        + " was not found. Update rejected");
                }
            }
        } else {
            final Document oldDoc;
            HTMLDocument newDoc = null;
            try {
                if (primaryKeySearcher.part() == null) {
                    oldDoc = filterDocument(
                        primaryKeySearcher.doc(),
                        preserveFields);

                    newDoc =
                        createUpdatedDoc(
                            oldDoc,
                            doc,
                            phantomQueueId,
                            queueName,
                            orderIndependentUpdate,
                            condAccessor);
                    //TODO: try to optimize this
//                    if (!newDoc.equals(oldDoc)) {
                    indexAccessor().deleteDocuments(
                        primaryKey.queryProducer());
                    indexDocument(newDoc, analyzer);
//                    }
                } else {
                    if (primaryKeySearcher.doc() != null) {
                        oldDoc = filterDocument(
                            primaryKeySearcher.doc(),
                            preserveFields);
                    } else {
                        oldDoc = filterDocument(
                            primaryKeySearcher.part().getDocumentByPrimaryKey(
                                primaryKey),
                            preserveFields);
                    }

                    newDoc =
                        createUpdatedDoc(
                            oldDoc,
                            doc,
                            phantomQueueId,
                            queueName,
                            orderIndependentUpdate,
                            condAccessor);
//                    if (!newDoc.equals(oldDoc)) {
                    primaryKeySearcher.part().deleteDocument(primaryKey);
                    indexDocument(newDoc, analyzer);
//                    }
                }
            } finally {
                if (newDoc != null) {
                    newDoc.close();
                }
            }
        }
    }

    @Override
    public void update(
        final DocumentsMessage message,
        final Map<String,String> conditions,
        final boolean addIfNotExists,
        final boolean orderIndependentUpdate,
        final Set<String> preserveFields)
        throws IOException, ParseException
    {
        final Prefix prefix = message.prefix();
        indexAccessor().prefixReadAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    final Analyzer searchAnalyzer = analyzerProvider().searchAnalyzer(prefix);
                    if (message.docProcessor() != null) {
                        message.docProcessor().init(analyzerProvider());
                    }
                    LinkedList<Callable<Void>> tasks = new LinkedList<Callable<Void>>();
                    try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                        final ConditionsAccessor condAccessor =
                                createCondAccessor(conditions, searcher.searcher(), searchAnalyzer);

                        for (final HTMLDocument doc: message.documents()) {
                            if (message.docProcessor() != null) {
                                message.docProcessor().process(
                                    PrimaryKeyPartBase.this,
                                    searcher.searcher(),
                                    condAccessor,
                                    doc);
                            }

                            tasks.add(
                                new Callable<Void>() {
                                    @Override
                                    public Void call() throws IOException {
                                        indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                            new IndexAccessor.AtomicOp() {
                                                @Override
                                                public void run() throws IOException {
                                                    Analyzer docAnalyzer = analyzerProvider().indexAnalyzer(prefix);
                                                    update(
                                                        doc,
                                                        docAnalyzer,
                                                        searcher.searcher(),
                                                        message.queueId().phantomQueueId(),
                                                        message.queueShard().service(),
                                                        condAccessor,
                                                        preserveFields,
                                                        addIfNotExists,
                                                        orderIndependentUpdate);
                                                }
                                            });
                                        return null;
                                    }
                                }
                            );
                        }
                        indexManager().executeTasks(tasks);
                    }
                }
            });
    }

    @Override
    public void updateIfNotMatches(final UpdateIfNotMatchesMessage message,
                                   final Map<String,String> conditions,
                                   final boolean addIfNotExists,
                                   final boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        final Prefix prefix = message.prefix();
        indexAccessor().prefixWriteAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    final Analyzer searchAnalyzer = analyzerProvider().searchAnalyzer(prefix);
                    final Query query = message.query(searchAnalyzer);

                    final QueryMatcher matcher = new QueryMatcher(query);
                    // here we are looking through memory index
                    traverseParts(matcher);

                    boolean matches = matcher.matches;

                    if (!matches) {
                        try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                            // and now we are looking on disk
                            PlainCollector col = new PlainCollector(1);
                            searcher.searcher().search(query, col, true);
                            col.flush();
                            if (col.docs().size() > 0) {
                                matches = true;
                            }

                            if (!matches) {
                                final ConditionsAccessor condAccessor = createCondAccessor(conditions, searcher.searcher(), searchAnalyzer);
                                for (final HTMLDocument doc: message.documents()) {
                                    indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                        new IndexAccessor.AtomicOp() {
                                            @Override
                                            public void run() throws IOException {
                                                final Analyzer indexAnalyzer = analyzerProvider().indexAnalyzer(prefix);
                                                update(
                                                    doc,
                                                    indexAnalyzer,
                                                    searcher.searcher(),
                                                    message.queueId().phantomQueueId(),
                                                    message.queueShard().service(),
                                                    condAccessor,
                                                    addIfNotExists,
                                                    orderIndependentUpdate);
                                            }
                                        });
                                }
                            }
                        }
                    }
                }
            });
    }

    @Override
    public void update(final UpdateMessage message,
                       final Map<String,String> conditions,
                       final boolean addIfNotExists,
                       final boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        if (addIfNotExists) {
            throw new ParseException("AddIfNotExists is not supported in Update By Query", 0);
        }
        final Prefix prefix = message.prefix();
        indexAccessor().prefixWriteAtomicOp(prefix,
            new IndexAccessor.PrefixOp() {
                @Override
                public void run() throws IOException, ParseException {
                    final PrefixingAnalyzerWrapper searchAnalyzer = analyzerProvider().searchAnalyzer(prefix);
                    final PrefixingAnalyzerWrapper indexAnalyzer = analyzerProvider().indexAnalyzer(prefix);
                    searchAnalyzer.resetLastPrefixedField();
                    QueryProducer query = message.query(searchAnalyzer);
                    boolean nonPrefixed = searchAnalyzer.lastPrefixedField() == null;

                    if (message.docProcessor() != null) {
                        message.docProcessor().init(analyzerProvider());
                    }

                    try (final CloseableSearcher searcher = indexAccessor().searcher()) {
                        final ConditionsAccessor condAccessor =
                            createCondAccessor(conditions, searcher.searcher(), searchAnalyzer);

                        final Updater updater =
                            new Updater(
                                query,
                                indexAnalyzer,
                                prefix,
                                message.queueId().phantomQueueId(),
                                message.queueShard().service(),
                                nonPrefixed,
                                condAccessor,
                                message.document(),
                                orderIndependentUpdate,
                                message.context().logger());
                        traverseParts(updater);
                        BaseUpdatingCollector col;
                        if (nonPrefixed) {
                            col = new KeepPrefixUpdatingCollector(
                                message.document(),
                                message.queueId().phantomQueueId(),
                                message.queueShard().service(),
                                condAccessor,
                                config,
                                orderIndependentUpdate,
                                message.context().logger());
                        } else {
                            col = new UpdatingCollector(
                                message.document(),
                                prefix,
                                message.queueId().phantomQueueId(),
                                message.queueShard().service(),
                                condAccessor,
                                orderIndependentUpdate,
                                config,
                                message.context().logger());
                        }

                        searcher.searcher().search(
                            query.produceQuery(),
                            col,
                            true);
                        List<HTMLDocument> found = col.documents();
                        for (final HTMLDocument doc: found) {
                            if (message.docProcessor() != null) {
                                message.docProcessor().process(
                                    PrimaryKeyPartBase.this,
                                    searcher.searcher(),
                                    condAccessor,
                                    doc);
                            }
                            try {
                                indexAccessor().primaryKeyAtomicOp(doc.primaryKey(),
                                    new IndexAccessor.AtomicOp() {
                                        @Override
                                        public void run() throws IOException {
                                            PrefixingAnalyzerWrapper docAnalyzer = indexAnalyzer;
                                            if (nonPrefixed) {
                                                // we should set prefix for analyzer for each doc, because internally
                                                // it is one object, we could occasionally use analyzer with prefix
                                                // from old doc
                                                // https://st.yandex-team.ru/PS-3662
                                                docAnalyzer = analyzerProvider().indexAnalyzer(doc.prefix());
                                                logger.warning(
                                                    "Non prefixed update"
                                                        + message.prefix() + ' ' + doc.prefix());
                                            }
                                            modify(doc, docAnalyzer, searcher.searcher());
                                        }
                                    });
                            } finally {
                                doc.close();
                            }
                        }
                    }
                }
            });
    }
}
