package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.SingleInstanceLockFactory;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.UnicodeUtil;

import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.lucene.util.ThreadInterruptedException;

/**
 * A memory-resident {@link Directory} implementation.  Locking
 * implementation is by default the {@link SingleInstanceLockFactory}
 * but can be changed with {@link #setLockFactory}.
 */
public class PrintStreamDirectory extends Directory {
  protected final Map<String,PrintStreamFile> fileMap = new ConcurrentHashMap<String,PrintStreamFile>();
  protected int filesCnt = 0;
  protected final AtomicLong sizeInBytes = new AtomicLong();
  boolean closed = false;
  private OutputStream outStream;
  private Directory tmpDir;
  public Object writeLock = new Object();

  public static class REMOTE_CODES
  {
    final static byte FILE_FLUSH = 1;
    final static byte FILE_CLOSE = 2;
    final static byte FILE_SEEK  = 3;
    final static byte FILE_WRITE_BYTE  = 4;
    final static byte FILE_WRITE_BYTES = 5;
    final static byte NEW_DIR = 6;
    final static byte NEW_FILE = 7;
    final static byte DELETE_FILE = 8;
    final static byte CLOSE_DIR = 9;
    final static byte FINISHED = 10;
    final static byte QUEUE_IDS_START = 11;
    final static byte QUEUE_IDS_NEXT_ID = 12;
  }

  /** Constructs an empty {@link Directory}. */
  public PrintStreamDirectory( OutputStream stream, Directory tmpDirectory ) throws IOException {
    try {
      setLockFactory(new SingleInstanceLockFactory());
    } catch (IOException e) {
      // Cannot happen
    }
    this.outStream = stream;
    this.tmpDir = tmpDirectory;
    synchronized( writeLock )
    {
	writeVInt( REMOTE_CODES.NEW_DIR );
    }
  }

    public void sendQueueIds(
        final Map<QueueShard, Long> ids,
        final int startShard,
        final int endShard)
        throws IOException
    {
        writeVInt(REMOTE_CODES.QUEUE_IDS_START);
        for (final Map.Entry<QueueShard, Long> entry : ids.entrySet()) {
            if (entry.getKey().shardId() >= startShard
                && entry.getKey().shardId() <= endShard)
            {
                writeVInt(REMOTE_CODES.QUEUE_IDS_NEXT_ID);
                writeString(entry.getKey().service());
                writeVInt(entry.getKey().shardId());
                writeVLong(entry.getValue());
            }
        }
    }

  public void finished() throws IOException
  {
    synchronized( writeLock )
    {
	writeVInt( REMOTE_CODES.FINISHED );
    }
  }

  @Override
  public final String[] listAll() {
    ensureOpen();
    // NOTE: fileMap.keySet().toArray(new String[0]) is broken in non Sun JDKs,
    // and the code below is resilient to map changes during the array population.
    Set<String> fileNames = fileMap.keySet();
    List<String> names = new ArrayList<String>(fileNames.size());
    for (String name : fileNames) names.add(name);
    return names.toArray(new String[names.size()]);
  }

    @Override
    public final String[] listAll(final String prefix) {
        ensureOpen();
        List<String> files = new ArrayList<>();
        for (String file: fileMap.keySet()) {
            if (file.startsWith(prefix)) {
                files.add(file);
            }
        }
        return files.toArray(new String[0]);
    }

  /** Returns true iff the named file exists in this directory. */
  @Override
  public final boolean fileExists(String name) {
    ensureOpen();
    return fileMap.containsKey(name);
  }

  /** Returns the time the named file was last modified.
   * @throws IOException if the file does not exist
   */
  @Override
  public final long fileModified(String name) throws IOException {
    ensureOpen();
    PrintStreamFile file = fileMap.get(name);
    if (file == null) {
      throw new FileNotFoundException(name);
    }
    return file.getLastModified();
  }

  /** Set the modified time of an existing file to now.
   * @throws IOException if the file does not exist
   */
  @Override
  public void touchFile(String name) throws IOException {
    ensureOpen();
    PrintStreamFile file = fileMap.get(name);
    if (file == null) {
      throw new FileNotFoundException(name);
    }

    long ts2, ts1 = System.currentTimeMillis();
    do {
      try {
        Thread.sleep(0, 1);
      } catch (InterruptedException ie) {
        throw new ThreadInterruptedException(ie);
      }
      ts2 = System.currentTimeMillis();
    } while(ts1 == ts2);

    file.setLastModified(ts2);
  }

  /** Returns the length in bytes of a file in the directory.
   * @throws IOException if the file does not exist
   */
  @Override
  public final long fileLength(String name) throws IOException {
    ensureOpen();
    PrintStreamFile file = fileMap.get(name);
    if (file == null) {
      throw new FileNotFoundException(name);
    }
    return file.getLength();
  }

  /**
   * Return total size in bytes of all files in this directory. This is
   * currently quantized to RAMOutputStream.BUFFER_SIZE.
   */
  public final long sizeInBytes() {
    ensureOpen();
    return sizeInBytes.get();
  }

  /** Removes an existing file in the directory.
   * @throws IOException if the file does not exist
   */
  @Override
  public void deleteFile(String name) throws IOException {
    ensureOpen();
    if( name.endsWith( ".tmp" ) )
    {
	tmpDir.deleteFile( name + "_" + this.toString() );
    }
    PrintStreamFile file = fileMap.remove(name);
    if (file != null) {
      file.directory = null;
      synchronized( writeLock )
      {
        writeVInt( REMOTE_CODES.DELETE_FILE );
        writeVInt( file.fileNumber() );
      }
      sizeInBytes.addAndGet(-file.sizeInBytes);
    } else {
      throw new FileNotFoundException(name);
    }
  }

  @Override
  public IndexOutput createOutput(String name, final int bufferSize) throws IOException {
    return createOutput(name);
  }

  /** Creates a new, empty file in the directory with the given name. Returns a stream writing this file. */
  @Override
  public IndexOutput createOutput(String name) throws IOException {
    ensureOpen();
    if( name.endsWith( ".tmp" ) )
    {
	IndexOutput tmpOut = tmpDir.createOutput( name + "_" + this.toString() );
	return tmpOut;
    }

    int newFileNumber;
    synchronized( this )
    {
	newFileNumber = filesCnt++;
    }
    PrintStreamFile existing = fileMap.remove(name);
    if (existing != null) {
      existing.directory = null;
      newFileNumber = existing.fileNumber();
    }
    synchronized( writeLock )
    {
	writeVInt( REMOTE_CODES.NEW_FILE );
	writeVInt( newFileNumber );
	writeString( name );
    }
    PrintStreamFile file = new PrintStreamFile( this, newFileNumber );
    fileMap.put(name, file);
    return new PrintStreamOutputStream(file);
  }

  @Override
  public void sync(Collection<String> names) throws IOException {
  }

  /** Returns a stream reading an existing file. */
  @Override
  public IndexInput openInput(String name) throws IOException {
    ensureOpen();
    if( name.endsWith( ".tmp" ) )
    {
	return tmpDir.openInput( name + "_" + this.toString() );
    }

    PrintStreamFile file = fileMap.get(name);
    if (file == null) {
      throw new FileNotFoundException(name);
    }
    throw new IOException( "openInput is not implemented in write-only directory: opening " + name );
  }

  /** Closes the store to future operations, releasing associated memory. */
  @Override
  public void close() throws IOException {
    isOpen = false;
    Iterator<Map.Entry<String,PrintStreamFile>> iter = fileMap.entrySet().iterator();
    while( iter.hasNext() )
    {
	Map.Entry<String,PrintStreamFile> entry = iter.next();
	PrintStreamFile f = entry.getValue();
	f.directory = null;
    }
    fileMap.clear();
    closed = true;
    synchronized( writeLock )
    {
	writeVInt( REMOTE_CODES.CLOSE_DIR );
    }
  }

  int checkCount = 0;
  public final void writeByte( byte b ) throws IOException
  {
    outStream.write( b );
  }
  
  public final void writeBytes( byte[] b, int off, int len ) throws IOException
  {
    outStream.write( b, off, len );
  }

  public final void writeVInt(int i) throws IOException {
    while ((i & ~0x7f) != 0) {
      writeByte((byte)((i & 0x7f) | 0x80));
      i >>>= 7;
    }
    writeByte((byte)i);
  }

  public final void writeVLong(long i) throws IOException {
    while ((i & ~0x7fL) != 0L) {
      writeByte((byte)((i & 0x7fL) | 0x80L));
      i >>>= 7;
    }
    writeByte((byte)i);
  }

  public void writeString(String s) throws IOException {
    final BytesRef utf8Result = new BytesRef(10);
    UnicodeUtil.UTF16toUTF8(s, 0, s.length(), utf8Result);
    writeVInt(utf8Result.length);
    writeBytes(utf8Result.bytes, 0, utf8Result.length);
  }
}
