package ru.yandex.msearch;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

import org.apache.lucene.store.Directory;

import ru.yandex.msearch.PrintStreamDirectory.REMOTE_CODES;

/** @lucene.internal */
public class PrintStreamFile {

  long length;
  PrintStreamDirectory directory;
  protected long sizeInBytes;
  boolean freed = false;
  private int fileNo;

  // This is publicly modifiable via Directory.touchFile(), so direct access not supported
  private long lastModified = System.currentTimeMillis();

  // File used as buffer, in no RAMDirectory
  protected PrintStreamFile() throws Exception
  {
    throw new Exception( "NativeRAMFile cannot be used outside of directory" );
  }

  PrintStreamFile(PrintStreamDirectory directory, int no) {
    this.directory = directory;
    this.fileNo = no;
  }

  // For non-stream access from thread that might be concurrent with writing
  public synchronized long getLength() {
    return length;
  }

  public int fileNumber()
  {
    return fileNo;
  }

  protected synchronized void setLength(long length) {
    this.length = length;
  }

  // For non-stream access from thread that might be concurrent with writing
  public synchronized long getLastModified() {
    return lastModified;
  }

  protected synchronized void setLastModified(long lastModified) {
    this.lastModified = lastModified;
  }

  public void remoteFlush() throws IOException
  {
    synchronized( directory.writeLock )
    {
	directory.writeVInt( REMOTE_CODES.FILE_FLUSH );
	directory.writeVInt( fileNo );
    }
  }

  public void remoteClose() throws IOException
  {
    synchronized( directory.writeLock )
    {
	directory.writeVInt( REMOTE_CODES.FILE_CLOSE );
	directory.writeVInt( fileNo );
    }
  }

  public void remoteWriteByte( byte b ) throws IOException
  {
    synchronized( directory.writeLock )
    {
	directory.writeVInt( REMOTE_CODES.FILE_WRITE_BYTE );
	directory.writeVInt( fileNo );
	directory.writeByte( b );
    }
  }

  public void remoteSeek( long p ) throws IOException
  {
    synchronized( directory.writeLock )
    {
	directory.writeVInt( REMOTE_CODES.FILE_SEEK );
	directory.writeVInt( fileNo );
	directory.writeVLong( p );
    }
  }

  public void remoteWriteBytes( byte[] b, int offset, int len ) throws IOException
  {
    synchronized( directory.writeLock )
    {
	directory.writeVInt( REMOTE_CODES.FILE_WRITE_BYTES );
	directory.writeVInt( fileNo );
	directory.writeVInt( len );
	directory.writeBytes( b, offset, len );
    }
  }
}
