package ru.yandex.msearch;

import java.util.Collection;

import org.apache.lucene.queryParser.QueryParser;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.msearch.collector.CollectingFieldToIndex;
import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.Prefix;

public class ProcessorRequestContext {
    private final Collection<Prefix> prefixes;
    private final Index index;
    private final RequestContext ctx;
    private final CollectingFieldToIndex fieldToIndex;
    private final CgiParams params;
    private final boolean debug;
    private final boolean experimental;
    private final String outScoreField;
    /**
     * if right side was not found, keep or drop record at all
     */
    private final boolean keepRightNull;
    private final boolean syncSearcher;
    private final int subQueryCacheSize;

    private QueryParser queryParser;
    private PrefixingAnalyzerWrapper prefixedAnalyzer;

    public ProcessorRequestContext(
        final Collection<Prefix> prefixes,
        final Index index,
        final RequestContext ctx,
        final CollectingFieldToIndex fieldToIndex,
        final CgiParams params)
        throws BadRequestException
    {
        this.prefixes = prefixes;
        this.index = index;
        this.ctx = ctx;
        this.fieldToIndex = fieldToIndex;
        this.params = params;
        this.debug = params.getBoolean("debug", false);
        this.syncSearcher =
            params.getBoolean("sync-searcher", index.config().syncSearcherDefault());
        this.keepRightNull =
            params.getBoolean("keep-right-null", true);
        this.experimental =
            params.getBoolean("experimental", false);
        this.subQueryCacheSize =
            params.getInt("sub-query-cache-size", -1);
        this.outScoreField = params.getString("left_dp_score", null);
    }

    public int subQueryCacheSize() {
        return subQueryCacheSize;
    }

    public String outScoreField() {
        return outScoreField;
    }

    public boolean experimental() {
        return experimental;
    }

    public CgiParams params() {
        return params;
    }

    public ProcessorRequestContext queryParser(final QueryParser queryParser) {
        this.queryParser = queryParser;
        if (queryParser.getAnalyzer() instanceof PrefixingAnalyzerWrapper) {
            this.prefixedAnalyzer =
                (PrefixingAnalyzerWrapper) queryParser.getAnalyzer();
        } else {
            this.prefixedAnalyzer = null;
        }

        return this;
    }

    public Collection<Prefix> prefix() {
        return prefixes;
    }

    public Index index() {
        return index;
    }

    public RequestContext ctx() {
        return ctx;
    }

    public CollectingFieldToIndex fieldToIndex() {
        return fieldToIndex;
    }

    public QueryParser queryParser() {
        return queryParser;
    }

    public boolean isPrefixedAnalyzer() {
        return prefixedAnalyzer != null;
    }

    public PrefixingAnalyzerWrapper prefixedAnalyzer() {
        return prefixedAnalyzer;
    }

    public boolean debug() {
        return debug;
    }

    public boolean syncSearcher() {
        return syncSearcher;
    }

    public FieldsCache fieldsCache() {
        return index.fieldsCache();
    }

    public boolean keepRightNull() {
        return keepRightNull;
    }
}
