package ru.yandex.msearch;

import java.text.ParseException;

import ru.yandex.search.prefix.Prefix;

public class QueueShard {
    public static final String QUEUE_ID_PREFIX = "queueId-";
    public static final long MAGIC_QUEUEID = Integer.MIN_VALUE;
    public static final int SHARDS_MAGIC = 65534;
    public static final String DEFAULT_SERVICE = "#default";
    public static final QueueShard EMPTY_SHARD_ID = new QueueShard(-1);

    private static final char SEPARATOR = ';';

    private final String service;
    private final int shardId;

    public QueueShard(final int shardId) {
        this.service = DEFAULT_SERVICE;
        this.shardId = shardId;
    }

    public QueueShard(final String service, final Prefix prefix) {
        this(service, (int) (prefix.hash() % SHARDS_MAGIC));
    }

    public QueueShard(final String service, final int shardId) {
        if (service.equals(DEFAULT_SERVICE)) {
            //intern
            this.service = DEFAULT_SERVICE;
        } else {
            this.service = service.intern();
        }
        this.shardId = shardId;
    }

    public int shardId() {
        return shardId;
    }

    public String service() {
        return service;
    }

    public QueueShard defaultServiceShard() {
        return new QueueShard(shardId);
    }

    @Override
    public int hashCode() {
        return service.hashCode() ^ shardId;
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof QueueShard) {
            final QueueShard other = (QueueShard) o;
            return service.equals(other.service) && shardId == other.shardId;
        } else {
            return false;
        }
    }

    @Override
    public String toString() {
        if (service == DEFAULT_SERVICE) {
            return QUEUE_ID_PREFIX + Integer.toString(shardId);
        } else {
            return QUEUE_ID_PREFIX + service + SEPARATOR
                + Integer.toString(shardId);
        }
    }

    public static QueueShard fromString(final String string)
        throws ParseException
    {
        if (!string.startsWith(QUEUE_ID_PREFIX)) {
            throw new ParseException("Can't parse queue shard|service id "
                + "from string: " + string, 0);
        }
        final String shardName = string.substring(QUEUE_ID_PREFIX.length());
        int sep = shardName.indexOf(SEPARATOR);
        final String service;
        final int shardId;
        if (sep == -1) {
            service = DEFAULT_SERVICE;
            shardId = Integer.parseInt(shardName);
        } else {
            service = shardName.substring(0, sep);
            shardId = Integer.parseInt(shardName.substring(sep + 1));
        }
        return new QueueShard(service, shardId);
    }
}
