package ru.yandex.msearch;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentProducer;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.lucene.index.FieldInvertState;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.DefaultSimilarity;
import org.apache.lucene.search.Explanation.IDFExplanation;
import org.apache.lucene.search.FilteredIndexSearcher;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.SimilarityProvider;

import ru.yandex.collection.PatternMap;
import ru.yandex.msearch.collector.CollectInterruptException;
import ru.yandex.msearch.collector.CollectorFactory;
import ru.yandex.msearch.collector.DocCollector;
import ru.yandex.msearch.collector.ParametrizedDocCollector;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunctionFactory;
import ru.yandex.http.server.sync.JsonContentProducerWriter;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.util.IOStater;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.stater.ImmutableStaterConfig;
import ru.yandex.util.timesource.TimeSource;

public class SearchHandler
    extends SearchHandlerBase
    implements HttpRequestHandler
{
    public static final Function<String, BadRequestException> BRE_GEN
        = (n) -> new BadRequestException("Undefined database " + n);

    private static final String USER_FIELD = "__prefix";
    private static final int CHECK_ALIVE_SKIP_INTERVAL = 100;
    private static final long CHECK_ALIVE_TIME_INTERVAL = 200;

    protected final OuterGroupFunctionFactory outerGroupFunctionFactory;

    public SearchHandler(
        final DatabaseManager dbManager,
        final Config config,
        final OuterGroupFunctionFactory outerGroupFunctionFactory,
        final PatternMap<RequestInfo, IOStater> ioStaters)
    {
        super(dbManager, config, ioStaters);
        this.outerGroupFunctionFactory = outerGroupFunctionFactory;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        CgiParams params = new CgiParams(request);
        Index index = dbManager.indexOrException(params, BRE_GEN);

        SearchRequest searchRequest =
            new NewSearchRequest(
                params,
                context,
                index,
                index.config(),
                outerGroupFunctionFactory);

        IOScheduler.setThreadReadPrio(
            params.getInt("IO_PRIO", IOScheduler.IOPRIO_SEARCH));
        Compress.updateSsdCache(params.getBoolean("update-ssd-cache", true));

        CollectorFactory colFactory =
            CollectorFactory.factoryFromRequest(searchRequest);
        String collectorCheckResult =
            colFactory.checkRequestParams(searchRequest);
        if (collectorCheckResult != null) {
            throw new BadRequestException(collectorCheckResult);
        }

        Charset charset = CharsetUtils.acceptedCharset(request);
        SearchProcessor searchProcessor =
            new SearchProcessor(
                index,
                searchRequest,
                colFactory,
                ioStaterFor(context));
        EntityTemplate entity;
        if (charset.equals(StandardCharsets.UTF_8)) {
            Utf8SearchResultProducer consumer =
                new Utf8SearchResultProducer(
                    JsonTypeExtractor.NORMAL.extract(params),
                    searchProcessor,
                    searchRequest.skipNulls());
            entity = new EntityTemplate(consumer);
        } else {
            SearchResultsProducer consumer =
                new SearchResultsProducer(
                    searchProcessor,
                    searchRequest.skipNulls());
            entity =
                new EntityTemplate(
                    new JsonContentProducerWriter(
                        consumer,
                        JsonTypeExtractor.NORMAL.extract(params),
                        charset));
        }
        entity.setChunked(true);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(charset).toString());
        final String service = params.getString("service", null);
        if (service != null && searchRequest.prefixes().size() == 1) {
            final Prefix prefix = searchRequest.prefixes().iterator().next();
            final QueueShard queueShard = new QueueShard(service, prefix);
            response.setHeader(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.toString(index.queueId(queueShard, false)));
        }
        response.setEntity(entity);
        response.setStatusCode(HttpStatus.SC_OK);
    }

    @Override
    public String toString() {
        return "https://wiki.yandex-team.ru/ps/Documentation/Lucene/"
            + "SearchHandlers/search";
    }

    static class Utf8SearchResultProducer
        implements ContentProducer, SearchResultsConsumer
    {
        private final JsonType jsonType;
        private final SearchProcessor processor;
        private final boolean skipNulls;
        protected Utf8JsonWriter writer;
        private OutputStream out;
        private long lastCheck = TimeSource.INSTANCE.currentTimeMillis();

        public Utf8SearchResultProducer(
            final JsonType jsonType,
            final SearchProcessor processor,
            final boolean skipNulls)
        {
            this.jsonType = jsonType;
            this.processor = processor;
            this.skipNulls = skipNulls;
        }

        public Utf8SearchResultProducer(
            final JsonType jsonType,
            final boolean skipNulls,
            final OutputStream out)
        {
            this.jsonType = jsonType;
            this.processor = null;
            this.skipNulls = skipNulls;
            this.out = out;
        }

        public void setWriter(Utf8JsonWriter writer) {
            this.writer = writer;
        }

        @Override
        public void checkAlive() throws IOException {
            long time = TimeSource.INSTANCE.currentTimeMillis();
            if (time - lastCheck > 100) {
                out.write(' ');
                out.flush();
                lastCheck = time;
            }
        }

        @Override
        public void totalHitsCount(int count) throws IOException {
            //this will break unittests
            //writer.key("totalHitsCount");
            //writer.value(count);
        }

        @Override
        public void uniqHitsCount(int count) throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            writer.key("hitsCount");
            writer.value(count);
        }

        @Override
        public void startHits() throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            writer.key("hitsArray");
            writer.startArray();
        }

        @Override
        public void endHits() throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            writer.endArray();
        }

        @Override
        public void startResults() throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            writer.startObject();
        }

        @Override
        public void endResults() throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            writer.endObject();
        }

        @Override
        public void document(YaDoc3 doc, Set<String> fields) throws IOException  {
            lastCheck = TimeSource.INSTANCE.currentTimeMillis();
            doc.writeUtf8Json(writer, fields, skipNulls);
        }

        @Override
        public void writeTo(final OutputStream out) throws IOException {
            this.out = out;
            this.writer = jsonType.create(out);
            checkAlive();
            processor.process(this);
            writer.close();
        }
    }

    static class SearchResultsProducer
        implements JsonValue, SearchResultsConsumer
    {
        private final SearchProcessor processor;
        private final boolean skipNulls;
        private JsonWriterBase writer;

        public SearchResultsProducer(
            final SearchProcessor processor,
            final boolean skipNulls)
        {
            this.processor = processor;
            this.skipNulls = skipNulls;
        }

        @Override
        public void checkAlive() {
        }

        @Override
        public void totalHitsCount(int count) throws IOException {
            //this will break unittests
            //writer.key("totalHitsCount");
            //writer.value(count);
        }

        @Override
        public void uniqHitsCount(int count) throws IOException  {
            writer.key("hitsCount");
            writer.value(count);
        }

        @Override
        public void startHits() throws IOException  {
            writer.key("hitsArray");
            writer.startArray();
        }

        @Override
        public void endHits() throws IOException  {
            writer.endArray();
        }

        @Override
        public void startResults() throws IOException  {
            writer.startObject();
        }

        @Override
        public void endResults() throws IOException  {
            writer.endObject();
        }

        @Override
        public void document(YaDoc3 doc, Set<String> fields) throws IOException  {
            doc.writeJson(writer, fields, skipNulls);
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            this.writer = writer;
            processor.process(this);
        }
    }

    public static class SearchProcessor {
        private final Index index;
        private final SearchRequest request;
        private final CollectorFactory colFuck;
        private final IOStater ioStater;

        public SearchProcessor(
            final Index index,
            final SearchRequest request,
            final CollectorFactory colFuck,
            final IOStater ioStater)
        {
            this.index = index;
            this.request = request;
            this.colFuck = colFuck;
            this.ioStater = ioStater;
        }

        private IndexSearcher wrapSearcher(
            final IndexSearcher searcher,
            final Prefix prefix)
            throws IOException
        {
            return new PerUserMaxDocIndexSearcher(
                searcher,
                USER_FIELD,
                prefix.toString());
        }

        public void process(SearchResultsConsumer consumer) throws IOException {
            consumer.startResults();
            Compress.resetStats();
            DocCollector col = colFuck.createCollector(request, consumer);
            Map<Prefix, Searcher> searchers = new HashMap<>();
            try {
                for (PrefixedQuery query : request.queries()) {
                    Prefix prefix = query.prefix;
                    Searcher searcher = searchers.get(prefix);
                    if (request.updatePrefixActivity()) {
                        index.updatePrefixActivity(prefix);
                    }
                    if (searcher == null) {
                        long time = System.currentTimeMillis();
                        searcher = index.getSearcher(
                            prefix,
                            request.syncSearcher());
                        time = System.currentTimeMillis() - time;
                        if (request.ctx().logger().isLoggable(Level.FINE)) {
                            request.ctx().logger().fine("Searcher get time: "
                                + time);
                        }
                        searchers.put(prefix, searcher);
                    }
                    col.setPrefix(prefix);
                    col.setFieldsCache(index.fieldsCache());
                    IndexSearcher indexSearcher = searcher.searcher();
                    if (request.scoring() && prefix != null) {
                        indexSearcher = wrapSearcher(indexSearcher, prefix);
                    }
                    try {
                        indexSearcher.search(
                            query.query,
                            col,
                            request.reverseTraverse());
                    } catch (CollectInterruptException ignore) {
                    }
                }
                col.flush();
                col.close();
                int count = col.uniqCount();
                if (request.ctx().logger().isLoggable(Level.INFO)) {
                    request.ctx().logger().info("Total docs found: " + count);
                }
                HttpRequestContext ctx = (HttpRequestContext)request.ctx();
                ((LoggingServerConnection) ctx.connection())
                    .setHitsCount(Integer.toString(count));
                consumer.endResults();
            } finally {
                for (Searcher searcher : searchers.values()) {
                    searcher.free();
                }
                accountStats(ioStater, request.ctx().logger());
            }
        }
    }

    static class PerUserMaxDocIndexSearcher
        extends FilteredIndexSearcher
    {
        private final UserMaxDocSimilarity similarity;

        public PerUserMaxDocIndexSearcher(
            final IndexSearcher orig,
            final String userField,
            final String user)
            throws IOException
        {
            super(orig);
            final Term userTerm = new Term(USER_FIELD, user);
            final int userDocCount = orig.docFreq(userTerm);
            similarity = new UserMaxDocSimilarity(userDocCount);
        }

        @Override
        public SimilarityProvider getSimilarityProvider() {
            return similarity;
        }
    }

    private static class UserMaxDocSimilarity extends DefaultSimilarity {
        private final int userDocCount;

        public UserMaxDocSimilarity(final int userDocCount) {
            this.userDocCount = userDocCount;
        }

        @Override
        public IDFExplanation idfExplain(
            final Term term,
            final IndexSearcher searcher,
            final int docFreq)
            throws IOException
        {
            final int df = docFreq;
            final float idf = idf(df, userDocCount);
            return new IDFExplanation() {
                @Override
                public String explain() {
                    return "idf(docFreq=" + df +
                        ", maxUserDocs=" + userDocCount + ")";
                }

                @Override
                public float getIdf() {
                    return idf;
                }};
        }

        @Override
        public IDFExplanation idfExplain(
            final Collection<Term> terms,
            final IndexSearcher searcher)
            throws IOException
        {
            final int max = userDocCount;
            float idf = 0.0f;
            final StringBuilder exp = new StringBuilder();
            for (final Term term : terms) {
                final int df = searcher.docFreq(term);
                idf += idf(df, max);
                exp.append(" ");
                exp.append(term.text());
                exp.append("=");
                exp.append(df);
            }
            final float fIdf = idf;
            return new IDFExplanation() {
                @Override
                public float getIdf() {
                    return fIdf;
                }
                @Override
                public String explain() {
                    return exp.toString();
                }
            };
        }
    }
}

