package ru.yandex.msearch;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiReader;
import org.apache.lucene.search.IndexSearcher;

public abstract class Searcher implements IndexReader.ReaderFinishedListener
{
    protected IndexSearcher	searcher;
    protected IndexReader	reader;
    private final AtomicInteger refs = new AtomicInteger(0);
    private volatile boolean finished = false;

    protected Searcher()
    {
    }

    public Searcher( IndexSearcher s, IndexReader r )
    {
        this.reader = r;
        this.searcher = s;
        reader.incRef();
        reader.addReaderFinishedListener( this );
    }

    public IndexSearcher searcher() throws IOException
    {
        ensureOpen();
        return searcher;
    }

    public IndexReader reader() throws IOException
    {
        ensureOpen();
        return reader;
    }

    private final void ensureOpen() throws IOException
    {
        if( searcher == null || reader == null ) throw new IOException( "Searcher already closed: " + this );
    }

    public abstract void free() throws IOException;

    @Override
    public void finished( IndexReader reader )
    {
        if( this.reader == null ) return;
        if( reader != this.reader )
        {
            try
            {
                this.reader.directory();
            }
            catch( Exception e )
            {
                System.err.println( "We have been closed by another reader" );
                e.printStackTrace();
            }
            return;
        }
        reader.removeReaderFinishedListener(this);
        finished = true;
    }

    public void close() throws IOException
    {
        if (searcher != null) {
            searcher.close();
        }
        searcher = null;
        try
        {
            int i = 0;
            while( !finished )
            {
                reader.decRef();
                i++;
            }
        }
        catch( org.apache.lucene.store.AlreadyClosedException e )
        {
            System.err.println( "Hit already closed exception: " + this );
            throw e;
        }
        reader = null;
    }

    public void incRef()
    {
        refs.incrementAndGet();
    }

    public int decRef()
    {
        return refs.decrementAndGet();
    }

    int refs()
    {
        return refs.get();
    }

    public int numDocs()
    {
        return reader.numDocs();
    }

    public long numDocsLong() {
        if (reader == null) {
            return 0;
        }
        return reader.numDocsLong();
    }

    static class MultiSearcher extends Searcher
    {
        Searcher[] subSearchers;

        protected MultiSearcher()
        {
            super();
        }

        public MultiSearcher( Searcher[] subSearchers ) throws IOException
        {
            super();
            initSubSearchers( subSearchers );
        }

        protected void initSubSearchers( Searcher[] subSearchers ) throws IOException
        {
            this.subSearchers = subSearchers;
            IndexReader[] readers = new IndexReader[subSearchers.length];
            for( int i = 0; i < subSearchers.length; i++ )
            {
                readers[i] = subSearchers[i].reader();
            }
            this.reader = new MultiReader( readers, true )
            {
                @Override
                protected synchronized void doClose() throws IOException {
                }
                @Override
                public void addReaderFinishedListener(ReaderFinishedListener listener) {
                    readerFinishedListeners.add(listener);
                }

                @Override
                public void removeReaderFinishedListener(ReaderFinishedListener listener) {
                    readerFinishedListeners.remove(listener);
                }
            };
            this.searcher = new IndexSearcher( this.reader );
            reader.incRef();
            reader.addReaderFinishedListener( this );
        }

        @Override
        public void free() throws IOException
        {
            if( decRef() == 0 )
            {
                close();
                for( int i = 0; i < subSearchers.length; i++ )
                {
                    subSearchers[i].free();
                }
            }
        }
/*
        @Override
        public void finalize()
        {
            if( refs() > 0 )
            {
                System.err.println( "Finalizer: MultiSearcher was not closed: refs() = " + refs() + " this: " + toString() );
            }
            try
            {
                while( refs() > 0 )
                {
                    free();
                }
            }
            catch( IOException e )
            {
            }
        }
*/
        @Override
        public String toString()
        {
            String str = "MultiSearcher: \n";
            for( Searcher sub : subSearchers )
            {
                str += "\t" + sub + "\n";
            }
            return str;
        }
    }
}
