package ru.yandex.msearch;

import java.io.File;
import java.io.IOException;

import java.text.ParseException;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.analysis.SimpleAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.MergeScheduler;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.TieredMergePolicy;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryProducer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;

import ru.yandex.concurrent.LockStorage;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.collector.CollectInterruptException;
import ru.yandex.msearch.collector.DocCollector;
import ru.yandex.msearch.collector.FlushableCollector;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.warmfunction.WarmFunction;

import ru.yandex.search.prefix.Prefix;

import ru.yandex.util.timesource.TimeSource;

public class Shard implements IndexAccessor, ActivityProvider {
    public static final String ACTIVE_PREFIX = "AP_";
    private static final boolean DEBUG = false;
    private static final int DEFAULT_QUERY_LOG_SIZE = 1024;
    private final LockStorage<PrimaryKey, PrimaryKey> keyStorage =
        new LockStorage<>();

    private final LockStorage<Prefix, ReadWriteLock> prefixStorage =
        new LockStorage<>();

    private final ConcurrentLinkedQueue<ReadWriteLock> prefixLockCache =
        new ConcurrentLinkedQueue<ReadWriteLock>();

    private static final String IndexStatusInfoName = "index-status";
    private static final String IndexStatusInfoDescription =
        "index-status-description";

    boolean closed;
    Directory fsDir;
    IndexWriter fsWriter;
    IndexSearcher fsSearcher;
    MemoryIndex   memoryIndex;
    public final int shardNo;
    private final int shardsCount;
    long previousCommit;
    Object lock = new Object();
    private CommonTaskor commonTaskor;

    public enum IndexStatus {
        INITIALIZING("Initializing: shard initialization is in progress"),
        INIT_FAILED("Init-failed: shard initialization has been failed"),
        RO_USER("Read-only: user requested"),
        RO_FAILED("Read-only: writer failed"),
        RW("Read-write");

        private final String description;
        IndexStatus(final String description) {
            this.description = description;
        }

        public String description() {
            return this.description;
        }
    }

    class ShardStatus {
        private final Map<String,Object> infos =
            new TreeMap<String,Object>();
        public ShardStatus() {
        }

        public synchronized void setInfo(final String infoName,
            final Object value)
        {
            infos.put(infoName, value);
        }

        public synchronized void setInfo(final IndexStatus info) {
            infos.put(IndexStatusInfoName, info.name());
            infos.put(IndexStatusInfoDescription, info.description());
        }

        public synchronized Map<String,Object> infos() {
            return Collections.unmodifiableSortedMap(
                new TreeMap<String,Object>(infos));
        }
    }


    private class CacheWarmer extends IndexWriter.IndexReaderWarmer {
        private final FieldsCache fieldsCache;

        CacheWarmer(final FieldsCache fieldsCache) {
            this.fieldsCache = fieldsCache;
        }

        public void unwarm(IndexReader reader) {
            System.err.println("UNWARM CALLED!!!!!!!!!!!!!!!!!!!!");
        }

        public void indexesAdded() {
            searcherLock.writeLock().lock();
        }

        public void warm(final IndexReader reader) {
            int readPrioSave = IOScheduler.getThreadReadPrio();
            int writePrioSave = IOScheduler.getThreadReadPrio();
            try {
                IOScheduler.setThreadReadPrio(IOScheduler.IOPRIO_WARM);
                IOScheduler.setThreadWritePrio(IOScheduler.IOPRIO_WARM);
                warmNoPrio(reader, true);
            } finally {
                IOScheduler.setThreadReadPrio(readPrioSave);
                IOScheduler.setThreadWritePrio(writePrioSave);
            }
        }

        public void warmNoPrio(final IndexReader reader, boolean warm) {
            try {
                long start = System.currentTimeMillis();
                Fields fields = reader.fields();
                if (fields != null) {
                    FieldsEnum fieldsEnum = fields.iterator();
                    String field = null;
                    while ((field = fieldsEnum.next()) != null) {
                        TermsEnum terms = fieldsEnum.terms();
                        terms.next();
                    }
                }
                if (!flushInProgress) {
//                    System.err.println("WARM ENQUEUE REOPEN");
                    enqueueReopen(indexReaderGeneration.get());
                }
                long time = System.currentTimeMillis();
                prewarmTime.add(time - start);

                start = time;
                fieldsCache.loadCache(reader, Shard.this);
                time = System.currentTimeMillis();
                fieldsCacheTime.add(time - start);

                start = time;
                if (warm) {
                    warmActivePrefixes(reader, sortedPrefixesStrings());
                }
                time = System.currentTimeMillis();
                warmTime.add(time - start);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        public void warmActivePrefixes(
            final IndexReader reader,
            final Collection<String> prefixes)
            throws IOException
        {
            Fields fields = reader.fields();
            if (fields != null) {
                FieldsEnum fieldsEnum = fields.iterator();
                String field = null;
                while ((field = fieldsEnum.next()) != null) {
                    boolean warm = false;
//                  System.err.println("WARM: " + reader + " , " + field);
                    FieldConfig fc = config.fieldConfigFast(field);
                    if (fc != null && fc.warm()) {
                        warm = true;
                    }
                    Terms terms = reader.terms(field);
                    if (warm && terms != null) {
                        System.err.println("warmer: "
                            + ", reader: " + reader
                            + ", warming field: " + field);
                        for (WarmFunction warmer: fc.warmers()) {
                            warmer.warm(
                                reader,
                                terms,
                                prefixes,
                                Shard.this);
                        }
                    }
                }
            }
        }

        @Override
        public void mergeSuccess() {
            if (!flushInProgress) {
                enqueueReopen(indexReaderGeneration.get());
            }
        }
    }

    public FieldsCache fieldsCache;
    Integer	currentVersion = null;

    long docsRamUsage = 0;
    private volatile ShardSearcher currentSearcher = null;
    private volatile int indexModified = 0;
    private final ReentrantReadWriteLock searcherLock = new ReentrantReadWriteLock();
    private final ReentrantReadWriteLock atomicOpLock = new ReentrantReadWriteLock();
    private final Object reopenReaderLock = new Object();
    private final Object getSearcherLock = new Object();
    private final AtomicInteger deletes = new AtomicInteger(0);
    private final AtomicLong indexReaderGeneration = new AtomicLong(0);
    private final AtomicLong deletesGeneration = new AtomicLong(0);
    private final AtomicLong searcherGeneration = new AtomicLong(0);
    private final AtomicLong deletesSearcherGeneration = new AtomicLong(0);
    private final AtomicLong multiIndexGeneration = new AtomicLong(0);
    private final AtomicLong multiSearcherGeneration = new AtomicLong(-1);
    private final AnalyzerProvider analyzerProvider;
    private final PrefixedLogger logger;
    private final Logger indexLogger;
    private volatile boolean shouldLockMemoryIndex = false;
    private volatile Searcher.MultiSearcher currentMultiSearcher = null;
    private final Object getMultiSearcherLock = new Object();
    private final Object replaceMultiSearcherLock = new Object();
    private final AtomicBoolean updateMultiSearcherInProgress =
        new AtomicBoolean(false);
    File dir;
    TieredMergePolicy mergePolicy;
    private final int indexingThreadsCount;
    private final ConcurrentHashMap<QueueShard, QueueId> queueIds =
        new ConcurrentHashMap<>();
    private final Lock expungeLock = new ReentrantLock();
    private final ConcurrentHashMap<Prefix, Long> prefixActivity =
        new ConcurrentHashMap<>();
    private final ConcurrentSkipListMap<String, String> sortedActivePrefixes =
        new ConcurrentSkipListMap<>();
    private final ConcurrentLinkedQueue<String> newActivePrefixes =
        new ConcurrentLinkedQueue<>();
    private final ConcurrentLinkedQueue<QueryProducer> deletesSinceFlush =
        new ConcurrentLinkedQueue<>();
    private final long prefixActivityTimeout;
//    private final MpscUnboundedArrayQueue<QueryProducer> deletesSinceFlush =
//        new MpscUnboundedArrayQueue<QueryProducer>(DEFAULT_QUERY_LOG_SIZE);
//    private final ConcurrentLinkedList<Query> deletesSinceFlush = new LinkedList<Query>();
    private final double autoExpungePct;
    private volatile boolean flushInProgress = false;
    private AtomicBoolean indexingEnabled = new AtomicBoolean(false);
    private final ShardStatus shardStatus = new ShardStatus();
    private final Lock flushLock = new ReentrantLock();
    private final DatabaseConfig config;
    private final MergeScheduler mergeScheduler;
    private final boolean queueIdServiceFallback;
    private final CacheWarmer warmer;
    private long lastFlush = 0;
    private final LongAdder prewarmTime = new LongAdder();
    private final LongAdder fieldsCacheTime = new LongAdder();
    private final LongAdder warmTime = new LongAdder();

    private final ThreadPoolExecutor partsParallelExecutor;
    private final File indexBasePath;
    private final AnalyzerProvider journalAnalyzerProvider;
    private final Supplier<CodecProvider> codecProviderSupplier;
    private final Supplier<CodecProvider> memoryCodecProviderSupplier;

    Shard(
        final AnalyzerProvider analyzerProvider,
        final File path,
        final int shardNo,
        final DatabaseConfig cfg,
        final AnalyzerProvider journalAnalyzerProvider,
        final MergeScheduler mergeScheduler,
        final CommonTaskor commonTaskor,
        final Supplier<CodecProvider> cp,
        final Supplier<CodecProvider> memCp,
        final FieldsCache fieldsCache,
        final PrefixedLogger logger,
        final Logger indexLogger,
        final ThreadPoolExecutor partsParallelExecutor)
        throws IOException
    {
        this.analyzerProvider = analyzerProvider;
        this.shardNo = shardNo;
        this.commonTaskor = commonTaskor;
        this.shardsCount = cfg.shards();
        this.indexingThreadsCount = cfg.indexThreads();
        this.autoExpungePct = cfg.autoExpungePct();
        this.config = cfg;
        this.mergeScheduler = mergeScheduler;
        this.queueIdServiceFallback = config.queueIdServiceFallback();
        this.fieldsCache = fieldsCache;
        this.logger =
            logger.replacePrefix("shard[" + String.valueOf(shardNo) + ']');
        this.indexLogger = indexLogger;
        warmer = new CacheWarmer(fieldsCache);

        this.journalAnalyzerProvider = journalAnalyzerProvider;
        this.indexBasePath = path;
        this.partsParallelExecutor = partsParallelExecutor;
        this.codecProviderSupplier = cp;
        this.memoryCodecProviderSupplier = memCp;
        this.prefixActivityTimeout = cfg.prefixActivityTimeout();

        shardStatus.setInfo(IndexStatus.INITIALIZING);
        try {
            init();
            shardStatus.setInfo(IndexStatus.RW);
        } catch (Throwable t) {
            shardStatus.setInfo(IndexStatus.INIT_FAILED);
            throw t;
        }
    }

    private void init() throws IOException {
        init(true);
    }

    private void init(final boolean rw) throws IOException {
        dir = new File(indexBasePath, Integer.toString(shardNo));
        int termInfosDivisor = config.indexDivisor();

        closed = false;

        if (!dir.exists()) {
            logger.warning("Creating folder for shard #" + shardNo);
            dir.mkdirs();
        }
        fsDir = NIOFSDirectory.get(dir, config.oDirectWrite());

        IndexWriterConfig writerConfig =
            new IndexWriterConfig(Version.LUCENE_40, new SimpleAnalyzer(true));
        writerConfig.setCodecProvider(codecProviderSupplier.get());
        writerConfig.setReaderTermsIndexDivisor( termInfosDivisor );
        writerConfig.setTermIndexInterval( 48 );
        writerConfig.setRAMBufferSizeMB( 100 );
        writerConfig.setMergedSegmentWarmer(warmer);
        writerConfig.setMaxThreadStates( indexingThreadsCount );
        writerConfig.setFieldsWriterBufferSize(
            config.yandexFieldsWriterBufferSize());
        mergePolicy = new TieredMergePolicy();
        mergePolicy.setUseCompoundFile( false );
        mergePolicy.setReclaimDeletesWeight(config.reclaimDeletesWeight());
        mergePolicy.setSegmentsPerTier(config.segmentsPerTier());
        mergePolicy.setMaxMergedSegmentMB(config.maxSegmentSize() / 1024 / 1024);
//        mergePolicy.setMergeFactor( cfg.mergeFactor() ); //Replace with segmentsPerTier. HOW ?!
//        mergePolicy.setMaxMergeMBForOptimize(mergePolicy.getMaxMergeMB());
        writerConfig.setMergePolicy( mergePolicy );
        writerConfig.setMergeScheduler( mergeScheduler );

        IndexWriterConfig ramConfig = new IndexWriterConfig( Version.LUCENE_40, new SimpleAnalyzer( true ) );
        ramConfig.setReaderTermsIndexDivisor( termInfosDivisor );
        ramConfig.setTermIndexInterval( 48 );
        ramConfig.setMaxThreadStates( indexingThreadsCount );
        ramConfig.setFieldsWriterBufferSize(
            config.yandexFieldsWriterBufferSize());

        fsWriter = new IndexWriter(
            fsDir,
            writerConfig,
            config.storedFields(),
            config.indexedFields());

        currentVersion = 0;
        try {
            Map<String, String> commitData =
                IndexReader.getCommitUserData(fsDir, codecProviderSupplier.get());
            if (commitData != null) {
                if (commitData.containsKey("version") ) {
                    currentVersion =
                        Integer.parseInt(commitData.get("version"));
                }
                for (Map.Entry<String,String> entry : commitData.entrySet()) {
                    final String name = entry.getKey();
                    if (name.startsWith(QueueShard.QUEUE_ID_PREFIX)) {
                        try {
                            final QueueShard shard = QueueShard.fromString(name);
                            long queueId = Long.parseLong(entry.getValue());
                            if (this.config.kosherShards()
                                && shard.shardId() % this.config.shards() != shardNo)
                            {
                                //prune invalid entries
                                continue;
                            }
                            queueIds.put(shard, new QueueId(queueId));
                        } catch (ParseException | RuntimeException e) {
                            logger.log(
                                Level.SEVERE,
                                "Error parsing queue shard|position from "
                                    + "commit message: "
                                    + name + ", with value: "
                                    + entry.getValue(),
                                e);
                        }
                    } else if (name.startsWith(ACTIVE_PREFIX)) {
                        try {
                            final String prefixStr =
                                name.substring(ACTIVE_PREFIX.length());
                            long lastActivity =
                                Long.parseLong(entry.getValue());
                            Prefix prefix =
                                config.prefixParser().apply(prefixStr);
                            prefixActivity.put(prefix, lastActivity);
                            sortedActivePrefixes.put(
                                prefixStr,
                                prefixStr);
                        } catch (Throwable t) {
                            logger.log(
                                Level.SEVERE,
                                "Error parsing queue prefix activity from "
                                    + "commit message: "
                                    + name + ", with value: "
                                    + entry.getValue(),
                                t);
                        }
                    }
                }
            }
        } catch( IndexNotFoundException e ) {}

        String journalPath = null;
        if(journalAnalyzerProvider != null) {
            journalPath = dir + "/journal/";
            new File(journalPath).mkdirs();
            Journal.replayJournal(
                new JournalHandler(
                    fsWriter,
                    writerConfig,
                    journalAnalyzerProvider,
                    queueIds,
                    prefixActivity,
                    currentVersion,
                    config,
                    logger,
                    indexLogger),
                journalPath,
                config,
                logger.replacePrefix("shard[" + shardNo + "].journal"));
        }
        memoryIndex = new MemoryIndex(
            ramConfig,
            journalPath,
            analyzerProvider,
            this,
            config,
            memoryCodecProviderSupplier.get(),
            partsParallelExecutor);
        indexingEnabled.set(rw);
        fsWriter.maybeMerge();
        mergePolicy.setExpungeAll(false);
        mergePolicy.setConvert(false);
        mergePolicy.setForceMergeDeletesPctAllowed(autoExpungePct);
        fsWriter.expungeDeletes(false);
    }

    public MergeScheduler mergeScheduler() {
        return mergeScheduler;
    }

    public ShardStatus status() {
        return this.shardStatus;
    }

    @Override
    public boolean activePrefix(final Prefix prefix) {
        return activePrefix(
            prefix,
            TimeSource.INSTANCE.currentTimeMillis() - prefixActivityTimeout);
    }

    @Override
    public boolean activePrefix(final String prefix) {
        return activePrefix(
            prefix,
            TimeSource.INSTANCE.currentTimeMillis() - prefixActivityTimeout);
    }

    @Override
    public boolean activePrefix(final Prefix prefix, final long minTime) {
        Long lastActivity = prefixActivity.get(prefix);
        if (lastActivity != null && lastActivity >= minTime) {
            return true;
        }
        return false;
    }

    @Override
    public boolean activePrefix(final String prefixStr, final long minTime) {
        try {
            Prefix prefix =
                config.prefixParser().apply(prefixStr);
            return activePrefix(prefix, minTime);
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public Collection<String> sortedPrefixesStrings() {
        return sortedActivePrefixes.keySet();
    }

    public Collection<String> sortedPrefixesStrings(final long timeout) {
        if (timeout > 0) {
            TreeSet<String> prefixes = new TreeSet();
            final long minTime =
                TimeSource.INSTANCE.currentTimeMillis() - timeout;
            for (String prefix: sortedActivePrefixes.keySet()) {
                if (activePrefix(prefix, minTime)) {
                    prefixes.add(prefix);
                }
            }
            return prefixes;
        } else {
            return sortedPrefixesStrings();
        }
    }

    public void updatePrefixActivity(final Prefix prefix) {
        System.err.println("UPDATE PREFIX ACTIVITY: " + prefix);
        if (prefixActivity.put(prefix, System.currentTimeMillis()) == null) {
            String prefixString = prefix.toString();
            sortedActivePrefixes.put(prefixString, prefixString);
            newActivePrefixes.add(prefixString);
        }
    }

    public void removePrefixActivity(final Prefix prefix) {
        prefixActivity.remove(prefix);
        sortedActivePrefixes.remove(prefix.toString());
        newActivePrefixes.remove(prefix.toString());
    }

    public long activePrefixes() {
        return prefixActivity.size();
    }

    public long timedActivePrefixes() {
        long count = 0;
        final long minTime =
            TimeSource.INSTANCE.currentTimeMillis() - prefixActivityTimeout;
        for (String prefix: sortedActivePrefixes.keySet()) {
            if (activePrefix(prefix, minTime)) {
                count++;
            }
        }
        return count;
    }

    public void checkReloadFieldsCache() throws IOException {
        if (newActivePrefixes.peek() != null) {
            HashSet<String> newPrefixes = new HashSet<>();
            for (
                String prefix = newActivePrefixes.poll();
                prefix != null;
                prefix = newActivePrefixes.poll())
            {
                newPrefixes.add(prefix);
            }
            System.err.println("shard[" + shardNo + "]: updating FieldsCache");
            ShardSearcher searcher = getShardSearcher();
            try {
                IndexReader[] subReaders =
                    searcher.reader().getSequentialSubReaders();
                if (subReaders != null) {
                    for (IndexReader subReader: subReaders) {
                        fieldsCache.reloadCache(subReader, newPrefixes);
                        warmer.warmActivePrefixes(subReader, newPrefixes);
                    }
                }
            } finally {
                if (searcher != null) {
                    searcher.free();
                }
            }
        }
    }

    @Override
    public void dirtifyQueueShard(final QueueShard shard, final long queueId) {
        commonTaskor.dirtifyQueueShard(shard, queueId);
    }

    public void dropAndRecopy(
            final PrefixedLogger logger,
            final ShardDropPolicy policy)
            throws IOException
    {
        dropAndRecopy(logger, policy, null);
    }

    public void dropAndRecopy(
        final PrefixedLogger logger,
        final ShardDropPolicy policy,
        final String version)
        throws IOException
    {
        if( version.isEmpty() && policy.equals(ShardDropPolicy.RESTORE)) {
            logger.warning("Cannot restore, because version doesn't set");
            throw new IOException("Cannot restore, because version doesn't set: "
                    + version
                    + " Shard: "
                    + shardNo);
        }

            Object status = shardStatus.infos().get(IndexStatusInfoName);
//        if (IndexStatus.RW.name().equalsIgnoreCase(String.valueOf(status))) {
//            throw new IOException("Shard not in rw state " + status);
//        }

        logger.warning("Dropping shard " + shardNo +  " status " + status + " " + status.getClass().getName());

        logger.warning("Dropping shard " + shardNo);
        boolean success = false;
        try {
            this.flush(false, true);
            this.doFlush(true);
            logger.warning("Dropping shard " + shardNo + " flush success");
            if (!policy.equals(ShardDropPolicy.RESTORE)) {
                commonTaskor.getJobsManager().setPending(shardNo);
            }
            for (QueueShard queueShard: queueIds.keySet()) {
                commonTaskor.clearDirtyQueueShard(queueShard);
            }
            success = true;
        } finally {
            // return ro back
            if (!success) {
                // nothing bad happened, just resume
                flush(false, false);
            }
        }

        switch (policy) {
            case MOVE:
                logger.warning("Dropping shard " + shardNo + " moving");
                fsWriter.close(true);
                move();
                queueIds.clear();
                logger.warning("Dropping shard " + shardNo + " moved");
                init();
                logger.warning("Dropping shard " + shardNo + " inited");
                break;
            case RESTORE:
                // TODO: here we need to move restored shard into new place
                File rdir = new File(indexBasePath, Integer.toString(shardNo) + "_restored" + version);
                // Check restored dir existing
                if (!rdir.exists()) {
                    logger.warning("Restored dir: " + rdir + "  doesnot exists. Exiting.");
                    break;
                }
                // close fswriter
                fsWriter.close(true);
                // moving shard to to dir123123214, and close all related stuff
                // TODO: what we need to do with old dirs? maybe we need just delete it?
                //  Or we need to create cron for delete this old stuff after some time.
                move();
                // clear queueIds
                queueIds.clear();

                //move, and re init after moving
                logger.warning("Moving restored shard from path: " + rdir + " to path:" + dir);
                rdir.renameTo(dir);
                logger.warning("Re-Init restored shard path: " + dir);
                // init after move
                init(false);
                logger.warning("Restored shard " + shardNo + " inited");
                break;
            case DROP:
                fsWriter.deleteAll();
                queueIds.clear();
                break;
            case RECOPY_OVER:
                break;
        }

        commonTaskor.getJobsManager().runJobs();
    }

    public synchronized void updateSchema() throws IOException {
        try {
            flushLock.lock();
            this.flush(false, true);
            this.doFlush(true);
            fsWriter.close(true);
            init();
        } finally {
            flushLock.unlock();
            this.flush(false, false);
        }
    }

    @Override
    public void updateQueueIds(Map<QueueShard, QueueId> ids) {
        for (final Map.Entry<QueueShard, QueueId> entry
            : ids.entrySet())
        {
            final QueueShard shard = entry.getKey();
            final QueueId id = entry.getValue();
            if (id.dirty()) {
                this.queueIds.put(shard, id);
            }
        }
        if (config.fakeQueueIdsPush() && !config.kosherShards()) {
            for (final Map.Entry<QueueShard, QueueId> entry
                : queueIds.entrySet())
            {
                final QueueShard shard = entry.getKey();
                final QueueId queueId = entry.getValue();
                final long maxQueueId = commonTaskor.maxSavedQueueId(shard);
                long dirtyQueueId =
                    commonTaskor.dirtyShardQueueId(shard);
                if (maxQueueId < dirtyQueueId && queueId.get() < maxQueueId) {
                    queueId.setId(maxQueueId);
                }
            }
        }
    }

    @Override
    public long queueId(final QueueShard shard) {
        long id = memoryIndex.queueId(shard);
        QueueId aid = queueIds.get(shard);
        if (aid == null && queueIdServiceFallback) {
            aid = queueIds.get(shard.defaultServiceShard());
        }
        if (aid != null) {
            if (id != -1) {
                if (aid.get() > id) {
                    id = aid.get();
                }
            } else {
                id = aid.get();
            }
        }
//        Logger.debug("shard<" + shardNo + ">.queueId(" + shard + "): "
//            + "id: " + id + ", aid: " + aid);
        return id;
    }

    public long savedQueueId(final QueueShard shard) {
        QueueId aid = queueIds.get(shard);
        if (aid == null && queueIdServiceFallback) {
            aid = queueIds.get(shard.defaultServiceShard());
        }
        if (aid != null) {
            return aid.get();
        }
        return -1;
    }

    public synchronized File createTemporaryDirectory() throws IOException
    {
        File tmp = File.createTempFile( "tmpdir", "", dir );
        String tmpName = tmp.getCanonicalPath();
        tmp.delete();
        tmp = new File( tmpName );
        tmp.mkdirs();
        return tmp;
    }

    public void deleteTemporaryDirectory( File dir ) throws IOException
    {
        File[] file = dir.listFiles();
        if( file != null )
        {
            for( int i = 0; i < file.length; i++ )
            {
                if( file[i].isDirectory() ) deleteTemporaryDirectory( file[i] );
                else file[i].delete();
            }
        }
        dir.delete();
    }

    public void initialize() throws IOException
    {
        logger.warning("Init shard: " + shardNo);
        File[] files = dir.listFiles();
        for( int i = 0; i < files.length; i++ )
        {
            if( files[i].isDirectory() )
            {
                if( files[i].getName().startsWith( "tmpdir" ) )
                {
                    logger.warning(
                        "Deleting unused temporary directory: "
                            + files[i].getName());
                    deleteTemporaryDirectory( files[i] );
                }
            }
        }
        reopenShardSearcher(0);
        previousCommit = System.currentTimeMillis();
        logger.warning("Shard initialized: " + shardNo);
    }

    public void move() throws IOException {
        if (fsSearcher != null) {
            fsSearcher.close();
            fsSearcher = null;
        }
        if (fsWriter != null) {
            fsWriter.close();
            fsWriter = null;
        }
        if (fsDir != null) {
            fsDir.close();
        }
        if (dir != null) {
            move(dir);
        }
    }

    public static void move(final int j, final File basePath)
        throws IOException
    {
        final File shardDir = new File(
            basePath,
            Integer.toString(j));
        move(shardDir);
    }

    public static void move(final File dir) throws IOException {
        dir.renameTo(new File(dir.getPath() + '-' + System.currentTimeMillis()));
    }

    private PrimaryKey getPrimaryKey(final Document doc,
        final Set<String> primaryKeyFields)
    {
        return null;
    }

    private void removeDuplicatesPrimaryKeyIndex(final IndexReader otherReader)
        throws IOException
    {
/*
        Set<String> primaryKeyFieldNames = config.primaryKey();
        FieldSelector fs = new MapFieldSelector(primaryKeyFieldNames);
        Set<String, String> primaryKeyFeilds = new HashSet<String, String>();
        for (int i = 0; i < otherReader.maxDoc(); i++) {
            Document doc = otherReader.document(i, fs);
            primaryKeyFields.clear();
            for (String field : primaryKeyFieldNames) {
                String value = doc.get(field);
                if (value == null) {
                    //invalid primary key
//                    primaryKeyFirlds.put(doc.get(
                }
            }
        }
*/
        Set<String> primaryKeyFieldNames = config.primaryKey();
        if (primaryKeyFieldNames.size() > 1) {
            logger.severe("IndexCopy: shard=" + shardNo
                + ", can't remove duplicates: "
                + "can't handle multifields primary keys");
            return;
        }
        final String keyFieldName = primaryKeyFieldNames.iterator().next();
        FieldConfig primaryKeyConfig = config.fieldConfigFast(keyFieldName);
        Fields otherFields = MultiFields.getFields(otherReader);
        if (otherFields == null) {
            return;
        }
        Terms otherTerms = otherFields.terms(keyFieldName);
        if (otherTerms == null) {
            return;
        }
        TermsEnum otherTe = otherTerms.iterator();
        Boolean ro = !indexingEnabled();
        flush(false, true);
        DumpSearcher thisSearcher = null;
        try {
            doFlush(true);
            thisSearcher = getDumpSearcher();

            Fields thisFields = MultiFields.getFields(thisSearcher.reader());
            if (thisFields == null) {
                return;
            }
            Terms thisTerms = thisFields.terms(keyFieldName);
            if (thisTerms == null) {
                return;
            }
            TermsEnum thisTe = thisTerms.iterator();

            int deletes = 0;
            BytesRef thisTerm = thisTe.next();
            BytesRef otherTerm = otherTe.next();
            while (thisTerm != null && otherTerm != null) {
                int cmp = thisTerm.compareTo(otherTerm);
                if (cmp < 0) {
                    thisTerm = thisTe.next();
                } else if (cmp > 0) {
                    otherTerm = otherTe.next();
                } else {
                    final String termText = thisTerm.utf8ToString();
                    final Term deleteTerm = new Term(keyFieldName, termText);
                    delete(deleteTerm);
                    if (logger.isLoggable(Level.FINER)) {
                        logger.finer("Deleting duplicate doc: " + deleteTerm);
                    }
                    deletes++;
                    if (deletes % 10000 == 0) {
                        fsWriter.commit(commitData());
                    }
                    thisTerm = thisTe.next();
                    otherTerm = otherTe.next();
                }
            }

            fsWriter.commit(commitData());
            if (logger.isLoggable(Level.INFO)) {
                logger.warning("Deleted <" + deletes + "> duplicate docs total.");
            }
            fsWriter.expungeDeletes(true);
            if (logger.isLoggable(Level.INFO)) {
                logger.warning("Expunge finished.");
            }
        } finally {
            flush(false, ro);
            if (thisSearcher != null) {
                thisSearcher.free();
            }
            doFlush(true);
        }
    }

    public void removeDuplicates(final IndexReader otherReader)
        throws IOException
    {
        if (config.primaryKey() == null) {
            logger.severe("IndexCopy: shard=" + shardNo
                + ", can't remove duplicates: part has no primary key");
        } else {
            removeDuplicatesPrimaryKeyIndex(otherReader);
        }
    }

    public void addIndexes(
        final boolean move,
        final Directory dir,
        final boolean replace,
        final Map<QueueShard, Long> newQueueIds)
        throws IOException
    {
        if (replace) {
            fsWriter.deleteAll();
            queueIds.clear();
        }
        fsWriter.addIndexes(move, dir);

        for (final Map.Entry<QueueShard, Long> entry : newQueueIds.entrySet()) {
            final QueueShard shard = entry.getKey();
            final Long queueId = entry.getValue();
            QueueId id = queueIds.get(shard);
            commonTaskor.clearDirtyQueueShard(shard);
            if (id == null) {
                QueueId newId = new QueueId(queueId);
                id = queueIds.putIfAbsent(shard, newId);
                if (id == null) {
                    continue;
                }
            }

            StringBuilder logSb =
                new StringBuilder("Updating queueIds, shard ");
            logSb.append(shard.toString());
            logSb.append(" current ");
            logSb.append(id.get());
            logSb.append(" new id ");
            logSb.append(queueId);

            logger.info(logSb.toString());

            long currentId;
            do {
                currentId = id.get();
            } while (currentId < queueId
                && !id.compareAndSet(currentId, queueId));
        }
        if (replace) {
            fsWriter.commit(commitData());
            indexReaderGeneration.incrementAndGet();
            multiIndexGeneration.incrementAndGet();
        }
        try {
            flush( false, null );
            doFlush(true);
        } catch(Exception e) {
            throw new IOException( e );
        }
    }

    public void delete(Term term) throws IOException {
        fsWriter.deleteDocuments(term);
    }

    private void warm(final IndexReader reader) throws IOException {
        prewarmTime.reset();
        fieldsCacheTime.reset();
        warmTime.reset();
        Compress.resetStats();
        IndexReader[] subReaders = reader.getSequentialSubReaders();
        if (subReaders != null) {
            for (IndexReader subReader: subReaders) {
                warmer.warmNoPrio(subReader, config.warmOnInit());
            }
        }
        logger.warning("Init times: prewarm: " + prewarmTime.sum()
            + ", fieldsCache: " + fieldsCacheTime.sum()
            + ", warm: " + warmTime.sum());
        logger.warning("IOStats: " + Compress.stats());
    }

    public void enqueueReopen(long readerGen)
    {
        commonTaskor.reopenShardsReader(shardNo, readerGen);
    }

    public ShardSearcher getShardSearcher() throws IOException
    {
        ShardSearcher searcher;
        long indexGen = indexReaderGeneration.get();
        long searchGen = searcherGeneration.get();
        long deletesGen = deletesGeneration.get();
        long deletesSearchGen = deletesSearcherGeneration.get();
        if( currentSearcher == null || indexGen > searchGen )
        {
            reopenShardSearcher( indexGen );
        }
        if( deletesGen > deletesSearchGen )
        {
            updateShardSearcher( deletesGen );
        }
        synchronized(getSearcherLock)
        {
            currentSearcher.incRef();
            return currentSearcher;
        }
    }

    public void reopenShardSearcher( long indexGen ) throws IOException {
//        searcherLock.writeLock().lock(); //This lock is to prevent inconsystent reader reopening while flushing memoryindex
        searcherLock.readLock().lock(); //This lock is to prevent inconsystent reader reopening while flushing memoryindex
        try{
            synchronized(reopenReaderLock) { //This one is to prevent double reopening from withing several threads
                if (indexGen <= searcherGeneration.get() && currentSearcher != null) return;

                if (currentSearcher == null) {
                    final IndexReader newReader = IndexReader.open(fsWriter, true);
                    DeleteHandlingIndexReader proxyReader = new DeleteHandlingIndexReader(newReader);
                    CacheTracker fieldsCacheTracker = new CacheTracker(newReader);
                    fieldsCacheTracker.incRef();
                    currentSearcher = new ShardSearcher(
                        new IndexSearcher(proxyReader),
                        proxyReader,
                        this,
                        fieldsCacheTracker);
                    currentSearcher.incRef();
                    if (indexGen == 0) {
//                        fieldsCache.loadCache(newReader, this);
                        warm(newReader);
                    }
                    fieldsCache.incRef(newReader);
                } else {
                    IndexReader newReader = IndexReader.open( fsWriter, true );
                    if (currentSearcher != null &&
                        ((DeleteHandlingIndexReader)currentSearcher.reader()).origReader() != newReader)
                    {
                        fieldsCache.incRef(newReader);
                        CacheTracker fieldsCacheTracker = new CacheTracker(newReader);
                        fieldsCacheTracker.incRef();
                        synchronized(getSearcherLock) { //This lock is for propper reader reference counting
                            ShardSearcher oldSearcher = currentSearcher;
                            DeleteHandlingIndexReader proxyReader = new DeleteHandlingIndexReader(newReader);

                            currentSearcher = new ShardSearcher(
                                new IndexSearcher(proxyReader),
                                proxyReader,
                                this,
                                fieldsCacheTracker);
                            currentSearcher.incRef();
                            if (deletesSinceFlush.size() > 0) { //There was deletes since last flush. So we cannot just reopen reader
                                                                //because it will not have new deletes. So we must reapply deletes to
                                                                //the new reader
                                proxyReader.deleteDocuments(deletesSinceFlush); //This will mark documents as deleted
                                proxyReader = proxyReader.clone(); //this will apply deletes to a reader
                                ShardSearcher newSearcher = new ShardSearcher(
                                    new IndexSearcher(proxyReader),
                                    proxyReader,
                                    this,
                                    fieldsCacheTracker);
                                fieldsCacheTracker.incRef();
                                newSearcher.incRef();
                                currentSearcher.free();
                                currentSearcher = newSearcher;
                            }
                            if (oldSearcher != null) {
                                oldSearcher.free();
                            }
                        }
                    }
                }
            }
            searcherGeneration.set( indexGen );
        } finally {
//            searcherLock.writeLock().unlock();
                searcherLock.readLock().unlock();
        }
    }

    public void updateShardSearcher( long deletesGen ) throws IOException {
        //see reopenShardSearcher for lock sequence desription
        searcherLock.readLock().lock();
        try {
            synchronized (reopenReaderLock) {
                if (deletesGen <= deletesSearcherGeneration.get()) return;
                synchronized (getSearcherLock) {
                    ShardSearcher oldSearcher = currentSearcher;
                    DeleteHandlingIndexReader proxyReader = ((DeleteHandlingIndexReader)currentSearcher.reader()).clone();
                    currentSearcher = new ShardSearcher(
                        new IndexSearcher(proxyReader),
                        proxyReader,
                        this,
                        oldSearcher.tracker());
                    oldSearcher.tracker().incRef();
                    currentSearcher.incRef();
                    if (oldSearcher != null) {
                        oldSearcher.free();
                    }
                }
                deletesSearcherGeneration.set(deletesGen);
            }
        } finally {
            searcherLock.readLock().unlock();
        }
    }

    public DumpSearcher getDumpSearcher() throws IOException {
        searcherLock.readLock().lock();
        try{
            final Map<QueueShard, Long> queueIdsCopy = new HashMap<>();
            final ShardSearcher searcher;
            synchronized(reopenReaderLock) {
                synchronized(getSearcherLock) {
                    searcher = currentSearcher;
                    searcher.incRef();
                }
            }
            for (final Map.Entry<QueueShard, QueueId> entry :
                queueIds.entrySet())
            {
                queueIdsCopy.put(entry.getKey(), entry.getValue().get());
            }
            return new DumpSearcher(searcher, queueIdsCopy) {
                @Override
                public IndexReader reader() throws IOException {
                    DeleteHandlingIndexReader reader =
                        (DeleteHandlingIndexReader) super.reader();
                    return reader.origReader();
                }};
        } finally {
                searcherLock.readLock().unlock();
        }
    }

    public Searcher getMultiSearcherDirty() throws IOException {
        long indexGen = multiIndexGeneration.get();
        long searchGen = multiSearcherGeneration.get();
        if (currentMultiSearcher == null) {
            return getMultiSearcher();
        }
        if (indexGen > searchGen
            && updateMultiSearcherInProgress.compareAndSet(false, true))
        {
            reopenMultiSearcherAsync();
        }
        synchronized(replaceMultiSearcherLock)
        {
            currentMultiSearcher.incRef();
            return currentMultiSearcher;
        }
    }

    public Searcher getMultiSearcher() throws IOException
    {
        long indexGen = multiIndexGeneration.get();
        long searchGen = multiSearcherGeneration.get();
        if( currentMultiSearcher == null || indexGen > searchGen )
        {
            synchronized(getMultiSearcherLock)
            {
                if( indexGen <= multiSearcherGeneration.get() && currentMultiSearcher != null )
                {
                    currentMultiSearcher.incRef();
                    return currentMultiSearcher;
                }
                Searcher.MultiSearcher oldSearcher = currentMultiSearcher;

                long time = System.currentTimeMillis();
                atomicOpLock.writeLock().lock();
                time = System.currentTimeMillis() - time;
                logger.fine("getMultiSearcher: aop.writeLock time: " + time);
                time = System.currentTimeMillis();
                searcherLock.readLock().lock();
                time = System.currentTimeMillis() - time;
                logger.fine("getMultiSearcher: searcherLock time: " + time);
                time = System.currentTimeMillis();
                try
                {
                    Searcher[] subSearchers = new Searcher[2];
                    subSearchers[0] = memoryIndex.getSearcher();
                    subSearchers[1] = getShardSearcher();
                    Searcher.MultiSearcher searcher = new Searcher.MultiSearcher( subSearchers );
                    searcher.incRef();
                    searcher.incRef();
                    synchronized (replaceMultiSearcherLock) {
                        currentMultiSearcher = searcher;
                        multiSearcherGeneration.set( indexGen );
                    }
                    return searcher;
                }
                finally
                {
                    searcherLock.readLock().unlock();
                    atomicOpLock.writeLock().unlock();
                    if( oldSearcher != null ) oldSearcher.free();
                    time = System.currentTimeMillis() - time;
                    logger.fine("getMultiSearcher: searcher replace time: " + time);
                }
            }
        }
        synchronized(replaceMultiSearcherLock)
        {
            currentMultiSearcher.incRef();
            return currentMultiSearcher;
        }
    }

    public void freeSearcher(final Searcher searcher) throws IOException {
        int refs = searcher.decRef();
        if (refs == 0) {
            searcher.close();
        }
    }

    public void search(
        final Query query,
        final FlushableCollector collector)
        throws IOException
    {
        Searcher searcher = getMultiSearcher();
        try {
            if (collector instanceof DocCollector) {
                ((DocCollector) collector).setFieldsCache(fieldsCache);
            }
            if (searcher != null) {
                try {
                    searcher.searcher().search(query, collector, true);
                } catch (CollectInterruptException ignore) {
                }
            }
            collector.flush();
        } finally {
            try {
                if (searcher != null) {
                    searcher.free();
                }
            } catch (IOException e) {
                logger.log(Level.SEVERE, "Can't free searcher resources", e);
            }
        }
    }

    public int getVersion()
    {
        int ret;
        ret = currentVersion;
        return ret;
    }

    private Map<String,String> commitData() {
        Map<String,String> commitData = new HashMap<String,String>();
        if (currentVersion != null) {
            commitData.put("version", currentVersion.toString());
        }
        for (final Map.Entry<QueueShard, QueueId> entry :
            queueIds.entrySet())
        {
            commitData.put(
                entry.getKey().toString(),
                Long.toString(entry.getValue().get()));
        }
        for (final Map.Entry<Prefix, Long> entry:
            prefixActivity.entrySet())
        {
            commitData.put(
                ACTIVE_PREFIX + entry.getKey().toString(),
                Long.toString(entry.getValue()));
        }
        return commitData;
    }

    public long lastFlush() {
        return lastFlush;
    }

    public void flush( boolean enqueue, Boolean ro ) throws IOException
    {
        lastFlush = System.currentTimeMillis();
        if (ro != null) {
            indexingEnabled.set(!ro);
            if (ro) {
                shardStatus.setInfo(IndexStatus.RO_USER);
            } else {
                shardStatus.setInfo(IndexStatus.RW);
            }
        }
        {
            long time = System.currentTimeMillis();
            deletes.set(0);
            memoryIndex.flush();
            if (logger.isLoggable(Level.INFO)) {
                logger.warning("PRE Flushing shard<" + shardNo + ">: ro=" + ro
                    + ", docs:" + memoryIndex.docCount()
                    + ", time: " + (System.currentTimeMillis() - time));
            }
            if (enqueue) {
                commonTaskor.flushShard( shardNo );
            }
        }
    }

    public Future doFlush(final boolean doWait) throws IOException
    {
        if (!flushLock.tryLock()) {
            if (!doWait) {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.warning("Shard<" + shardNo
                        + "> is already flushing. Dropping task");
                }
                return null;
            } else {
                flushLock.lock();
            }
        }
        Future queueIdsUpdaterWait = null;
        try { //No concurrent flushes
            long time = System.currentTimeMillis();
            if (logger.isLoggable(Level.WARNING)) {
                logger.warning("Flushing shard<" + shardNo + ">: docs: "
                    + memoryIndex.docCount() + " size=" + memoryIndex.size()
                    + " parts: " + memoryIndex.getPartsStats().toString());
            }
            AbstractPart part;
            while( (part = memoryIndex.firstPart()) != null ) {
                flushInProgress = true;
                try
                {
                    //part.asyncWriteTo will apply deletes with was belong to a part's live to the fsWriter first
                    //and the write actual part's data to the fsWriter's directory
                    List<org.apache.lucene.index.SegmentInfo> segmentInfos = part.asyncWriteTo(fsWriter);
                    memoryIndex.lockIndexing();

                    QueryProducer[] deletes = part.getDeletes();
                    if (deletes != null) {
                        if (logger.isLoggable(Level.WARNING)) {
                            logger.warning("Flushing shard<" + shardNo
                                + ">: deletes count: " + deletes.length);
                        }
                    } else {
                        if (logger.isLoggable(Level.WARNING)) {
                            logger.warning("Flushing shard<" + shardNo
                                + ">: deletes count: " + 0);
                        }
                    }
                    if (segmentInfos != null) {
                        if (logger.isLoggable(Level.CONFIG)) {
                            logger.config("AbstractPart<" + part + "> is now "
                                + segmentInfos);
                            logger.config("Flushing shard<" + shardNo
                                + ">: segmentInfos: " + segmentInfos);
                        }
                        //This will finally enable new segments
                        fsWriter.addIndexesInfos(segmentInfos); //This operation will lock searcherLock on success
                    } else {
                        //Part can return no new segments if it was empty
                        //but it is still can handle deletes so we must lock searcherLock by the hands
                        searcherLock.writeLock().lock();
                        if (logger.isLoggable(Level.CONFIG)) {
                            logger.config("Flushing shard<" + shardNo
                                + ">: segmentInfos: " + segmentInfos);
                        }
                    }
                    updateQueueIds(part.getQueueIds());
                    fsWriter.commit(commitData());

                    //This is deletes which came betwean commit() and lockIndexing()
                    //they are from a new part so we cannot commit them now.
                    List<QueryProducer> pendingDeletes = part.getPendingDeletes();
                    if (pendingDeletes != null) {
                        if (DEBUG) {
                            for (QueryProducer q : pendingDeletes) {
                                if (logger.isLoggable(Level.FINEST)) {
                                    logger.finest("Flushing shard<" + shardNo
                                        + ">: pending delete: " + q);
                                }
                            }
                        }
                        fsWriter.deleteDocuments(
                            pendingDeletes.toArray(new QueryProducer[pendingDeletes.size()]));
                    }

                    //now remove deletes belongs to this part from our deletes handling reader
                    if (deletes != null) {
                        for (int i = 0; i < deletes.length; i++) {
                            if (DEBUG) {
                                if (logger.isLoggable(Level.FINEST)) {
                                    logger.finest("Flushing shard<" + shardNo
                                        + ">: removing delete: "
                                        + deletesSinceFlush.remove());
                                }
                            } else {
                                deletesSinceFlush.remove();
                            }
                        }
                        if (logger.isLoggable(Level.WARNING)) {
                            logger.warning("Flushing shard<" + shardNo
                                + ">: deletesSinceFlush left: "
                                + deletesSinceFlush.size());
                        }
                    }
                    reopenShardSearcher(indexReaderGeneration.incrementAndGet());
                    memoryIndex.lockParts();
                    memoryIndex.removeFirstPart( part );
                    multiIndexGeneration.incrementAndGet();
                }
                finally
                {
                    memoryIndex.unlockParts();
                    memoryIndex.unlockIndexing();
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.warning("Flushing shard<" + shardNo
                            + ">: unlocking indexing.");
                    }
                    if (searcherLock.isWriteLockedByCurrentThread()) {
                        searcherLock.writeLock().unlock();
                    }
                    if (config.fakeQueueIdsPush() && !config.kosherShards()) {
                        queueIdsUpdaterWait =
                            commonTaskor.pushQueueIds(part.getQueueIds(), this);
                    }
                    flushInProgress = false;
                }
                part.free();
                part.closeJournal();
            }
            if (logger.isLoggable(Level.WARNING)) {
                logger.warning("Flushing shard<" + shardNo + ">: finished. time: "
                    + (System.currentTimeMillis() - time));
            }
            if (!closed) {
                fsWriter.maybeMerge();
                mergePolicy.setExpungeAll(false);
                mergePolicy.setConvert(false);
                mergePolicy.setForceMergeDeletesPctAllowed(autoExpungePct);
                fsWriter.expungeDeletes(false);
            }
            getMultiSearcher().free();
        } finally {
            flushLock.unlock();
        }
        return queueIdsUpdaterWait;
    }

    public final void reopenShardReader(long readerGen) throws IOException {
        if (indexReaderGeneration.get() > readerGen || flushInProgress) return;
        reopenShardSearcher(indexReaderGeneration.incrementAndGet());
        multiIndexGeneration.incrementAndGet();
    }

    public final void reopenShardReader() throws IOException {
        if (flushInProgress) return;
        reopenShardSearcher(indexReaderGeneration.incrementAndGet());
        multiIndexGeneration.incrementAndGet();
    }

    public void reopenMultiSearcherAsync() {
        commonTaskor.reopenMultiSearcherAsync(this);
    }

    public void reopenMultiSearcherSync() throws IOException {
        updateMultiSearcherInProgress.set(false);
        Searcher searcher = getMultiSearcher();
        searcher.free();
    }

    public final void reopenMultiSearcher() throws IOException {
        synchronized (getMultiSearcherLock) {
            if (currentMultiSearcher == null) return;
        }
        multiIndexGeneration.incrementAndGet();
        Searcher searcher = getMultiSearcher();
        searcher.free();
    }

    public int getDocsInMem() throws IOException
    {
        return memoryIndex.docCount();
    }

    public long getRamUsed()
    {
        return memoryIndex.size();
    }

    public AbstractPart.PartsStats getPartsStats() throws IOException
    {
        return memoryIndex.getPartsStats();
    }

    public long getActiveRamUsed()
    {
        return memoryIndex.activeSize();
    }

    public void optimize( int optimize ) throws IOException
    {
        if (!expungeLock.tryLock()) {
            throw new IOException("Can't run optimize on shard " + shardNo +
                ": expunge or optimize is already in progress");
        }
        try {
            fsWriter.optimize( optimize );
            flush(true, null);
        } finally {
            expungeLock.unlock();
        }
    }

    public void expunge( boolean all, Integer version, double minPct, boolean convert ) throws IOException
    {
        if (!expungeLock.tryLock()) {
            throw new IOException("Can't run expunge on shard " + shardNo +
                ": expunge or optimize is already in progress");
        }
        boolean ok = false;
        Integer savedVersion = currentVersion;
        try {
            mergePolicy.setExpungeAll( all );
            mergePolicy.setConvert(convert);
            mergePolicy.setForceMergeDeletesPctAllowed(minPct);
            if (version != null) {
                currentVersion = version;
            }
            fsWriter.expungeDeletes( true );
            flush(true, null);
            ok = true;
        } finally {
            if (!ok) {
                currentVersion = savedVersion;
            }
            mergePolicy.setConvert(false);
            mergePolicy.setExpungeAll(false);
            expungeLock.unlock();
        }
    }

    public boolean indexingEnabled() {
        return indexingEnabled.get();
    }

    public void close() throws IOException
    {
        closed = true;
        flush(false, true);
        memoryIndex.close();
        doFlush(true);
        memoryIndex.free();
        synchronized(getMultiSearcherLock)
        {
            if( currentMultiSearcher != null )
            {
                currentMultiSearcher.free();
            }
        }
        fsWriter.commit();
        while( currentSearcher.refs() > 0 ) currentSearcher.free();
        fsWriter.close(false);
        fsDir.close();
    }

    public long process(final JournalableMessage message)
        throws IOException, ParseException
    {
        if( closed ) throw new IOException( "Shard has been closed" );
        if (!indexingEnabled.get()) {
            throw new ReadOnlyIndexException(shardNo);
        }
        ProcessingResult result = memoryIndex.process(message);
        multiIndexGeneration.incrementAndGet();
        if( result.needFlush() )
        {
            try {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.warning("Shards <" + shardNo
                        + "> memory index has set request for flush");
                }
                flush(true, null);
            } catch (Exception ign) {
                logger.log(
                    Level.SEVERE,
                    "Exception caused by flush() while processing message",
                    ign);
            }
        }
        return result.ramUsage();
    }

    @Override
    public CloseableSearcher searcher() throws IOException {
        return new CloseableShardSearcher(getShardSearcher());
    }

    @Override
    public void primaryKeyAtomicOp(final PrimaryKey primaryKey,
        final AtomicOp op)
        throws IOException
    {
        PrimaryKey lock = keyStorage.acquire(primaryKey, primaryKey);
        boolean locked =  false;
        try {
            synchronized (lock) {
                atomicOpLock.readLock().lock();
                locked = true;
                op.run();
            }
        } finally {
            if( locked ) atomicOpLock.readLock().unlock();
            keyStorage.release(lock);
        }
    }

    public ReadWriteLock getReadWriteLock() {
        ReadWriteLock lock = prefixLockCache.poll();
        if (lock == null) {
            lock = new ReentrantReadWriteLock(false);
        }
        return lock;
    }

    public void returnReadWriteLock(final ReadWriteLock lock) {
        prefixLockCache.add(lock);
    }

    @Override
    public void prefixReadAtomicOp(final Prefix prefix,
        final PrefixOp op)
        throws IOException, ParseException
    {
        ReadWriteLock lock = prefixStorage.acquire(prefix, getReadWriteLock());
        try {
            lock.readLock().lock();
            op.run();
        } finally {
            lock.readLock().unlock();
            if (prefixStorage.release(prefix) == 0) {
                returnReadWriteLock(lock);
            }
        }
    }

    @Override
    public void prefixWriteAtomicOp(final Prefix prefix,
        final PrefixOp op)
        throws IOException, ParseException
    {
        ReadWriteLock lock = prefixStorage.acquire(prefix, getReadWriteLock());
        try {
            lock.writeLock().lock();
            op.run();
        } finally {
            lock.writeLock().unlock();
            if (prefixStorage.release(prefix) == 0) {
                returnReadWriteLock(lock);
            }
        }
    }

    @Override
    public void deleteDocuments(final QueryProducer query) throws IOException {
        DeleteHandlingIndexReader reader;
        searcherLock.readLock().lock(); //This lock is to prevent inconsystent reader reopening while flushing memoryindex
        try {
            if (!deletesSinceFlush.add(query)) {
                throw new IOException("Memory full?: can't buffer delete "
                    + "query: " + query);
            }
            ShardSearcher searcher;
            synchronized(getSearcherLock) {
                searcher = currentSearcher;
                searcher.incRef();
            }
            try {
                reader = ((DeleteHandlingIndexReader)searcher.reader());
                if (DEBUG) {
                    if (logger.isLoggable(Level.FINEST)) {
                        logger.finest(searcher + ": Deleting: " + query);
                    }
                }
                reader.deleteDocuments(query);
                long delgen = deletesGeneration.incrementAndGet();
                deletes.incrementAndGet();
                if (DEBUG) {
                    if (logger.isLoggable(Level.FINEST)) {
                        logger.finest("Shard Deleted: " + query);
                    }
                }
            } finally {
                searcher.free();
            }
        } finally {
            searcherLock.readLock().unlock(); //This lock is to prevent inconsystent reader reopening while flushing memoryindex
        }
//        }
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public Logger indexLogger() {
        return indexLogger;
    }

    public void traverseParts(final PartVisitor visitor) throws IOException {
        memoryIndex.traverseParts(visitor, Integer.MAX_VALUE);
    }

    public long numDocsLong() throws IOException {
        Searcher searcher = getMultiSearcher();
        try
        {
            return searcher.numDocsLong();
        }
        finally
        {
            searcher.free();
        }
    }

    public long indexSizeMb() {
        long size = 0;
        try {
            for (String file : fsDir.listAll()) {
                try {
                    size += fsDir.fileLength(file);
                } catch (IOException ignore) {
                    //file can be deleted while iterating
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return size / 1024 / 1024;
    }

    class CacheTracker {
        private final AtomicInteger refs = new AtomicInteger(0);
        private final IndexReader reader;

        CacheTracker(final IndexReader reader) {
            this.reader = reader;
        }

        public void incRef() {
            refs.incrementAndGet();
        }

        public int decRef() {
            int refs = this.refs.decrementAndGet();
            if (refs == 0) {
                fieldsCache.decRef(reader);
            }
            return refs;
        }
    }
}

