package ru.yandex.msearch;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.CgiParams;

public class ShardDropAndCopyHandler implements HttpRequestHandler {
    private final DatabaseManager dbManager;
    private final String password;

    public ShardDropAndCopyHandler(
        final DatabaseManager dbManager)
    {
        this.dbManager = dbManager;
        this.password = dbManager.daemonConfig().dropPassword();
    }

    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        PrefixedLogger logger =
            (PrefixedLogger) context.getAttribute(BaseHttpServer.LOGGER);

        try {
            CgiParams params = new CgiParams(request);
            Index index = dbManager.index(params);
            if (password == null) {
                throw new BadRequestException(
                    "Current Password is not valid");
            }

            Header passwordHeader = request.getFirstHeader("ticket");

            if (passwordHeader == null
                || !password.equalsIgnoreCase(
                    passwordHeader.getValue()))
            {
                throw new BadRequestException(
                    "Ugu, You want it, but Authentification failed");
            }

            int shardNo = params.getInt("shard");
            Shard shard = index.getShard(shardNo);
            if (shard == null) {
                throw new BadRequestException("Shard not found");
            }

            ShardDropPolicy dropPolicy =
                params.getEnum(ShardDropPolicy.class, "drop_policy", ShardDropPolicy.RECOPY_OVER);
            shard.dropAndRecopy(logger, dropPolicy);

        } catch (IOException ioe) {
            response.setStatusCode(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            response.setEntity(new StringEntity(
                "Failed to drop shard, " + ioe.getMessage(),
                StandardCharsets.UTF_8));
            logger.log(Level.WARNING, "Failed to drop shard", ioe);
            return;
        } catch (Throwable t) {
            HttpIndexerServer.handleException(t, response);
            return;
        }

        response.setStatusCode(HttpStatus.SC_OK);
    }

    @Override
    public String toString() {
        return "Completle remove index and shutdown server.";
    }
}
