package ru.yandex.msearch;

import java.io.IOException;

import java.text.ParseException;

import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.logging.Level;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.search.QueryProducer;

import ru.yandex.msearch.collector.BaseUpdatingCollector;
import ru.yandex.msearch.collector.KeepPrefixUpdatingCollector;
import ru.yandex.msearch.collector.UpdatingCollector;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.Prefix;

public class SimplePart extends AbstractPart {
    private static class Deleter implements PartVisitor {
        private final QueryProducer query;

        public Deleter(final QueryProducer query) {
            this.query = query;
        }

        @Override
        public void visit(final AbstractPart part) throws IOException {
            part.deleteDocuments(query);
        }
    }

    public SimplePart(
        final AnalyzerProvider analyzerProvider,
        final IndexManager indexManager,
        final IndexAccessor indexAccessor,
        final Map<QueueShard, QueueId> prevQueueIds,
        final long version,
        final DatabaseConfig config)
        throws IOException
    {
        super(
            analyzerProvider,
            indexManager,
            indexAccessor,
            prevQueueIds,
            version,
            config);
    }

    @Override
    public void cleanup() {
    }

    @Override
    public void add(final DocumentsMessage message, final Map<String, String> conditions)
        throws IOException, ParseException
    {
        Analyzer analyzer = analyzerProvider().indexAnalyzer(message.prefix());
        for (HTMLDocument doc: message.documents()) {
            doc.prepare(null);
            indexDocument(doc, analyzer);
        }
    }

    @Override
    public void modify(final DocumentsMessage message, final Map<String, String> conditions) {
        throw new UnsupportedOperationException(
            "Simple part can't handle modify messages");
    }

    @Override
    public void delete(final DocumentsMessage message) {
        throw new UnsupportedOperationException(
            "Simple part can't handle delete by primary key messages");
    }

    @Override
    public void delete(final DeleteMessage message)
        throws IOException, ParseException
    {
        QueryProducer query =
            message.query(analyzerProvider().searchAnalyzer(message.prefix()));
        traverseParts(new Deleter(query));
        indexAccessor().deleteDocuments(query);
    }

    @Override
    public void update(final DocumentsMessage message,
                       final Map<String, String> conditions,
                       final boolean addIfNotExists,
                       final boolean orderIndependentUpdate,
                       final Set<String> preservedFields) {
        throw new UnsupportedOperationException(
            "Simple part can't handle update by primary key messages");
    }

    @Override
    public void update(final UpdateMessage message,
                       final Map<String, String> conditions,
                       boolean addIfNotExists,
                       boolean orderIndependentUpdate)
        throws IOException, ParseException
    {
        if (addIfNotExists) {
            throw new ParseException("AddIfNotExists is not supported in Update By Query", 0);
        }
        Prefix prefix = message.prefix();
        PrefixingAnalyzerWrapper searchAnalyzer = analyzerProvider().searchAnalyzer(prefix);
        QueryProducer query = message.query(searchAnalyzer);
        boolean nonPrefixedQuery = searchAnalyzer.lastPrefixedField() == null;

        BaseUpdatingCollector col;
        if (nonPrefixedQuery) {
            col = new KeepPrefixUpdatingCollector(
                message.document(),
                message.queueId().phantomQueueId(),
                message.queueShard().service(),
                null,
                config,
                orderIndependentUpdate,
                message.context().logger());
        } else {
            col = new UpdatingCollector(
                    message.document(),
                    prefix,
                    message.queueId().phantomQueueId(),
                    message.queueShard().service(),
                    null,
                    orderIndependentUpdate,
                    config,
                    message.context().logger());
        }

        try (CloseableSearcher searcher = indexAccessor().searcher()) {
            searcher.searcher().search(query.produceQuery(), col, true);
            search(query.produceQuery(), col);
        }
        List<HTMLDocument> documents = col.documents();
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Going to update " + documents.size() + " documents");
        }
        traverseParts(new Deleter(query));
        indexAccessor().deleteDocuments(query);
        for (HTMLDocument document: documents) {
            if (nonPrefixedQuery && !message.prefix().equals(document.prefix())) {
                indexDocument(document, analyzerProvider().indexAnalyzer(document.prefix()));
            } else {
                indexDocument(document, analyzerProvider().indexAnalyzer(prefix));
            }
        }
    }

    @Override
    public void updateIfNotMatches(final UpdateIfNotMatchesMessage message,
                                   final Map<String, String> conditions,
                                   final boolean addIfNotExists,
                                   final boolean orderIndependentUpdate) {
        throw new UnsupportedOperationException(
            "Update is not supported for versioned indexes");
    }
}
