package ru.yandex.msearch.collector;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.TreeMap;

import org.apache.lucene.document.AllFieldSelector;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.IndexReader;

import ru.yandex.msearch.ClusteringConfig;
import ru.yandex.msearch.PrimaryKey;
import ru.yandex.msearch.collector.cluster.AbstractCluster;
import ru.yandex.msearch.collector.cluster.ClusterDoc;
import ru.yandex.msearch.collector.cluster.ClusterWithGroup;

public class ClusteringCollector<T extends AbstractCluster>
    extends BaseCollector
{
    private final Map<YaField, NavigableMap<Long, T>> clusters =
        new HashMap<>();
    private final ClusteringConfig config;
    private final ClusterFactory<T> clusterFactory;
    private final YaDocFieldVisitor fieldVisitor;

    public ClusteringCollector(
        final ClusteringConfig config,
        final ClusterFactory<T> clusterFactory)
    {
        this.config = config;
        this.clusterFactory = clusterFactory;
        fieldVisitor = new YaDocFieldVisitor(
            new MapFieldSelector(config.readFields()),
            config.fieldToIndex(),
            config.loadFields());
    }

    public List<ClusterWithGroup> clusters() {
        List<ClusterWithGroup> result = new ArrayList<>(clusters.size());
        for (Map.Entry<YaField, NavigableMap<Long, T>> entry
            : clusters.entrySet())
        {
            Iterator<T> iter;
            if (config.asc()) {
                iter = entry.getValue().values().iterator();
            } else {
                iter = entry.getValue().descendingMap().values().iterator();
            }
            T merged = null;
            while (iter.hasNext()) {
                T cluster = iter.next();
                if (cluster.size() < config.minClusterSize()) {
                    if (merged == null) {
                        merged = cluster;
                    } else {
                        merged.add(cluster);
                    }
                } else {
                    if (merged != null) {
                        result.add(
                            new ClusterWithGroup(merged, entry.getKey()));
                        merged = null;
                    }
                    result.add(new ClusterWithGroup(cluster, entry.getKey()));
                }
            }
            if (merged != null) {
                result.add(new ClusterWithGroup(merged, entry.getKey()));
            }
        }
        if (config.asc()) {
            Collections.sort(result);
        } else {
            Collections.sort(result, Collections.reverseOrder());
        }
        return result;
    }

    @Override
    public void processDoc(
        final IndexReader reader,
        final int base,
        final int document)
        throws IOException
    {
        checkAbort();
        YaDoc3 yadoc = new YaDoc3(config);
        fieldVisitor.doc(yadoc);
        reader.readDocument(document, fieldVisitor);

        if (!config.docProcessor().processWithFilter(yadoc)
            || !config.postFilter().test(yadoc))
        {
            return;
        }
        ClusterDoc doc = new ClusterDoc(yadoc, config);

        YaField group = config.groupFunc().group(yadoc);
        NavigableMap<Long, T> groupClusters = clusters.get(group);
        if (groupClusters == null) {
            groupClusters = new TreeMap<>();
            clusters.put(group, groupClusters);
        }
        long dateValue = doc.date();

        long minLocalDate = dateValue + config.localInterval();
        long maxLocalDate = dateValue - config.localInterval();
        NavigableMap<Long, T> clusters =
            groupClusters.subMap(maxLocalDate, true, minLocalDate, true);
        T cluster = null;
        Iterator<T> iter = clusters.values().iterator();
        long minDate = dateValue + config.interval();
        long maxDate = dateValue - config.interval();
        while (iter.hasNext()) {
            T current = iter.next();
            if ((minDate >= current.minDate()
                && maxDate <= current.maxDate())
                || (doc.hasValidCoords()
                    && minLocalDate >= current.minDate()
                    && maxLocalDate <= current.maxDate()
                    && current.isClose(
                        doc,
                        maxLocalDate,
                        minLocalDate,
                        config.distance())))
            {
                if (cluster == null) {
                    cluster = current;
                } else {
                    cluster.add(current);
                }
                iter.remove();
            }
        }
        Long date = new Long(dateValue);
        clusters = groupClusters.headMap(date, false);
        if (!clusters.isEmpty()) {
            T last = clusters.lastEntry().getValue();
            if (last.maxDate() >= maxDate
                || (doc.hasValidCoords()
                    && maxLocalDate <= last.maxDate()
                    && last.isClose(
                        doc,
                        maxLocalDate,
                        minLocalDate,
                        config.distance())))
            {
                clusters.pollLastEntry();
                if (cluster == null) {
                    cluster = last;
                } else {
                    cluster.add(last);
                }
            }
        }

        if (cluster == null) {
            Map<String, Map<String, T.Counter>> counters;
            int size = config.countDistinct().size();
            if (size == 0) {
                counters = null;
            } else {
                counters = new LinkedHashMap<>(size << 1);
                for (String field: config.countDistinct()) {
                    counters.put(
                        field,
                        new TreeMap<String, T.Counter>());
                }
            }
            cluster = clusterFactory.create(config, counters, doc);
        } else {
            cluster.add(doc);
        }

        for (String field: config.countDistinct()) {
            String value = yadoc.getString(field);
            if (value != null) {
                cluster.incrementCounter(field, value);
            }
        }
        groupClusters.put(cluster.minDate(), cluster);
    }

    @Override
    protected void checkAbort() throws IOException {
        config.ctx().checkAbort();
    }

    @Override
    public void close() {
        super.close();
        config.docProcessor().after();
    }

    public interface ClusterFactory<T extends AbstractCluster> {
        T create(
            ClusteringConfig config,
            Map<String, Map<String, T.Counter>> counters,
            ClusterDoc doc);
    }
}

