package ru.yandex.msearch.collector;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldFunctionIOException;
import ru.yandex.msearch.HTMLDocument;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.json.fieldfunction.ConditionsAccessor;
import ru.yandex.search.json.fieldfunction.FieldFunction;
import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.prefix.Prefix;

public class KeepPrefixUpdatingCollector extends BaseUpdatingCollector {
    protected final List<HTMLDocument> documents = new ArrayList<>();
    protected final Map<String, FieldFunction> fields;
    protected final long phantomQueueId;
    protected final String queueName;
    protected final ConditionsAccessor condAccessor;
    protected final DatabaseConfig config;
    protected final boolean orderIndependentUpdate;
    protected final Logger logger;

    public KeepPrefixUpdatingCollector(
        final Map<String, FieldFunction> fields,
        final long phantomQueueId,
        final String queueName,
        final ConditionsAccessor condAccessor,
        final DatabaseConfig config,
        final boolean orderIndependentUpdate,
        final Logger logger)
    {
        this.fields = fields;
        this.phantomQueueId = phantomQueueId;
        this.queueName = queueName;
        this.condAccessor = condAccessor;
        this.config = config;
        this.orderIndependentUpdate = orderIndependentUpdate;
        this.logger = logger;
    }

    @Override
    protected void processDoc(IndexReader reader, int base, int doc) throws IOException {
        try {
            Prefix docPrefix;
            Document luceneDoc = reader.document(doc);
            String prefixStr = luceneDoc.get(Config.PREFIX_FIELD_KEY);
            if (prefixStr != null) {
                docPrefix = config.indexPrefixParser().parse(prefixStr);
            } else {
                logger.log(
                    Level.SEVERE,
                    "Unable to update, no field" + Config.PREFIX_FIELD_KEY
                        + " in " + luceneDoc);
                throw new ParseException("Unable to parse null prefix for " + luceneDoc, 0);
            }

            HTMLDocument document =
                new HTMLDocument(
                    luceneDoc,
                    docPrefix,
                    phantomQueueId,
                    queueName,
                    orderIndependentUpdate,
                    config,
                    logger);
            document.updateWith(fields, condAccessor);
            documents.add(document);
        } catch (ParseException e) {
            throw new CorruptIndexException(e.getMessage());
        } catch (FieldFunctionException e) {
            throw new FieldFunctionIOException(e);
        }
    }

    public List<HTMLDocument> documents() throws IOException {
        flush();
        return Collections.unmodifiableList(documents);
    }
}
