package ru.yandex.msearch.collector;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonWriter;

public class MergeFuncFactory
{
    public static MergeFunc create( String type, boolean hasPrimaryKey )
    {
        if( type == null ) type = "values";
        type = type.toLowerCase().trim();
        switch (type) {
            case "all":
                if( hasPrimaryKey )
                {
                    return new MergeAllFast();
                }
                else
                {
                    return new MergeAllSlow();
                }

            case "count":
                if( hasPrimaryKey )
                {
                    return new MergeCountFast();
                }
                else
                {
                    return new MergeCountSlow();
                }

            case "none":
                return null;

            case "replace":
                return new ReplaceMergeFunc();

            default:
                if( hasPrimaryKey )
                {
                    return new MergeValuesFast();
                }
                else
                {
                    return new MergeValuesSlow();
                }
        }
    }

    private final static void formJsonValuesCommon(
        final JsonWriterBase jw,
        final Set<String> getFields,
        final Collection<YaDoc3> merged,
        final boolean skipNulls)
        throws IOException
    {
        if( merged != null && merged.size() > 0 )
        {
	    jw.key( "merged_docs" );
	    jw.startArray();
	    {
	        for( YaDoc3 doc : merged )
	        {
		    doc.writeJson( jw, getFields, skipNulls );
		}
	    }
	    jw.endArray();
	}
    }

    private final static void formJsonValuesCommon(Utf8JsonWriter jw, Set<String> getFields, Collection<YaDoc3> merged, boolean skipNulls ) throws IOException
    {
        if( merged != null && merged.size() > 0 )
        {
	    jw.key( "merged_docs" );
	    jw.startArray();
	    {
	        for( YaDoc3 doc : merged )
	        {
		    doc.writeUtf8Json( jw, getFields, skipNulls );
		}
	    }
	    jw.endArray();
	}
    }

    private static class MergeValuesSlow extends MergeFunc
    {
	protected HashSet<YaDoc3> merged = null;

	public void merge( YaDoc3 other )
	{
	    if( merged == null )
	    {
		merged = new HashSet<YaDoc3>();
		merged.add( other );
	    }
	    else
	    {
		if( !merged.contains( other ) )
		{
		    merged.add( other );
		}
	    }
	}

	public Collection<YaDoc3> merged()
	{
	    return merged;
	}

	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	}

	public MergeFunc clone()
	{
	    return new MergeValuesSlow();
	}

        public String toString()
        {
            return "MergeValuesSlow";
        }
    }

    private static class MergeValuesFast extends MergeFunc
    {
	protected ArrayList<YaDoc3> merged = null;

	public void merge( YaDoc3 other )
	{
	    if( merged == null )
	    {
		merged = new ArrayList<YaDoc3>();
	    }
	    merged.add( other );
	}

	public Collection<YaDoc3> merged()
	{
	    return merged;
	}

	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	}

	public MergeFunc clone()
	{
	    return new MergeValuesFast();
	}

        public String toString()
        {
            return "MergeValuesFast";
        }
    }

    public static class MergeCountSlow extends MergeValuesSlow
    {
	public Collection<YaDoc3> merged()
	{
	    return null;
	}

	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public MergeFunc clone()
	{
	    return new MergeCountSlow();
	}

        public String toString()
        {
            return "MergeCountSlow";
        }
    }

    public static class MergeCountFast extends MergeFunc
    {
	private int merged = 0;

	public void merge( YaDoc3 other )
	{
	    ++merged;
	}

	public Collection<YaDoc3> merged()
	{
	    return null;
	}

	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    jw.key( "merged_docs_count" );
	    jw.value( merged );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    jw.key( "merged_docs_count" );
	    jw.value( merged );
	}

	public MergeFunc clone()
	{
	    return new MergeCountFast();
	}

        public String toString()
        {
            return "MergeCountFast";
        }
    }

    public static class MergeAllSlow extends MergeValuesSlow
    {
	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public MergeFunc clone()
	{
	    return new MergeAllSlow();
	}

        public String toString()
        {
            return "MergeAllSlow";
        }
    }

    public static class MergeAllFast extends MergeValuesFast
    {
	public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls ) throws IOException
	{
	    formJsonValuesCommon( jw, getFields, merged, skipNulls );
	    jw.key( "merged_docs_count" );
	    jw.value( merged == null ? 0 : merged.size() );
	}

	public MergeFunc clone()
	{
	    return new MergeAllFast();
	}

        public String toString()
        {
            return "MergeAllFast";
        }
    }

    public static class ReplaceMergeFunc extends MergeFunc {
        @Override
        public void merge(YaDoc3 other) {
        }

        @Override
        public Collection<YaDoc3> merged() {
            return null;
        }

        @Override
        public void formJson(
            final JsonWriterBase jw,
            final Set<String> getFields,
            final boolean skipNulls)
            throws IOException
        {
        }

        @Override
        public void formJson(Utf8JsonWriter jw, Set<String> getFields, boolean skipNulls)
            throws IOException
        {
        }

        @Override
        public MergeFunc clone() {
            return this;
        }

        @Override
        public String toString() {
            return "ReplaceMergeFunc";
        }
    }
}
