package ru.yandex.msearch.collector;

import java.io.IOException;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import java.util.logging.Level;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldMapping;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.FieldInfo;
import org.apache.lucene.index.FieldInfos;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.RequestContext;
import ru.yandex.msearch.SearchRequest;
import ru.yandex.msearch.SearchResultsConsumer;
import ru.yandex.msearch.collector.docprocessor.ModuleFieldsAggregator;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.msearch.fieldscache.FieldCache;
import ru.yandex.msearch.collector.aggregate.Aggregator;
import ru.yandex.msearch.collector.docprocessor.DocProcessor;
import ru.yandex.msearch.collector.docprocessor.NullDocProcessor;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.group.GroupFuncFactory;
import ru.yandex.msearch.collector.group.NullGroupFunc;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunction;
import ru.yandex.msearch.collector.postfilter.NullPostFilter;
import ru.yandex.msearch.collector.postfilter.PostFilter;
import ru.yandex.msearch.collector.sort.NullSortFunc;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.collector.sort.SortFuncFactory;

import ru.yandex.search.YandexScorer;
import ru.yandex.search.YandexScorerFactory;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.Prefix;

import ru.yandex.util.string.StringUtils;

public class PassThruCollector extends DocCollector implements SearchParams
{
    public static final int DELAYED_OUTPUT_DOC_COUNT = 10000;

    private final SearchResultsConsumer consumer;
    private final int maxDelayCount;
    private final int offset;
    private final int length;
    private final Set<String> getFields;
    private final GroupFunc groupFunc;
    private final DocProcessor docProcessor;
    private final PostFilter postFilter;
    private final FieldToIndex fieldToIndex;
    private final FieldConfig[] loadFields;
    private final boolean earlyInterrupt;

    private AtomicReaderContext currentContext;
    private IndexReader currentReader;
    private int totalCount = 0;
    private int uniqCount = 0;
    private boolean debug = false;
    private boolean hitsStarted = false;
    private boolean passThru = false;
    private int pos = 0;
    private boolean ignoreEmpty;
    private HashSet<YaField> deduplicator;
    private FieldsCache fieldsCache;
    private OuterGroupFunction outerGroup;
    private RequestContext ctx;
    private ArrayList<ResultsBucket> delayedResults;
    private int docCount;
    private int bufferedDocs;
    private YaDoc3Delayed[] docsBuffer;
    private YandexScorer scorer = null;
    private final YandexScorerFactory scorerFactory;
    private final YaDocVisitor visitor;
    private final boolean easyCounting;

    private static class ResultsBucket {
        public final YaDoc3Delayed[] docs;
        public IndexReader reader;
        public ResultsBucket(
            final YaDoc3Delayed[] docs,
            final IndexReader reader)
        {
            this.docs = docs;
            this.reader = reader;
        }
    }

    public static String checkRequestParams(final SearchRequest request) {
        if (request.mergeFunc() != null) {
            return "PassThruCollector can not be used with non NULL MergeFunc";
        }
        if (request.aggregator() != null) {
            return "PassThruCollector can not be used with non NULL Aggregator";
        }
        if (!(request.sort() instanceof NullSortFunc)) {
            return "PassThruCollector can not be used with non NULL SortFunc";
        }
        return null;
    }

    public PassThruCollector(
        final SearchRequest request,
        final SearchResultsConsumer consumer,
        final int maxDelayCount)
    {
        this.consumer = consumer;
        this.offset = request.offset();
        this.length = request.length();
        if (length < Integer.MAX_VALUE && length != 0) {
            int lengthTotal = offset + length;
            this.maxDelayCount = Math.min(maxDelayCount, lengthTotal + 1);
        } else {
            this.maxDelayCount = maxDelayCount;
        }
        this.outerGroup = request.outerGroup();
        this.ctx = request.ctx();
        this.earlyInterrupt = request.earlyInterrupt();

        this.getFields = request.getFields();
        this.docProcessor = request.docProcessor();
        this.postFilter = request.postFilter();

        this.groupFunc = request.group();
        this.scorerFactory = request.scorerFactory();

        DatabaseConfig config = request.config();
        Set<String> readFields = checkGetFields(request);

        CollectingFieldToIndex fieldToIndex = request.fieldToIndex();
        for (String field: readFields) {
            fieldToIndex.indexFor(field);
        }
        for (String field: scorerFactory.outFields()) {
            fieldToIndex.indexFor(field);
        }

        this.fieldToIndex = fieldToIndex.compactFieldToIndex();
        loadFields = new FieldConfig[this.fieldToIndex.fieldsCount()];
        for (String field: readFields) {
            loadFields[this.fieldToIndex.indexFor(field)] =
                config.fieldConfigFast(field);
        }
        visitor = new YaDocVisitor(
            readFields,
            this.fieldToIndex,
            loadFields);

        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine(
                "New PassThruCollector (" + maxDelayCount + "): getFields:"
                + (getFields != null ? getFields.toString() : "null")
                + " / readFields: "
                + (readFields != null ? readFields.toString() : "null")
                + " / groupFunc: " + groupFunc);
        }

        deduplicator = new HashSet<>(1000, (float)0.5);

        ignoreEmpty = false;
        delayedResults =
            new ArrayList<ResultsBucket>(5);
        docsBuffer = new YaDoc3Delayed[32];
        docCount = 0;
        bufferedDocs = 0;
        if (earlyInterrupt) {
            easyCounting = true;
        } else {
            easyCounting =
                groupFunc == NullGroupFunc.INSTANCE
                && postFilter == NullPostFilter.INSTANCE
                && docProcessor == NullDocProcessor.INSTANCE
                && outerGroup == null;
        }
    }

    private YaDocVisitor visitor(final YaDoc3 doc) {
        visitor.doc(doc);
        return visitor;
    }

    @Override
    public GroupFunc groupFunc() {
        return groupFunc;
    }

    @Override
    public void setPrefix(final Prefix prefix) {
    }

    @Override
    public SortFunc sortFunc() {
        return NullSortFunc.INSTANCE;
    }

    @Override
    public FieldToIndex fieldToIndex() {
        return fieldToIndex;
    }

    @Override
    public Set<String> mergeFields() {
        return getFields;
    }

    @Override
    public MergeFunc mergeFunc() {
        return null;
    }

    @Override
    public void setDebug(boolean debug) {
        this.debug = debug;
    }

    private Set<String> checkGetFields(final SearchRequest request) {
        final DatabaseConfig config = request.config();
        Set<String> readFields = new HashSet<>(getFields.size() << 1);
        for (String field: getFields) {
            readFields.add(StringHelper.intern(field));
        }
        for (FieldsFunction function: new FieldsFunction[] {
                groupFunc,
                outerGroup,
                postFilter
            })
        {
            if (function != null) {
                for (String field: function.loadFields()) {
                    FieldConfig fieldConfig = config.fieldConfig(field);
                    if (fieldConfig != null && fieldConfig.store()) {
                        readFields.add(field);
                    }
                }
            }
        }

        if (docProcessor != null) {
            ModuleFieldsAggregator aggregator = new ModuleFieldsAggregator();
            docProcessor.apply(aggregator);
            if (!request.dpReadOutFields()) {
                readFields.removeAll(aggregator.outFields());
            }
            readFields.addAll(aggregator.loadFields());
        }
        return readFields;
    }

    public void setFieldsCache(FieldsCache fieldsCache) {
        this.fieldsCache = fieldsCache;
    }

    public boolean acceptsDocsOutOfOrder() {
        return true;
    }

    public void setNextReader(
        final AtomicReaderContext readerContext)
        throws IOException
    {
        if (!passThru && currentContext != null && docCount > 0) {
            delayedResults.add(new ResultsBucket(Arrays.copyOf(docsBuffer,
                docCount), currentReader));
            docCount = 0;
        }
        currentContext = readerContext;
        currentReader = readerContext.reader;
    }

    public void setScorer(Scorer scorer) {
        this.scorer = scorerFactory.scorer(currentContext, scorer);
    }

    public void collect(int docId) throws IOException {
        if (passThru) {
            if (easyCounting && pos >= length) {
                if (earlyInterrupt) {
                    throw new CollectInterruptException();
                }
                uniqCount++;
                return;
            }
            YaDoc3Delayed doc = new YaDoc3Delayed(this, docId);
            scorer.preprocess(docId);
            scorer.process(docId, doc);
            doc = produceDocument(readDocument(currentReader, doc));
            if (doc != null) {
                consumer.document(doc, getFields);
            }
            return;
        }
        if (bufferedDocs < maxDelayCount) {
            final YaDoc3Delayed doc = new YaDoc3Delayed(this, docId);
            scorer.preprocess(docId);
            scorer.process(docId, doc);
            docsBuffer[docCount++] = doc;
            if (docCount == docsBuffer.length) {
                docsBuffer =
                    Arrays.copyOf(docsBuffer, docsBuffer.length << 1);
            }
            bufferedDocs++;
        } else {
            passThru = true;
            flushDelayed();
            collect(docId);
        }
    }

    private void flushAndForget() throws IOException {
        for (ResultsBucket rb : delayedResults) {
            Arrays.sort(rb.docs);
            for (YaDoc3Delayed doc : rb.docs) {
                doc = produceDocument(readDocument(rb.reader, doc));
                if (doc != null) {
                    consumer.document(doc, getFields);
                }
            }
        }
        for (int i = 0; i < docCount; i++) {
            YaDoc3 doc =
                produceDocument(readDocument(currentReader, docsBuffer[i]));
            if (doc != null) {
                consumer.document(doc, getFields);
            }
        }
    }

    private ArrayList<YaDoc3> flushAndRemember() throws IOException {
        ArrayList<YaDoc3> docs = new ArrayList<>(bufferedDocs);
        for (ResultsBucket rb : delayedResults) {
            Arrays.sort(rb.docs);
            for (YaDoc3Delayed doc : rb.docs) {
                doc = produceDocument(readDocument(rb.reader, doc));
                if (doc != null) {
                    docs.add(doc);
                }
            }
        }
        for (int i = 0; i < docCount; i++) {
            YaDoc3 doc = produceDocument(readDocument(currentReader, docsBuffer[i]));
            if (doc != null) {
                docs.add(doc);
            }
        }
        return docs;
    }

    private void flushDelayed() throws IOException {
        if (passThru) {
            consumer.startHits();
            flushAndForget();
        } else {
            List<YaDoc3> docs = flushAndRemember();
            //flushDelayed called from close
            consumer.totalHitsCount(totalCount);
            consumer.uniqHitsCount(uniqCount);
            consumer.startHits();
            for (YaDoc3 doc : docs) {
                consumer.document(doc, getFields);
            }
        }
        delayedResults = null;
        docsBuffer = null;
        docCount = 0;
    }

    private YaDoc3Delayed readDocument(
        final IndexReader reader,
        final YaDoc3Delayed yaDoc)
        throws IOException
    {
//        Document doc = reader.document(yaDoc.docId, fs);
//        if (doc == null) return null;
//        yaDoc.readFields(doc, loadFields);
        reader.readDocument(yaDoc.docId, visitor(yaDoc));
        totalCount++;
        return yaDoc;
    }

    public void flush() throws IOException {
    }

    public void close() throws IOException {
        if (!passThru) {
            flushDelayed();
            consumer.endHits();
        } else {
            consumer.endHits();
            consumer.totalHitsCount(totalCount);
            consumer.uniqHitsCount(uniqCount);
        }
        //do not close doc processors before flush
        if (docProcessor != null) {
            docProcessor.after();
        }
    }

    public Set<YaDoc3> hits() throws IOException {
        throw new UnsupportedOperationException(
            "PassThruCollector does not collets hits");
    }

    public int uniqCount() {
        return uniqCount;
    }

    public int getTotalCount() {
	return totalCount;
    }

    public void setIgnoreEmpty(boolean i) {
	ignoreEmpty = i;
    }

    final private boolean duplicated(YaDoc3 doc) {
	return outerGroup.modifyAndCheckDuplicated(doc);
    }

    final private YaDoc3Delayed produceDocument(
        final YaDoc3Delayed doc)
        throws IOException
    {
        if (debug) {
            doc.setField(
                "#debug_reader",
                new YaField.StringYaField(
                    StringUtils.getUtf8Bytes(currentReader.toString())));
        }
        if (!docProcessor.processWithFilter(doc) || !postFilter.test(doc)) {
            totalCount--;
            return null;
        }
	if (outerGroup != null) {
	    if (duplicated(doc)) {
		if (!ignoreEmpty) return null;
	    }
	}
	doc.initSort();
        YaField group = doc.getGroupField();

        boolean produce = false;
	if (uniqCount >= offset && pos < length) {
	    produce = true;
	}
	if (group != null) {
	    if (deduplicator.add(group)) {
	        uniqCount++;
	    } else {
	        produce = false;
	    }
	} else {
	    uniqCount++;
	}
	if (produce) {
            ++pos;
	    return doc;
	}
	return null;
    }

    final public Set<String> getGetFields() {
        return getFields;
    }

    final public RequestContext context() {
        return ctx;
    }

    final public int offset() {
        return offset;
    }

    @Override
    final public int length() {
        return length;
    }

    @Override
    final public void length(int length) {
        throw new UnsupportedOperationException("length");
    }

    private static class YaDoc3Delayed extends YaDoc3 {
        public final int docId;

        YaDoc3Delayed(final SearchParams params, final int docId) {
            super(params);
            this.docId = docId;
        }
    }

    private static final class YaDocVisitor extends AbstractFieldVisitor {
        private final Set<String> fields;
        private final FieldConfig[] configs;
        private YaDoc3 doc;

        public YaDocVisitor(
            final Set<String> fields,
            final FieldToIndex fieldToIndex,
            final FieldConfig[] configs)
        {
            super(fieldToIndex);
            this.fields = fields;
            this.configs = configs;
        }

        public void doc(final YaDoc3 doc) {
            this.doc = doc;
        }

        @Override
        public FieldSelectorResult fieldSelectorResult(final String key) {
            if (fields.contains(key)) {
                return FieldSelectorResult.LOAD;
            } else {
                return FieldSelectorResult.NO_LOAD;
            }
        }

        @Override
        public void storeFieldValue(final int index, final byte[] value) {
            doc.setField(index, configs[index].type().create(value));
        }

        @Override
        public void storeFieldSize(final int index, final int size) {
        }
    }
}
