package ru.yandex.msearch.collector;

import java.io.IOException;
import java.util.Comparator;
import java.util.Set;
import java.util.logging.Level;

import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.TermToBytesRefAttribute;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.ReusableStringReader;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.store.Lockable;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.OpenBitSet;

import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.RequestContext;
import ru.yandex.msearch.SearchRequest;
import ru.yandex.msearch.SearchResultsConsumer;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.group.NullGroupFunc;
import ru.yandex.msearch.collector.group.SimpleGroupFunc;
import ru.yandex.msearch.collector.sort.SimpleSortFunc;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.fieldscache.CacheInput;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.string.StringUtils;

//import java.util.SortedSet;

public class PruningCollector extends ParametrizedDocCollector {
    public static final int MAX_FLUSH_INTERVAL = 10000;
    public static final int MIN_FLUSH_INTERVAL = 100;
    public static final BytesRef REVERSE_SEEK_SUFFIX =
        new BytesRef(new byte[]{-1, -1, -1, -1});

    private final OpenBitSet docIdsSet = new OpenBitSet();
    private final SearchRequest request;
    private final SearchResultsConsumer consumer;
    private final int limit;
    private final boolean debug;
    private final int maxFlushInterval;
    private final ParametrizedDocCollector nextCollector;
    private final String pruningField;
    private final boolean prefixed;
    private final boolean reverse;
    private final String groupField;
    private final String sortField;
    private final String pruningGroupField;
    private final FieldSelector groupFieldSelector;
    private final boolean groupPrefixed;
    private final Comparator<BytesRef> termComp;
    private final String userIdField;
    private final BytesRef userIdTerm;
    private IndexReader currentReader = null;
    private int docIdsCount = 0;
    private int maxDocId = 0;
    private BytesRef maxTerm;
    private Prefix prefix = null;
    private int totalCollected = 0;
    private int totalCollectedWithPruning = 0;
    private int totalPruned = 0;
    private int totalCount = 0;
    private int pruned = 0;
    private int collected = 0;
    private int flushInterval = 0;
    private int docsRead = 0;
    private int totalDocsRead = 0;
    private int docsFromCache = 0;
    private int totalDocsFromCache = 0;
    private int length;
    private final PrefixingAnalyzerWrapper analyzer;
    private ReusableStringReader stringReader;
    private FieldsCache fieldsCache = null;

    private static String forcedPruningField(final SearchRequest request) {
        String collectorName = request.forcedCollectorName();
        if (collectorName != null) {
            int idx1 = collectorName.indexOf('(');
            if (idx1 != -1) {
                int idx2 = collectorName.indexOf(')', idx1 + 1);
                if (idx2 != -1) {
                    return collectorName.substring(idx1 + 1, idx2);
                }
            }
        }
        return null;
    }

    public static String checkRequestParams(final SearchRequest request) {
        String pruningField = forcedPruningField(request);
        FieldConfig pruningFieldConfig;
        if (pruningField == null) {
            SortFunc sortFunc = request.sort();
            if (!(sortFunc instanceof SimpleSortFunc)) {
                return "PruningCollector can only be used with SimpleSortFunc "
                    + "(i.e.: single field sort)";
            }
            pruningField = sortFunc.loadFields().iterator().next();
            pruningFieldConfig =
                request.config().fieldConfigFast(pruningField);
        } else {
            pruningFieldConfig = request.config().fieldConfig(pruningField);
        }
        if (pruningFieldConfig == null || !pruningFieldConfig.index()) {
            return "PruningCollector requires pruning field " + pruningField
                + " to be indexed";
        }
        GroupFunc groupFunc = request.group();
        if (!(groupFunc instanceof NullGroupFunc)) {
            if (groupFunc instanceof SimpleGroupFunc) {
                String groupField = request.pruningGroupField();
                if (groupField == null) {
                    groupField = groupFunc.loadFields().iterator().next();
                }
                FieldConfig groupFieldConfig =
                    request.config().fieldConfigFast(groupField);
                if (groupFieldConfig == null || !groupFieldConfig.index()) {
                    return "PruningCollector requires group field "
                        + groupField + " to be indexed (config: "
                        + groupFieldConfig;
                }
                if (!groupFieldConfig.searchFilters().isEmpty()
                    && !request.forcePruningGroupField())
                {
                    return "PruningCollector requires empty"
                        + " filters list for group field <"
                        + groupField + ">: "
                        + groupFieldConfig.searchFilters();
                }
            } else {
                return "PruningCollector required either null"
                    + " or simple group func";
            }
        }
        String userIdField = request.userIdField();
        if (userIdField != null) {
            FieldConfig userIdFieldConfig =
                request.config().fieldConfig(userIdField);
            if (userIdFieldConfig == null || !userIdFieldConfig.index()) {
                return "PruningCollector requires user id field " + userIdField
                    + " to be indexed";
            }
            String userIdTerm = request.userIdTerm();
            if (userIdTerm == null || userIdTerm.isEmpty()) {
                return "PruningCollector requires non empty user id term";
            }
        }
        return null;
    }

    public PruningCollector(
        final SearchRequest request,
        final SearchResultsConsumer consumer)
    {
        this.request = request;
        this.consumer = consumer;
        this.limit = request.offset() + request.length();
        this.length = request.length();
        this.debug = request.debug();
        maxFlushInterval =
            Math.max(MIN_FLUSH_INTERVAL, Math.min(MAX_FLUSH_INTERVAL, limit));
        nextCollector = new SortedCollector(request, null);
        String pruningField = forcedPruningField(request);
        FieldConfig pruningFieldConfig;
        sortField = request.sort().loadFields().iterator().next();
        if (pruningField == null) {
            this.pruningField = request.sort().loadFields().iterator().next();
            pruningFieldConfig =
                request.config().fieldConfigFast(this.pruningField);
        } else {
            this.pruningField = pruningField;
            pruningFieldConfig = request.config().fieldConfig(pruningField);
        }
        if (request.ctx().logger().isLoggable(Level.FINE)) {
            request.ctx().logger().fine(
                "New PruningCollector: sortFunc: " + request.sort()
                + ", pruning field: " + this.pruningField);
        }
        prefixed = pruningFieldConfig.prefixed();
        reverse = !request.sortDirection();

        GroupFunc groupFunc = request.group();
        if (groupFunc instanceof NullGroupFunc) {
            groupField = null;
            pruningGroupField = null;
            groupFieldSelector = null;
            groupPrefixed = false;
            analyzer = null;
        } else {
            groupField = groupFunc.loadFields().iterator().next();
            if (request.pruningGroupField() == null) {
                pruningGroupField = groupField;
            } else {
                pruningGroupField = request.pruningGroupField();
            }
            groupFieldSelector = new MapFieldSelector(groupField);
            FieldConfig groupFieldConfig =
                request.config().fieldConfigFast(pruningGroupField);
            groupPrefixed = groupFieldConfig.prefixed();
            if (!groupFieldConfig.searchFilters().isEmpty()) {
                analyzer = request.index().searchAnalyzer();
            } else {
                analyzer = null;
            }
        }

        if (reverse) {
            termComp =
                BytesRef.getUTF8SortedAsUnicodeComparator().reversed();
        } else {
            termComp = BytesRef.getUTF8SortedAsUnicodeComparator();
        }

        String userIdField = request.userIdField();
        if (userIdField == null) {
            this.userIdField = null;
            this.userIdTerm = null;
        } else {
            String userIdTerm = request.userIdTerm();
            FieldConfig userIdFieldConfig =
                request.config().fieldConfig(userIdField);
            if (userIdFieldConfig == null
                || !userIdFieldConfig.index()
                || userIdTerm == null
                || userIdTerm.isEmpty())
            {
                this.userIdField = null;
                this.userIdTerm = null;
            } else {
                this.userIdField = userIdField;
                this.userIdTerm = new BytesRef(userIdTerm);
            }
        }
    }

    @Override
    public void setPrefix(final Prefix prefix) {
        this.prefix = prefix;
        if (analyzer != null) {
            analyzer.setPrefix(prefix.toString());
        }
        maxTerm = null;
    }

    @Override
    public GroupFunc groupFunc() {
        return request.group();
    }

    @Override
    public SortFunc sortFunc() {
        return request.sort();
    }

    @Override
    public FieldToIndex fieldToIndex() {
        return nextCollector.fieldToIndex();
    }

    @Override
    public Set<String> mergeFields() {
        return nextCollector.mergeFields();
    }

    @Override
    public MergeFunc mergeFunc() {
        return nextCollector.mergeFunc();
    }

    @Override
    public void setDebug(boolean debug) {
        nextCollector.setDebug(debug);
    }

    public void setFieldsCache(FieldsCache fieldsCache) {
        this.fieldsCache = fieldsCache;
        nextCollector.setFieldsCache(fieldsCache);
    }

    public boolean acceptsDocsOutOfOrder() {
        return true;
    }

    public boolean canPopulateFromCache(final IndexReader reader) {
        return false;
    }

    private void doPrune() throws IOException {
        totalCount += docIdsCount;
        prune();
        docIdsSet.clear(0, maxDocId + 1);
        maxDocId = 0;
        docIdsCount = 0;
        if (request.ctx().logger().isLoggable(Level.FINE)
            && groupField != null)
        {
            request.ctx().logger().fine("Grouping docsProcessed: "
                + "docsRead: " + docsRead
                + "docsFromCache: " + docsFromCache);
        }
    }

    public void setNextReader(AtomicReaderContext readerContext) throws IOException {
        if (currentReader != null && docIdsCount > 0) {
            doPrune();
            docsRead = 0;
            docsFromCache = 0;
        }
        nextCollector.setNextReader(readerContext);
        currentReader = readerContext.reader;
    }

    private void passThrought() throws IOException {
        final DocIdSetIterator iter = docIdsSet.iterator();
        for (
            int docId = iter.nextDoc();
            docId != DocIdSetIterator.NO_MORE_DOCS
                && docId <= maxDocId;
            docId = iter.nextDoc())
        {
            nextCollector.collectPreprocessed(docId);
            checkAbort();
        }
        totalPruned += docIdsCount;
        nextCollector.length(
            Math.max(length,nextCollector.offset() + docIdsCount));
        final int flushed = nextCollector.countedFlush();
        nextCollector.length(this.length);
        totalCollected += flushed;
    }

    private void forceFlush() throws IOException {
        if (pruned > 0) {
            nextCollector.length(
                Math.max(length,nextCollector.offset() + pruned));
            int flushed = nextCollector.countedFlush();
            nextCollector.length(this.length);
            collected += flushed;
            totalCollected += flushed;
            totalCollectedWithPruning += flushed;
            totalPruned += pruned;
            pruned = 0;
        }
    }

    private void doCollect(final int docId) throws IOException {
        nextCollector.collectPreprocessed(docId);
        ++pruned;
        if (pruned >= flushInterval) {
            nextCollector.length(
                Math.max(length, nextCollector.offset() + pruned));
//            System.err.println("SET LENGTH: " + nextCollector.length());
            int flushed = nextCollector.countedFlush();
            nextCollector.length(this.length);
            collected += flushed;
            totalCollected += flushed;
            totalCollectedWithPruning += flushed;
            totalPruned += pruned;
            flushInterval *= flushInterval / Math.max(1, flushed);
            if (flushInterval > maxFlushInterval) {
                flushInterval = maxFlushInterval;
            }
            if (debug) {
                if (request.ctx().logger().isLoggable(Level.INFO)) {
                    request.ctx().logger().info(
                        "Pruning: pruned=" + pruned
                        + ", flushed=" + flushed
                        + ", collected=" + collected
                        + ", flushInterval=" + flushInterval);
                }
            }
            pruned = 0;
        }
    }

    private void prune() throws IOException {
        if ((currentReader instanceof
            org.apache.lucene.index.PerFieldMergingIndexReader)
            && reverse)
        {
            passThrought();
            return;
        }
        if (request.ctx().logger().isLoggable(Level.INFO)) {
            request.ctx().logger().info(
                "Pruning on " + currentReader.getClass().getName()
                + ", totalCollected is: " + totalCollected
                + ", docIdsCount: " + docIdsCount);
        }
        if (totalCollected + docIdsCount <= limit) {
            //too little docs found no need to prune
            //just collect all
            if (request.ctx().logger().isLoggable(Level.INFO)) {
                request.ctx().logger().info(
                    "Too little docs found so far: totalCollected: "
                    + totalCollected + ", currentSegment: " + docIdsCount
                    + ", limit: " + limit
                    + ", skipping pruning.");
            }
            passThrought();
            return;
        }

        if (nextCollector.canPopulateFromCache(currentReader)) {
            if (request.ctx().logger().isLoggable(Level.INFO)) {
                request.ctx().logger().info(
                    "SortedCollector can populate docs from cache, skipping "
                    + "pruning.");
            }
            passThrought();
            return;
        }

        final Terms terms = currentReader.terms(pruningField);
        if (terms == null) {
            if (request.ctx().logger().isLoggable(Level.SEVERE)) {
                request.ctx().logger().severe(
                    "Can't prune on field <" + pruningField
                    + ">, no terms available");
            }
            passThrought();
            return;
        }
        if (prefix == null && prefixed) {
            if (request.ctx().logger().isLoggable(Level.SEVERE)) {
                request.ctx().logger().severe(
                    "Can't prune prefixed field <" + pruningField
                    + "> with empty prefix");
            }
            passThrought();
            return;
        }
        final Terms groupTerms;
        CacheInput groupFieldCache = null;
        if (groupField == null) {
            groupTerms = null;
        } else {
            groupTerms = currentReader.terms(pruningGroupField);
            if (groupTerms == null) {
                if (request.ctx().logger().isLoggable(Level.SEVERE)) {
                    request.ctx().logger().severe(
                        "Can't group with pruning on field <"
                        + pruningGroupField + ">, no terms available");
                }
                passThrought();
                return;
            }
            if (fieldsCache != null) {
                groupFieldCache = fieldsCache.getCacheFor(
                    currentReader,
                    groupField);
            }
        }
        if (prefix == null && groupPrefixed) {
            if (request.ctx().logger().isLoggable(Level.SEVERE)) {
                request.ctx().logger().severe(
                    "Can't group with pruning on prefixed field <"
                    + pruningGroupField + "> with empty prefix");
            }
            passThrought();
            return;
        }
        final TermsEnum te;
        if (reverse) {
            try {
                te = terms.reverseIterator();
            } catch (UnsupportedOperationException ign) {
                if (request.ctx().logger().isLoggable(Level.SEVERE)) {
                    request.ctx().logger().severe("Can't prune on field <"
                        + pruningField
                        + ">, reverse terms enum is not supported");
                }
                passThrought();
                return;
            }
        } else {
            te = terms.iterator(true);
        }
        if (userIdField != null) {
            Terms userIdTerms = currentReader.terms(userIdField);
            if (userIdTerms != null) {
                int freq = userIdTerms.docFreq(userIdTerm);
                if (docIdsCount * 100 < freq) {
                    if (request.ctx().logger().isLoggable(Level.INFO)) {
                        request.ctx().logger().info(
                            "Too many user docs in current segment: " + freq
                            + ", docs found: " + docIdsCount
                            + ", skipping pruning.");
                    }
                    passThrought();
                    return;
                }
            }
        }
        final BytesRef prefixFilter;
        if (prefixed) {
            prefixFilter = new BytesRef(prefix.toString() + '#');
            BytesRef seekPrefix;
            if (reverse) {
                seekPrefix = new BytesRef(prefixFilter);
                seekPrefix.append(REVERSE_SEEK_SUFFIX);
            } else {
                seekPrefix = prefixFilter;
            }
            te.seek(seekPrefix);
        } else {
            prefixFilter = null;
            te.next();
        }
        DocsEnum de = null;
        final int maxDoc = currentReader.maxDoc();
        pruned = 0;
        collected = 0;
        flushInterval =
            Math.max(MIN_FLUSH_INTERVAL, Math.min(maxFlushInterval, limit));
        if (debug) {
            if (request.ctx().logger().isLoggable(Level.FINEST)) {
                request.ctx().logger().finest("Pruning: maxFlushInterval="
                    + maxFlushInterval
                    + ", limit=" + limit);
            }
        }
        DocsEnum groupDocsEnum = null;
        BytesRef groupTerm;
        TermsEnum groupTermsEnum;
        boolean groupFast = request.mergeFunc() == null;
        boolean groupCount =
            !groupFast
            && ((request.mergeFunc() instanceof MergeFuncFactory.MergeCountSlow)
            || (request.mergeFunc() instanceof MergeFuncFactory.MergeCountFast));
        if (groupField == null) {
            groupTerm = null;
            groupTermsEnum = null;
        } else {
            groupTerm = new BytesRef();
            groupTermsEnum = groupTerms.getThreadTermsEnum(true);
        }
        try {
            for (
                BytesRef term = te.term();
                term != null && collected < limit;
                term = te.next())
            {
                checkAbort();
                if (debug) {
                    if (request.ctx().logger().isLoggable(Level.FINEST)) {
                        request.ctx().logger().finest("term: "
                            + term.utf8ToString());
                    }
                }
                if (prefixFilter != null && !term.startsWith(prefixFilter)) {
                    break;
                }
                if (groupFast) {
                    if (maxTerm == null) {
                        maxTerm = new BytesRef(term);
                    }
//                    System.err.println("Pruning: " + term.utf8ToString());
//                    System.err.println("Max: " + maxTerm.utf8ToString());
//                    System.err.println("collected: " + totalCollected);
//                    System.err.println("collectedWithPruning: "
//                        + totalCollectedWithPruning);
                    int maxTermCmp = termComp.compare(term, maxTerm);
                    if (totalCollectedWithPruning < length() + offset()) {
                        if (maxTermCmp > 0) {
                            maxTerm.copy(term);
                        }
                    } else {
                        if (maxTermCmp > 0 || (maxTermCmp == 0 && groupField == null)) {
                            if (groupField != null) {
                                totalPruned += docIdsCount;
                            }
//                            String lastHitTerm = ((SortedSet<YaDoc3>) nextCollector.hits())
//                                .last().getString(sortField);
                            if (debug) {
                                request.ctx().logger().info("break maxterm: "
                                    + maxTerm.utf8ToString()
                                    + " < " + term.utf8ToString());
//                                    + " lasthit:" + lastHitTerm
//                                    + " size: " + totalCollectedWithPruning);
                            }
                            break;
                        }
                    }
                }
                de = te.docs(null, de);
                for (
                    int docId = de.nextDoc();
                    docId != DocIdSetIterator.NO_MORE_DOCS;
                    docId = de.nextDoc())
                {
                    if (docId > maxDoc) {
                        throw new IOException("out of order docId: "
                            + docId + ", " + currentReader.maxDoc());
                    }
                    if (docIdsSet.get(docId)) {
                        String group = null;
                        if (groupField != null) {
                            group = readGroupField(docId, groupFieldCache);
                            if (group != null) {
                                if (analyzer != null) {
                                    tokenizeAndFilterGroup(group, groupTerm);
                                } else {
                                    if (groupPrefixed) {
                                        groupTerm.copy(
                                            StringUtils.concat(
                                                prefix.toString(),
                                                '#',
                                                group));
                                    } else {
                                        groupTerm.copy(group);
                                    }
                                }
                            }
                        }
                        doCollect(docId);
                        if (group != null) {
                            pruneGrouping(
                                docId,
                                groupDocsEnum,
                                groupTerm,
                                groupTermsEnum,
                                groupFast,
                                groupCount);
                        }
//                            forceFlush();
                    }
                }
//                forceFlush();
            }
            forceFlush();
        } finally {
            if (groupTermsEnum != null) {
                groupTermsEnum.close();
            }
        }
    }

    private String readGroupField(
        final int docId,
        final CacheInput groupFieldCache)
        throws IOException
    {
        if (groupFieldCache != null) {
            YaField cachedField;
            if (groupFieldCache.seek(docId)) {
                cachedField = groupFieldCache.field();
                if (cachedField != null) {
                    docsFromCache++;
                    totalDocsFromCache++;
                    return cachedField.toString();
                }
            }
        }
        docsRead++;
        totalDocsRead++;
        Document doc = currentReader.document(
            docId,
            groupFieldSelector);
        Field field = doc.getField(groupField);
        if (field != null) {
            return field.stringValue();
        } else {
            return null;
        }
    }

    private void tokenizeAndFilterGroup(
        final String group,
        final BytesRef groupTerm)
        throws IOException
    {
        if (stringReader == null) {
            stringReader = new ReusableStringReader();
        }
        stringReader.init(group);
        try (TokenStream source =
            analyzer.reusableTokenStream(
                pruningGroupField,
                stringReader))
        {
            source.reset();
            TermToBytesRefAttribute termAtt =
                source.getAttribute(TermToBytesRefAttribute.class);
            if (source.incrementToken()) {
                termAtt.toBytesRef(groupTerm);
            } else {
                if (groupPrefixed) {
                    groupTerm.copy(
                        StringUtils.concat(
                            prefix.toString(),
                            '#',
                            group));
                } else {
                    groupTerm.copy(group);
                }
            }
        }
    }

    private void pruneGrouping(
        final int docId,
        DocsEnum groupDocsEnum,
        final BytesRef groupTerm,
        final TermsEnum groupTermsEnum,
        final boolean groupFast,
        final boolean groupCount)
        throws IOException
    {
        docIdsSet.fastClear(docId);
        int firstDocId;
        if (reverse) {
            firstDocId = 0;
        } else {
            firstDocId = docId + 1;
        }
        checkAbort();
        if (!groupTermsEnum.seekExact(groupTerm, true)) {
            request.ctx().logger().info("groupTerms.seek failed for: "
                + groupTerm.utf8ToString());
            return;
        }
        boolean lockedDocsEnum = false;
//        int groupCnt = 0;
        try {
            groupDocsEnum =
                groupTermsEnum.docs(null, groupDocsEnum);
                    if (groupDocsEnum instanceof Lockable) {
                        lockedDocsEnum =
                        ((Lockable) groupDocsEnum).lockBuffer();
                    }
                    // Find next doc id set
                    int nextDocId =
                        docIdsSet.nextSetBit(firstDocId);
                    while (nextDocId != -1) {
//                        groupCnt++;
                        // Advance to next document in kishka
                        int groupDocId =
                            groupDocsEnum.advance(nextDocId);
                        if (groupDocId
                            == DocIdSetIterator.NO_MORE_DOCS)
                        {
                        // No more docs left in kishka, break now
                            nextDocId = -1;
                        } else {
                            if (docIdsSet.get(groupDocId)) {
                                // Next document in kishka presents
                                // in
                                // doc id set, let's collect it
                                if (groupFast) {
                                    totalPruned++;
                                    //just skip
                                } else if (groupCount) {
                                    //collect same (prevent unnecessary
                                    //doc reading from disk
                                    doCollect(docId);
                                } else {
                                    doCollect(groupDocId);
                                }
                                docIdsSet.fastClear(groupDocId);
                            }
                            nextDocId = docIdsSet.nextSetBit(groupDocId + 1);
                        }
                    }
            } finally {
                if (lockedDocsEnum) {
                    ((Lockable) groupDocsEnum).releaseBuffer();
                }
            }
//        System.err.println("GroupCount: " + groupCnt);
    }

    final void checkAbort() throws IOException {
        request.ctx().checkAbort();
        if (consumer != null) {
            consumer.checkAlive();
        }
    }

    public void setScorer(Scorer scorer) throws IOException {
        nextCollector.setScorer(scorer);
    }

    @Override
    public void preprocess(final int docId) throws IOException {
    }

    @Override
    public void collectPreprocessed(final int docId) throws IOException {
    }

    @Override
    public void collect(int docId) throws IOException {
        nextCollector.preprocess(docId);
        docIdsSet.set(docId);
        checkAbort();
        if (maxDocId < docId) {
            maxDocId = docId;
        }
        docIdsCount++;
    }

    public int countedFlush() throws IOException {
        int oldTotalCollected = totalCollected;
        if (currentReader != null && docIdsCount > 0) {
            doPrune();
        }
        return totalCollected - oldTotalCollected;
    }

    public void flush() throws IOException {
        countedFlush();
    }

    public void close() throws IOException {
        flush();
        nextCollector.close();
        if (consumer != null) {
            consumer.uniqHitsCount(uniqCount());
            consumer.totalHitsCount(getTotalCount());
            consumer.startHits();
            int pos = 0;
            int added = 0;
            final int length = length();
            final int offset = offset();
            final Set<String> getFields = getGetFields();
            final Set<? extends YaDoc3> hits = nextCollector.hits();
            for (YaDoc3 doc : hits) {
                if (added >= length) {
                    break;
                }
                if (pos >= offset) {
                    consumer.document(doc, getFields);
                    ++added;
                }
                pos++;
            }
            consumer.endHits();
        }
    }

    public Set<? extends YaDoc3> hits() throws IOException {
        return nextCollector.hits();
    }

    public int uniqCount() {
        if (totalPruned == totalCount) {
            return totalCollected;
        }
        double collectPruneRatio = totalPruned / Math.max(totalCollected, 1d);
        double averageUniqCount = totalCount / Math.max(collectPruneRatio, 1d);
        if (request.ctx().logger().isLoggable(Level.INFO)) {
            request.ctx().logger().info(
                "totalPruned: " + totalPruned
                + ", totalCollected: " + totalCollected
                + ", totalCount: " + totalCount
                + ", collectPruneRatio: " + collectPruneRatio
                + ", averageUniqCount: " + averageUniqCount
                + ", docsRead: " + totalDocsRead
                + ", docsFromCache: " + totalDocsFromCache);
        }
        return (int) averageUniqCount;
    }

    public int getTotalCount() {
	return totalCount;
    }

    public void setIgnoreEmpty(boolean i) {
	nextCollector.setIgnoreEmpty(i);
    }

    final public Set<String> getGetFields() {
        return nextCollector.getGetFields();
    }

    final public RequestContext context() {
        return request.ctx();
    }

    final public int offset() {
        return request.offset();
    }

    final public int length() {
        return length;
    }

    final public void length(int length) {
        this.length = length;
    }
}
