package ru.yandex.msearch.collector;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;

import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.util.StringHelper;

import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.RequestContext;
import ru.yandex.msearch.SearchRequest;
import ru.yandex.msearch.SearchResultsConsumer;
import ru.yandex.msearch.collector.aggregate.Aggregator;
import ru.yandex.msearch.collector.docprocessor.DocProcessor;
import ru.yandex.msearch.collector.docprocessor.ModuleFieldsAggregator;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.outergroup.OuterGroupFunction;
import ru.yandex.msearch.collector.postfilter.PostFilter;
import ru.yandex.msearch.collector.sort.NullSortFunc;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.fieldscache.CacheInput;
import ru.yandex.msearch.fieldscache.FieldsCache;
import ru.yandex.search.YandexScorer;
import ru.yandex.search.YandexScorerFactory;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.string.StringUtils;

public class SortedCollector extends ParametrizedDocCollector {
    public static final int MAX_GET_DOCS_SIZE = 100000;
    public static final String DEBUG_READER_FIELD =
        StringHelper.intern("#debug_reader");

    public SortedCollector(
        final SearchRequest request,
        final SearchResultsConsumer consumer)
    {
        this.consumer = consumer;
        this.offset = request.offset();
        this.length = request.length();
        this.memoryLimit = request.memoryLimit();
        totalSizeInBytes = 0;
        this.outerGroup = request.outerGroup();
        this.ctx = request.ctx();
        this.asc = request.sortDirection();
        this.debug = request.debug();

        this.mergeFunc = request.mergeFunc();

        this.getFields = request.getFields();
        this.aggregator = request.aggregator();
        this.docProcessor = request.docProcessor();
        this.postFilter = request.postFilter();

        this.groupFunc = request.group();
        this.sortFunc = request.sort();
        this.scorerFactory = request.scorerFactory();

        DatabaseConfig config = request.config();
        checkGetFields(request);

        orderByAggregateValue = false;
        if( aggregator != null && !(sortFunc instanceof NullSortFunc) )
        {
            HashSet<String> sortFields = new HashSet<String>( sortFunc.loadFields() );
            for( String field : aggregator.outFields() )
            {
                if( sortFields.contains(field) )
                {
                    orderByAggregateValue = true;
                    break;
                }
            }
        }

        Map<String, FieldSelectorResult> step1ReadMap =
            new IdentityHashMap<>();
        for (final String field: step1ReadFields) {
            step1ReadMap.put(field, FieldSelectorResult.LOAD);
        }
        for (final String field: step2ReadFields) {
            step1ReadMap.put(field, FieldSelectorResult.SIZE);
        }
        step1Fs = new MapFieldSelector(step1ReadMap);
        step2Fs = new MapFieldSelector(step2ReadFields);
        CollectingFieldToIndex fieldToIndex = request.fieldToIndex();
        for (String field: step1ReadFields) {
            fieldToIndex.indexFor(field);
        }
        for (String field: step2ReadFields) {
            fieldToIndex.indexFor(field);
        }
        for (String field: scorerFactory.outFields()) {
            fieldToIndex.indexFor(field);
        }

        if (debug) {
            fieldToIndex.indexFor(DEBUG_READER_FIELD);
        }
        docBase = 0;
        getDocs = new ArrayList<>();
        this.fieldToIndex = fieldToIndex.compactFieldToIndex();
        step1LoadFields = new FieldConfig[this.fieldToIndex.fieldsCount()];
        for (String field: step1ReadFields) {
            step1LoadFields[this.fieldToIndex.indexFor(field)] =
                config.fieldConfigFast(field);
        }
        step2LoadFields = new FieldConfig[this.fieldToIndex.fieldsCount()];
        for (String field: step2ReadFields) {
            step1LoadFields[this.fieldToIndex.indexFor(field)] =
                new FieldConfig(field);
            step2LoadFields[this.fieldToIndex.indexFor(field)] =
                config.fieldConfigFast(field);
        }

        step1Visitor =
            new YaDocFieldVisitor(step1Fs, this.fieldToIndex, step1LoadFields);
        step2Visitor =
            new YaDocFieldVisitor(step2Fs, this.fieldToIndex, step2LoadFields);

        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine(
                "New SortedCollector: getFields:" + getFields
                + " / step1ReadFields: " + step1ReadFields
                + " / step2ReadFields: " + step2ReadFields
                + " / sortFunc: " + sortFunc
                + " / groupFunc: " + groupFunc
                + " merge_func: " + this.mergeFunc
                + ", aggregator: " + aggregator
                + ", orderByAggregateValue: " + orderByAggregateValue
                + ", scorerFactory: " + scorerFactory
                + ", fieldToIndex: " + fieldToIndex
                + (memoryLimit != Long.MAX_VALUE
                    ? (", memoryLimit: " + memoryLimit)
                : ""));
        }

        currentComparator = asc ? Comparator.reverseOrder() : Comparator.naturalOrder();
        sorter = new TreeMap<>(currentComparator);

        deduplicator = new HashMap<>(1000, 0.5f);

        ignoreEmpty = false;
        collectStartTime = System.currentTimeMillis();
    }

    @Override
    public GroupFunc groupFunc() {
        return groupFunc;
    }

    @Override
    public void setPrefix(final Prefix prefix) {
    }

    @Override
    public SortFunc sortFunc() {
        return sortFunc;
    }

    @Override
    public FieldToIndex fieldToIndex() {
        return fieldToIndex;
    }

    @Override
    public Set<String> mergeFields() {
        return getFields;
    }

    @Override
    public MergeFunc mergeFunc() {
        return mergeFunc;
    }

    @Override
    public void setDebug(boolean debug) {
        this.debug = debug;
    }

    protected Set<String> fieldsFromFieldsFunction(FieldsFunction function) {
        return function.loadFields();
    }

    protected Set<String> fieldsFromFieldsAggregator(ModuleFieldsAggregator aggregator) {
        return aggregator.loadFields();
    }

    protected void checkGetFields(final SearchRequest request) {
        DatabaseConfig config = request.config();
        step1ReadFields = new HashSet<>();
        step2ReadFields = new HashSet<>(getFields.size() << 1);
        for (String field: getFields) {
            step2ReadFields.add(StringHelper.intern(field));
        }
        for (FieldsFunction function: new FieldsFunction[] {
                groupFunc,
                sortFunc,
                aggregator,
                outerGroup,
                postFilter
            })
        {
            if (function != null) {
                for (String field: fieldsFromFieldsFunction(function)) {
                    FieldConfig fieldConfig = config.fieldConfigFast(field);
                    if (fieldConfig != null && fieldConfig.store()) {
                        step1ReadFields.add(field);
                    }
                    step2ReadFields.remove(field);
                }
            }
        }

        if (docProcessor != null) {
            ModuleFieldsAggregator fieldsAggregator
                = new ModuleFieldsAggregator();

            docProcessor.apply(fieldsAggregator);
            for (String field: fieldsFromFieldsAggregator(fieldsAggregator)) {
                FieldConfig fieldConfig = config.fieldConfigFast(field);
                if (fieldConfig != null && fieldConfig.store()) {
                    step1ReadFields.add(field);
                }
                step2ReadFields.remove(field);
            }
            // we do not read docs that generated from doc processor

            if (!request.dpReadOutFields()) {
                step2ReadFields.removeAll(fieldsAggregator.outFields());
            }
        }
    }

    public void setFieldsCache( FieldsCache fieldsCache )
    {
	this.fieldsCache = fieldsCache;
    }

    public boolean acceptsDocsOutOfOrder()
    {
	return true;
    }

    public void setNextReader(AtomicReaderContext readerContext) throws IOException
    {
        this.context = readerContext;
        docsRead = 0;
        docsFromCache = 0;
        final long collectTime =
            (System.currentTimeMillis() - collectStartTime);
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("Collect time: " + collectTime);
        }
        totalCollectTime += collectTime;

        populateDocs( reader, getDocs, fieldsCache );

        this.reader = readerContext.reader;
        this.docBase = readerContext.docBase;
        collectStartTime = System.currentTimeMillis();
    }

    public void setScorer(final Scorer scorer) {
        this.scorer = scorerFactory.scorer(context, scorer);
    }

    public void collectPreprocessed(final int docId) throws IOException {
//        System.err.println("CollectPreprocessed: " + docId);
        //pruning same docId costyl'
        ++totalCount;
        if (getDocs.size() > 0) {
            Collectable last = getDocs.get(getDocs.size() - 1);
            if (last.docId() == docId) {
                getDocs.add(last);
                return;
            }
        }
        final YaDoc3Delayed doc = new YaDoc3Delayed(this, docId, reader);
        scorer.process(docId, doc);
        getDocs.add(new Collectable(docId, doc));
        if (getDocs.size() >= MAX_GET_DOCS_SIZE) {
            ctx.logger().fine("Forced getDocs flush");
            setNextReader(context);
        }
    }

    public void preprocess(final int docId) throws IOException {
        scorer.preprocess(docId);
    }

    public void collect(int docId) throws IOException {
        checkAbort();
        preprocess(docId);
        collectPreprocessed(docId);
    }

    public boolean canPopulateFromCache(final IndexReader reader) {
        if (fieldsCache != null) {
            return fieldsCache.getCachesFor(reader, step1ReadFields) != null;
        }
        return false;
    }

    protected int populateDocs(
        final IndexReader reader,
        final List<Collectable> getDocs,
        final FieldsCache fieldsCache)
        throws IOException
    {
        long start = System.currentTimeMillis();
        int populated = 0;
        List<CacheInput> caches = null;
        if (getDocs.size() > 0) {
            boolean useCache = false;
            if (fieldsCache != null) {
                caches = fieldsCache.getCachesFor(reader, step1ReadFields);
                if (caches != null) {
                    useCache = true;
                }
            }


            boolean filledFromCache;

            if (!useCache) {
                ctx.logger().fine("useCache = false");
                long sortStart = System.currentTimeMillis();
                Collections.sort(
                    getDocs,
                    (a, b) -> Integer.compare(a.docId(), b.docId()));
                if (ctx.logger().isLoggable(Level.FINE)) {
                    ctx.logger().fine(
                        "SortingCollector: populateDocs: "
                        + "Collections.sort(getDocs) took "
                        + (System.currentTimeMillis() - sortStart) + " ms");
                }
            }

            for (int i = 0; i < getDocs.size(); i++) {
                final Collectable collected = getDocs.get(i);
                final YaDoc3Delayed yadoc = collected.doc();
                if (!collected.processed) {
                    int docId = collected.docId();
                    checkAbort();
                    filledFromCache = false;

                    checkAbort();

                    if (useCache) {
                        boolean success = true;
                        for (CacheInput oneFieldCache: caches) {
                            YaField cachedField;
                            if (oneFieldCache.seek(docId)) {
                                cachedField = oneFieldCache.field();
                            } else {
    //                            System.err.println("seek Failed");
                                success = false;
                                break;
                            }
                            yadoc.setField(
                                oneFieldCache.fieldname(),
                                cachedField);
                        }
                        if (success) {
                            filledFromCache = success;
                            docsFromCache++;
                            totalDocsFromCache++;
    //                        System.err.println("got from cache");
                        }
                    }
    
                    if (!filledFromCache) {
    //                    System.err.println("load from disk");
                        step1Visitor.doc(yadoc);
                        reader.readDocument(docId, step1Visitor);
                        docsRead++;
                        totalDocsRead++;
    //                    Document doc = reader.document(docId, step1Fs);
    //                    yadoc.readFields(doc, step1LoadFields);
                    }
                }

//                System.err.println("Populate: " + docId + ", " + yadoc);

                populated += addToSorter(yadoc);
                collected.processed = true;
                //Allow GC to collect heavyweight yadocs evicted by sorted
                getDocs.set(i, null);
            }
        }
        long populateTime = System.currentTimeMillis() - start;
        if (ctx.logger().isLoggable(Level.FINE)) {
            ctx.logger().fine("Populate time: " + populateTime
                + ", docsRead: " + docsRead
                + ", docsFromCache: " + docsFromCache);
        }
        totalPopulateTime += populateTime;
        getDocs.clear();
        return populated;
    }

    public void step2LoadFields(Collection<? extends YaDoc3> delayed)
        throws IOException
    {
        if (step2ReadFields.size() == 0) {
            ctx.logger().fine("Skipping step2LoadFields");
            return;
        } else {
            if (ctx.logger().isLoggable(Level.FINE)) {
                ctx.logger().fine("Step2readfields: " + step2ReadFields);
            }
        }
        List<YaDoc3Delayed> sorted = new ArrayList<>(delayed.size());
        for (YaDoc3 d: delayed) {
            final YaDoc3Delayed yadoc = (YaDoc3Delayed) d;
            sorted.add(yadoc);
            final Collection<YaDoc3> merged = yadoc.merged();
            if (merged != null) {
                for (YaDoc3 m: merged) {
                    final YaDoc3Delayed mergedDoc = (YaDoc3Delayed) m;
                    sorted.add(mergedDoc);
                }
            }
        }
        sorted.sort(
            (a, b) -> {
                int cmp =
                    Integer.compare(a.reader.hashCode(), b.reader.hashCode());
                if (cmp == 0) {
                    cmp = Integer.compare(a.docId, b.docId);
                }
                return cmp;
            });
        IndexReader currentReader = null;
        List<CacheInput> caches = null;
        for (YaDoc3Delayed yadoc: sorted) {
            if (fieldsCache != null
                && (currentReader == null || currentReader != yadoc.reader))
            {
                currentReader = yadoc.reader;
                caches = fieldsCache.getCachesFor(
                    yadoc.reader,
                    step2ReadFields);
            }
            boolean filledFromCache = true;
            if (caches != null) {
                for (CacheInput oneFieldCache: caches) {
                    YaField cachedField;
                    if (oneFieldCache.seek(yadoc.docId)) {
                        cachedField = oneFieldCache.field();
                    } else {
                        filledFromCache = false;
                        break;
                    }
                    yadoc.setField(
                        oneFieldCache.fieldname(),
                        cachedField);
                }
            } else {
                filledFromCache = false;
            }

            if (!filledFromCache) {
                step2Visitor.doc(yadoc);
                yadoc.reader.readDocument(yadoc.docId, step2Visitor);
            }
        }
    }

    public int countedFlush() throws IOException {
        return populateDocs(reader, getDocs, fieldsCache);
    }

    public void flush() throws IOException {
        step2LoadFields(sorter.keySet());
        countedFlush();
    }

    public void close() throws IOException {
        flush();
        if (consumer != null) {
            consumer.uniqHitsCount(uniqCount());
            consumer.totalHitsCount(getTotalCount());
            consumer.startHits();
            int pos = 0;
            int added = 0;
            final Set<String> getFields = getGetFields();
            for (YaDoc3 doc : sorter.keySet()) {
                if (added >= length) {
                    break;
                }
                if (pos >= offset) {
                    consumer.document(doc, getFields);
                    ++added;
                }
                pos++;
            }
            consumer.endHits();
        }
        if (ctx.logger().isLoggable(Level.CONFIG)) {
            ctx.logger().config("Total collect time: " + totalCollectTime);
            ctx.logger().config("Total populate time: " + totalPopulateTime
                + ", docsReadTotal: " + totalDocsRead
                + ", docsFromCacheTotal: " + totalDocsFromCache);
            if (docProcessor != null) {
                docProcessor.after();
            }
        }
    }

    public Set<? extends YaDoc3> hits() throws IOException {
        flush();
        return sorter.keySet();
    }

    public int uniqCount() {
        return uniqCount;
    }

    public int getTotalCount() {
	return totalCount;
    }

    public void clear()
    {
	sorter.clear();
        totalCount = 0;
        uniqCount = 0;
    }

    public void setIgnoreEmpty( boolean i )
    {
	ignoreEmpty = i;
    }

    final long sizeInBytes(final YaDoc3 doc) {
        long size = 0;
        if (doc == null) {
            return size;
        }
        for (YaField field: doc.fields()) {
            if (field != null) {
                size += field.sizeInBytes();
            }
        }
        final Collection<YaDoc3> merged = doc.merged();
        if (merged != null) {
            for (YaDoc3 m: merged) {
                size += sizeInBytes(m);
            }
        }
        return size;
    }

    protected void doAddToSorter(
        final DocEntry docEntry,
        final long sizeInBytes)
    {
        if (sorter.size() - offset >= length) {
            if (length != 0
                && sorter.comparator().compare(
                    sorter.lastKey(),
                    docEntry.doc) > 0)
            {
                DocEntry oldEntry = sorter.pollLastEntry().getValue();
                if (memoryLimit != Long.MAX_VALUE) {
                    // otherwise size in bytes will be 0
                    totalSizeInBytes += sizeInBytes;
                    totalSizeInBytes -= sizeInBytes(oldEntry.doc);
                }
                oldEntry.doc = null; //this will cause deduplicator's hashmap value can be GS'ed.
                sorter.put(docEntry.doc, docEntry);
            }
        } else {
            totalSizeInBytes += sizeInBytes;
            sorter.put(docEntry.doc, docEntry);
        }
        while (totalSizeInBytes > memoryLimit && sorter.size() > offset + 1) {
            DocEntry oldEntry = sorter.pollLastEntry().getValue();
            totalSizeInBytes -= sizeInBytes(oldEntry.doc);
            oldEntry.doc = null;
        }
    }

    protected void removeFromSorter(final YaDoc3Delayed doc) {
        sorter.remove(doc);
    }

    protected boolean duplicated(
        final YaDoc3Delayed doc)
    {
        return outerGroup.modifyAndCheckDuplicated( doc );
    }

    protected int addToSorter(
            final YaDoc3Delayed doc)
        throws IOException
    {
        int added = 0;
        if (debug) {
            doc.setField(
                DEBUG_READER_FIELD,
                new YaField.StringYaField(
                    StringUtils.getUtf8Bytes(this.reader.toString())));
        }
        if (!docProcessor.processWithFilter(doc) || !postFilter.test(doc)) {
            --totalCount;
            return 0;
        }
        if (outerGroup != null) {
            if (duplicated(doc)) {
                if (!ignoreEmpty) {
                    return added;
                }
            }
        }
        doc.initSort();
        YaField group = doc.getGroupField();

        final long sizeInBytes;
        if (memoryLimit == Long.MAX_VALUE) {
            sizeInBytes = 0;
        } else {
            sizeInBytes = sizeInBytes(doc);
        }

        if (group != null) {
            DocEntry newDocEntry = new DocEntry(doc);
            DocEntry oldDocEntry = deduplicator.putIfAbsent(group, newDocEntry);
            if (oldDocEntry == null) {
                if (aggregator != null) {
                    newDocEntry.aggregator = aggregator.clone();
                    newDocEntry.aggregator.pushFields(newDocEntry.doc);
                    if (orderByAggregateValue) {
                        doc.initSort();
                    }
                    newDocEntry.aggregator.updateValues(newDocEntry.doc);
                }
                doAddToSorter(newDocEntry, sizeInBytes);
                ++uniqCount;
                ++added;
            } else {
                if (mergeFunc != null) {
                    if (oldDocEntry.doc == null) {
                        //This docEntry was already removed from sorter.
                        //Must readd
                        oldDocEntry.doc = doc;
                        if (aggregator != null) {
                            oldDocEntry.aggregator.pushFields(doc);
                            if (orderByAggregateValue) {
                                doc.initSort();
                            }
                            oldDocEntry.aggregator.updateValues(doc);
                        }
                        doAddToSorter(oldDocEntry, sizeInBytes);
                    } else {
                        //Must compare the new doc with the doc already in
                        //the sorter to see if the former must be reinseted
                        //for a proper sorting.
                        if (orderByAggregateValue) {
                            //must reinsert as aggregate value may
                            //have been changed
                            removeFromSorter(oldDocEntry.doc);
                            oldDocEntry.aggregator.updateValues(doc);
                            doAddToSorter(oldDocEntry, sizeInBytes);
                            oldDocEntry.doc.merge(doc);
                        } else {
                            int cmp =
                                currentComparator.compare(
                                    doc,
                                    oldDocEntry.doc);
                            if (cmp < 0) {
                                //must reinsert
                                removeFromSorter(oldDocEntry.doc);
                                MergeFunc mf = oldDocEntry.doc.mergeFunc();

                                if (aggregator != null) {
                                    oldDocEntry.aggregator.updateValues(
                                        doc);
                                }

                                if (mf == null) {
                                //just insert new doc to sorter and forgot
                                //about the old one
                                    oldDocEntry.doc = doc;
                                    doAddToSorter(newDocEntry, sizeInBytes);
                                } else {
                                    oldDocEntry.doc.removeMergeFunc();
                                    mf.merge(oldDocEntry.doc);
                                    doc.setMergeFunc(mf);
                                    oldDocEntry.doc = doc;
                                    if (aggregator != null) {
                                        oldDocEntry.aggregator.pushFields(
                                            doc);
                                    }
                                    doAddToSorter(oldDocEntry, sizeInBytes);
                                }
                            } else {
                            //just add to merged list
                                oldDocEntry.doc.merge(doc);
                                if (aggregator != null) {
                                    oldDocEntry.aggregator.updateValues(
                                        doc);
                                }
                            }
                        }
                    }
                }
            }
        } else {
            ++uniqCount;
            ++added;
            doAddToSorter(new DocEntry(doc), sizeInBytes);
        }
        return added;
    }

    final void checkAbort() throws IOException {
        ctx.checkAbort();
        if (consumer != null) {
            consumer.checkAlive();
        }
    }

    final public Set<String> getGetFields() {
        Set<String> getFields;
        if (aggregator == null) {
            getFields = this.getFields;
        } else {
            getFields = new LinkedHashSet<>(this.getFields);
            getFields.addAll(aggregator.outFields());
        }
        return getFields;
    }

    final public RequestContext context() {
        return ctx;
    }

    final public int offset() {
        return offset;
    }

    final public int length() {
        return length;
    }

    final public void length(final int length) {
        this.length = length;
    }

    protected final SearchResultsConsumer consumer;
    protected final MergeFunc mergeFunc;
    protected FieldSelector step1Fs;
    protected FieldSelector step2Fs;
    protected final YaDocFieldVisitor step1Visitor;
    protected final YaDocFieldVisitor step2Visitor;
    protected TreeMap<YaDoc3Delayed, DocEntry> sorter;
    protected int totalCount = 0;
    protected int uniqCount = 0;

    protected final List<Collectable> getDocs;

    protected boolean asc; // turns on asceding sort order
    protected int offset;
    protected int length;
    protected final long memoryLimit;
    protected long totalSizeInBytes;
    protected final Set<String> getFields;
    protected Set<String> step1ReadFields;
    protected Set<String> step2ReadFields;
    protected final FieldConfig[] step1LoadFields;
    protected final FieldConfig[] step2LoadFields;
    protected IndexReader reader;
    protected int docBase;
    protected boolean ignoreEmpty;
    protected Map<YaField, DocEntry> deduplicator;
    protected FieldsCache fieldsCache;
    protected OuterGroupFunction outerGroup;
    protected RequestContext ctx;
    protected long prevCheckAbort;
    protected boolean xUrls;
    protected Comparator<YaDoc3Delayed> currentComparator;
    protected final GroupFunc groupFunc;
    protected final SortFunc sortFunc;
    protected Aggregator aggregator;
    protected DocProcessor docProcessor;
    protected final PostFilter postFilter;
    protected boolean orderByAggregateValue;
    protected final FieldToIndex fieldToIndex;
    protected boolean debug = false;
    protected YandexScorer scorer = null;
    protected long collectStartTime;
    protected long totalCollectTime = 0L;
    protected long totalPopulateTime = 0L;
    protected int docsRead = 0;
    protected int totalDocsRead = 0;
    protected int docsFromCache = 0;
    protected int totalDocsFromCache = 0;
    protected IndexSearcher searcher = null;
    protected Query query = null;
    protected final YandexScorerFactory scorerFactory;
    protected AtomicReaderContext context;

    protected static final class Collectable {
        protected final int docId;
        protected final YaDoc3Delayed doc;
        protected boolean processed = false;

        public Collectable(final int docId, final YaDoc3Delayed doc) {
            this.docId = docId;
            this.doc = doc;
        }

        public int docId() {
            return docId;
        }

        public YaDoc3Delayed doc() {
            return doc;
        }
    }

    public static class YaDoc3Delayed extends YaDoc3 {
        public final int docId;
        public final IndexReader reader;

        YaDoc3Delayed(
            final SearchParams params,
            final int docId,
            final IndexReader reader)
        {
            super(params);
            this.docId = docId;
            this.reader = reader;
        }
    }

    protected static final class DocEntry {
        public DocEntry (final YaDoc3Delayed doc) {
            this.doc = doc;
        }

        public YaDoc3Delayed doc;
        public Aggregator aggregator;
    }
}
