package ru.yandex.msearch.collector;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Set;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.util.BytesRef;

import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.util.string.StringUtils;

public class YaDoc3 implements Comparable<YaDoc3> {
    public static final Comparator<YaField> COMPARATOR =
        Comparator.nullsFirst(Comparator.naturalOrder());

    private final SearchParams params;
    private final YaField[] fields;
    private YaField groupField = null;
    private YaField orderField = null;
    private YaField[] mergeFieldsArray = null;
    private int hashCode = 0;
    private MergeFunc mergeFuncClone = null;

    public YaDoc3(final SearchParams params) {
        this.params = params;
        fields = new YaField[params.fieldToIndex().fieldsCount()];
    }

    public void clear() {
        Arrays.fill(fields, null);
        groupField = null;
        orderField = null;
        mergeFieldsArray = null;
        hashCode = 0;
        mergeFuncClone = null;
    }

    public void initSort() {
        if (params.groupFunc() != null) {
            groupField = params.groupFunc().group(this);
        }
        if (params.sortFunc() != null) {
            orderField = params.sortFunc().sortField(this);
        }
    }

    public YaField[] fields() {
        return fields;
    }

    public YaField getGroupField() {
        return groupField;
    }

    public String getString(final String field) {
        YaField yf = getField(field);
        if (yf == null) {
            return null;
        }
        return yf.toString();
    }

    public YaField getField(final String field) {
        return fields[params.fieldToIndex().indexFor(field)];
    }

    public YaField getField(final int index) {
        return fields[index];
    }

    public void setField(final int index, final YaField f) {
        fields[index] = f;
    }

    public void setField(final String field, final YaField f) {
        setField(params.fieldToIndex().indexFor(field), f);
    }

    private YaField[] mergeFieldsArray() {
        if (mergeFieldsArray == null) {
            mergeFieldsArray = new YaField[params.mergeFields().size()];
            int i = 0;
            for (String f : params.mergeFields()) {
                mergeFieldsArray[i++] = getField(f);
            }
        }
        return mergeFieldsArray;
    }

    @Override
    public int hashCode() {
        if (hashCode != 0) {
            return hashCode;
        }

        int code = 0;
        if (params.mergeFields() == null) {
            code = Arrays.hashCode(fields);
        } else {
            code = Arrays.hashCode(mergeFieldsArray());
            hashCode = code;
        }
        return code;
    }

    @Override
    public boolean equals(final Object other) {
        return other instanceof YaDoc3
            && Arrays.equals(
                mergeFieldsArray(),
                ((YaDoc3) other).mergeFieldsArray());
    }

    public void merge(final YaDoc3 otherDoc) {
        if (mergeFunc() != null) {
            mergeFuncClone.merge(otherDoc);
        }
    }

    public Collection<YaDoc3> merged() {
        if (mergeFuncClone == null) {
            return null;
        }
        return mergeFuncClone.merged();
    }

    public MergeFunc mergeFunc() {
        if (mergeFuncClone == null && params.mergeFunc() != null) {
            mergeFuncClone = params.mergeFunc().clone();
        }
        return mergeFuncClone;
    }

    public void removeMergeFunc() {
        mergeFuncClone = null;
    }

    public void setMergeFunc(final MergeFunc mf) {
        mergeFuncClone = mf;
    }

    @Override
    public int compareTo(final YaDoc3 other) {
        int cmp = COMPARATOR.compare(other.orderField, orderField);
        if (cmp != 0) {
            return cmp;
        }

        cmp = COMPARATOR.compare(groupField, other.groupField);
        if (cmp == 0 && groupField == null) {
            return 1;
        } else {
            return cmp;
        }
    }

    public void writeJson(final JsonWriterBase jw, final Set<String> getFields)
        throws IOException
    {
        writeJson(jw, getFields, false);
    }

    public void writeJson(
        final JsonWriterBase jw,
        final Set<String> getFields,
        final boolean skipNulls)
        throws IOException
    {
        jw.startObject();
        for (String field: getFields) {
            YaField yaField = getField(field);
            if (yaField != null || !skipNulls) {
                jw.key(field);
                jw.value(yaField);
            }
        }
        if (mergeFuncClone != null) {
            mergeFuncClone.formJson(jw, getFields, skipNulls);
        }
        jw.endObject();
    }

    public void writeUtf8Json(
        final Utf8JsonWriter jw,
        final Set<String> getFields,
        final boolean skipNulls)
        throws IOException
    {
        jw.startObject();
        for (String field: getFields) {
            YaField yaField = getField(field);
            if (yaField != null || !skipNulls) {
                jw.key(field);
                jw.value((Utf8JsonValue)yaField);
            }
        }
        if (mergeFuncClone != null) {
            mergeFuncClone.formJson(jw, getFields, skipNulls);
        }
        jw.endObject();
    }

    @Override
    public String toString() {
        return "YaDoc3(" + Arrays.toString(fields)
            + '/' + params.mergeFields()
            + '/' + Arrays.toString(mergeFieldsArray) + ')';
    }
}

