package ru.yandex.msearch.collector;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.lucene.document.Field;
import org.apache.lucene.util.BytesRef;

import ru.yandex.charset.StringDecoder;
import ru.yandex.function.GenericFunction;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.msearch.fieldscache.CacheInput;
import ru.yandex.msearch.fieldscache.CacheOutput;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.search.json.fieldfunction.value.FloatValue;
import ru.yandex.search.json.fieldfunction.value.FunctionValue;
import ru.yandex.search.json.fieldfunction.value.IntValue;
import ru.yandex.search.json.fieldfunction.value.StringValue;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.UnhexStrings;
import ru.yandex.util.unicode.ByteSequence;

public interface YaField
    extends Comparable<YaField>, JsonValue, Utf8JsonValue
{
    long sizeInBytes();

    long longValue();

    default float floatValue() {
        return (float) longValue();
    }

    int doCompareTo(YaField other);

    FieldType type();

    @Override
    default int compareTo(final YaField other) {
        FieldType type = type();
        FieldType otherType = other.type();
        if (type == otherType) {
            return doCompareTo(other);
        } else {
            return type.compareTo(otherType);
        }
    }

    @Override
    default void writeValue(final JsonWriterBase writer) throws IOException {
        writer.value(toString());
    }

    @Override
    default void writeValue(final Utf8JsonWriter writer) throws IOException {
        writeValue((JsonWriterBase) writer);
    }

    public enum FieldType {
        STRING {
            @Override
            public YaField create(final byte[] buf) {
                return new StringYaField(buf);
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new StringValue(field.stringValue());
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                byte[] val = new byte[in.readVInt()];
                in.read(val, 0, val.length);
                return new StringYaField(val);
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                out.writeVInt(text.length);
                out.write(text.bytes, text.offset, text.length);
                return true;
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return value.stringValue();
            }
        },
        INTEGER {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new IntegerYaField(
                        BytesRef.parseInt(10, buf, 0, buf.length));
                } catch (NumberFormatException exc) {
                    return FieldType.LONG.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new IntValue(Integer.parseInt(field.stringValue()));
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                return new IntegerYaField(in.readVInt());
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    int value = text.parseInt();
                    out.writeVInt(value);
                    return true;
                } catch (NumberFormatException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return Integer.toString((int) value.intValue());
            }
        },
        LONG {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new LongYaField(
                        BytesRef.parseLong(10, buf, 0, buf.length));
                } catch (NumberFormatException exc) {
                    return FieldType.BIG_INTEGER.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new IntValue(Long.parseLong(field.stringValue()));
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                return new LongYaField(in.readVLong());
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    long value = text.parseLong();
                    out.writeVLong(value);
                    return true;
                } catch (NumberFormatException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                try {
                    return Long.toString(value.intValue());
                } catch (NumberFormatException e) {
                    //Expand to BigInteger
                    //TODO: Fix this
                    return BIG_INTEGER.indexableString(value);
                }
            }
        },
        FLOAT {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new FloatYaField(
                        Float.parseFloat(
                            StringDecoder.UTF_8.get()
                                .process(buf, 0, buf.length)));
                } catch (NumberFormatException exc) {
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new FloatValue(Double.parseDouble(field.stringValue()));
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                return new FloatYaField(Float.intBitsToFloat(in.readInt()));
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    float value = text.parseFloat();
                    out.writeInt(Float.floatToRawIntBits(value));
                    return true;
                } catch (NumberFormatException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return Double.toString(value.floatValue());
            }
        },
        BIG_INTEGER {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new BigIntegerYaField(
                        new BigInteger(
                            StringDecoder.UTF_8.get()
                                .process(buf, 0, buf.length)));
                } catch (NumberFormatException exc) {
                    return FieldType.BIG_DECIMAL.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new FloatValue(Double.parseDouble(field.stringValue()));
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                byte[] val = new byte[in.readVInt()];
                in.read(val, 0, val.length);
                return
                    new BigIntegerYaField(
                        new BigInteger(
                            StringDecoder.UTF_8.get()
                                .process(val, 0, val.length)));
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    final BigInteger value =
                        new BigInteger(text.utf8ToString());
                    final byte[] bytes = value.toByteArray();
                    out.writeVInt(bytes.length);
                    out.write(bytes, 0, bytes.length);
                    return true;
                } catch (NumberFormatException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return new BigInteger(value.stringValue()).toString();
            }
        },
        BIG_DECIMAL {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new BigDecimalYaField(
                        new BigDecimal(
                            StringDecoder.UTF_8.get()
                                .process(buf, 0, buf.length)));
                } catch (NumberFormatException exc) {
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                return new FloatValue(Double.parseDouble(field.stringValue()));
            }
            @Override
            public YaField create(final CacheInput in) throws IOException {
                //TODO: Implement
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                //TODO: Implement
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return new BigDecimal(value.stringValue()).toString();
            }
        },
        DESCENDING {
            @Override
            public YaField create(final byte[] buf) {
                throw new UnsupportedOperationException();
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                throw new UnsupportedOperationException();
            }
        },
        MULTI {
            @Override
            public YaField create(final byte[] buf) {
                throw new UnsupportedOperationException();
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                throw new UnsupportedOperationException();
            }
        },
        SIZE_ONLY {
            @Override
            public YaField create(final byte[] buf) {
                return new YaField.SizeOnlyYaField(buf.length);
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                throw new UnsupportedOperationException();
            }
        },
        FLOAT_ARRAY {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new FloatArrayYaField(buf);
                } catch (NumberFormatException e) {
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                //TODO: Implement
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return value.stringValue();
            }
        },
        INT_ARRAY {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new IntArrayYaField(buf);
                } catch (NumberFormatException e) {
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                //TODO: Implement
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                throw new UnsupportedOperationException();
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return value.stringValue();
            }
        },
        BYTE_ARRAY {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new ByteArrayYaField(
                        ByteArrayYaField.parse(buf, 0, buf.length));
                } catch (Exception e) {
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                byte[] value = new byte[in.readVInt()];
                in.read(value, 0, value.length);
                return new ByteArrayYaField(value);
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    byte[] value = ByteArrayYaField.parse(
                        text.bytes,
                        text.offset,
                        text.length);
                    out.writeVInt(value.length);
                    out.write(value, 0, value.length);
                    return true;
                } catch (IllegalArgumentException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return value.stringValue();
            }
        },
        LOW_PRECISION_BYTE_ARRAY {
            @Override
            public YaField create(final byte[] buf) {
                try {
                    return new ByteArrayYaField(
                        ByteArrayYaField.parse(buf, 0, buf.length));
                } catch (Exception e) {
//                    e.printStackTrace();
                    return FieldType.STRING.create(buf);
                }
            }

            @Override
            public FunctionValue createFieldFunctionValue(final Field field) {
                throw new UnsupportedOperationException();
            }

            @Override
            public YaField create(final CacheInput in) throws IOException {
                final int len = in.readVInt();
                byte[] value = new byte[len];
                byte b = 0;
                for (int i = 0; i < len; i++) {
                    final boolean even = (i & 1) == 0;
                    if (even) {
                        b = in.readByte();
                        value[i] = (byte) (b & 0xF0);
                    } else {
                        value[i] = (byte) ((b & 0x0F) << 4);
                    }
                }
                return new ByteArrayYaField(value);
            }

            @Override
            public boolean parseAndSerialize(
                final BytesRef text,
                final CacheOutput out)
                throws IOException
            {
                try {
                    byte[] value = ByteArrayYaField.parse(
                        text.bytes,
                        text.offset,
                        text.length);
                    out.writeVInt(value.length);
                    byte b = 0;
                    for (int i = 0; i < value.length; i++) {
                        final boolean odd = (i & 1) == 1;
                        if (odd) {
                            b |= (value[i] & 0xF0) >>> 4;
                            out.writeByte(b);
                            b = 0;
                        } else {
                            b |= value[i] & 0xF0;
                        }
                    }
                    if (value.length > 0 && (value.length & 0x1) == 1) {
                        out.writeByte(b);
                    }
                    return true;
                } catch (IllegalArgumentException e) {
                    return false;
                }
            }

            @Override
            public String indexableString(final FunctionValue value) {
                return value.stringValue();
            }
        };

        public abstract YaField create(final byte[] buf);

        public abstract YaField create(final CacheInput in) throws IOException;

        public abstract boolean parseAndSerialize(
            final BytesRef text,
            final CacheOutput out)
            throws IOException;


        public abstract FunctionValue createFieldFunctionValue(
            final Field field);

        public abstract String indexableString(final FunctionValue value);
    }

    public static class StringYaField implements YaField {
        private static final byte[] EMPTY_BUF = new byte[0];

        public static StringYaField EMPTY = new StringYaField(EMPTY_BUF);

        private final byte[] buf;

        public StringYaField(final byte[] buf) {
            this.buf = buf;
        }

        @Override
        public FieldType type() {
            return FieldType.STRING;
        }

        @Override
        public long sizeInBytes() {
            return buf.length << 1;
        }

        @Override
        public long longValue() {
            try {
                return BytesRef.parseLong(10, buf, 0, buf.length);
            } catch (NumberFormatException e) {
                return Long.MIN_VALUE;
            }
        }

        @Override
        public float floatValue() {
            try {
                return Float.parseFloat(toString());
            } catch (NumberFormatException e) {
                return 0.0f;
            }
        }

        @Override
        public String toString() {
            return StringDecoder.UTF_8.get().process(buf, 0, buf.length);
        }

        @Override
        public int doCompareTo(final YaField o) {
            StringYaField other = (StringYaField) o;
            byte[] otherBuf = other.buf;
            int end = Math.min(buf.length, otherBuf.length);
            for (int pos = 0; pos < end; ++pos) {
                int lhs = buf[pos] & 0xff;
                int rhs = otherBuf[pos] & 0xff;
                int diff = lhs - rhs;
                if (diff != 0) {
                    return diff;
                }
            }
            return buf.length - otherBuf.length;
        }

        @Override
        public void writeValue(final Utf8JsonWriter writer) throws IOException {
            writer.value(buf, 0, buf.length);
        }

        // Object implementation
        @Override
        public int hashCode() {
            int hash = 0;
            for (byte b: buf) {
                hash = hash * 31 + b;
            }
            return hash;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof StringYaField) {
                StringYaField other = (StringYaField) o;
                if (buf.length == other.buf.length) {
                    byte[] otherBuf = other.buf;
                    for (int pos = 0; pos < buf.length; ++pos) {
                        if (buf[pos] != otherBuf[pos]) {
                            return false;
                        }
                    }
                    return true;
                } else {
                    return false;
                }
            }
            return false;
        }
    }

    public static class SizeOnlyYaField implements YaField {
        private final int sizeInBytes;

        public SizeOnlyYaField(final int sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
        }

        public SizeOnlyYaField(final Field field) {
            int size = 0;
            if (field.isBinary()) {
                byte[] data = field.getBinaryValue();
                if (data.length == 4) {
                    size =
                        ((data[0] & 0xFF) << 24)
                        | ((data[1] & 0xFF) << 16)
                        | ((data[2] & 0xFF) << 8)
                        | (data[3] & 0xFF);
                } else {
                    size = data.length;
                }
            } else {
                size = field.stringValue().length() << 1;
            }
            sizeInBytes = size;
        }

        @Override
        public FieldType type() {
            return FieldType.SIZE_ONLY;
        }

        @Override
        public long sizeInBytes() {
            return sizeInBytes;
        }

        @Override
        public long longValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public float floatValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public String toString() {
//            return "SIZE_ONLY";
            throw new UnsupportedOperationException();
        }

        @Override
        public int hashCode() {
            return sizeInBytes;
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof SizeOnlyYaField
                && sizeInBytes == ((SizeOnlyYaField) o).sizeInBytes;
        }

        @Override
        public int doCompareTo(final YaField other) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            throw new UnsupportedOperationException();
        }
    }

    public static class IntegerYaField implements YaField {
        private final int value;

        public IntegerYaField(final int value) {
            this.value = value;
        }

        @Override
        public FieldType type() {
            return FieldType.INTEGER;
        }

        @Override
        public long sizeInBytes() {
            return Integer.BYTES;
        }

        @Override
        public long longValue() {
            return value;
        }

        @Override
        public float floatValue() {
            return (float) value;
        }

        @Override
        public String toString() {
            return Integer.toString(value);
        }

        @Override
        public int hashCode() {
            return value;
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof IntegerYaField
                && value == ((IntegerYaField) o).value;
        }

        @Override
        public int doCompareTo(final YaField other) {
            return Integer.compare(value, ((IntegerYaField) other).value);
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.value(value, true);
        }
    }

    public static class LongYaField implements YaField {
	private long value;

        public LongYaField(final long value) {
            this.value = value;
        }

        public void setValue(final long value) {
            this.value = value;
        }

        @Override
        public FieldType type() {
            return FieldType.LONG;
        }

        @Override
        public long sizeInBytes() {
            return Long.BYTES;
        }

        @Override
        public long longValue() {
            return value;
        }

        @Override
        public float floatValue() {
            return (float) value;
        }

        @Override
        public String toString() {
            return Long.toString(value);
        }

        @Override
        public int hashCode() {
            return (int) (value ^ (value >>> 32));
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof LongYaField
                && value == ((LongYaField) o).value;
        }

        @Override
        public int doCompareTo(final YaField other) {
            return Long.compare(value, other.longValue());
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.value(value, true);
        }
    }

    public static class FloatYaField implements YaField {
	private final float value;

	public FloatYaField(final float value) {
            this.value = value;
	}

        @Override
        public FieldType type() {
            return FieldType.FLOAT;
        }

        @Override
        public long sizeInBytes() {
            return Float.BYTES;
        }

        @Override
        public long longValue() {
            return (long) value;
        }

        @Override
        public float floatValue() {
            return value;
        }

        @Override
        public String toString() {
            return Float.toString(value);
        }

        @Override
        public int hashCode() {
            return Float.floatToIntBits(value);
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof FloatYaField
                && value == ((FloatYaField) o).value;
        }

        @Override
        public int doCompareTo(final YaField other) {
            return Float.compare(value, ((FloatYaField) other).value);
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.value(value, true);
        }
    }

    public static abstract class ObjectYaField<T extends Comparable<T>>
        implements YaField
    {
        protected final T value;

        public ObjectYaField(final T value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return value.toString();
        }

        @Override
        public int hashCode() {
            return value.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof ObjectYaField
                && value.equals(((ObjectYaField<?>) o).value);
        }
    }

    public static class BigIntegerYaField extends ObjectYaField<BigInteger> {
        public BigIntegerYaField(final BigInteger value) {
            super(value);
        }

        @Override
        public FieldType type() {
            return FieldType.BIG_INTEGER;
        }

        @Override
        public long sizeInBytes() {
            return value.bitLength() / 8 + 1;
        }

        @Override
        public long longValue() {
            return value.longValue();
        }

        @Override
        public float floatValue() {
            return value.floatValue();
        }

        @Override
        public int doCompareTo(final YaField other) {
            return value.compareTo(((BigIntegerYaField) other).value);
        }
    }

    public static class BigDecimalYaField extends ObjectYaField<BigDecimal> {
        private int size = -1;

	public BigDecimalYaField(final BytesRef value) {
            this(new BigDecimal(value.utf8ToString()));
	}

        public BigDecimalYaField(final BigDecimal value) {
            super(value);
        }

        @Override
        public FieldType type() {
            return FieldType.BIG_DECIMAL;
        }

        @Override
        public long sizeInBytes() {
            if (size == -1) {
                size = 4 + (value.toString().length() << 1);
            }
            return size;
        }

        @Override
        public long longValue() {
            return value.longValue();
        }

        @Override
        public float floatValue() {
            return value.floatValue();
        }

        @Override
        public int doCompareTo(final YaField other) {
            return value.compareTo(((BigDecimalYaField) other).value);
        }
    }

    public static class FloatArrayYaField implements YaField {
        private static final CollectionParser<
            String,
            List<String>,
            RuntimeException> PARSER = new CollectionParser<>(
                GenericFunction.identity(),
                ArrayList::new,
                '\n');

        private final float[] value;

        public FloatArrayYaField(final byte[] value) {
            this(parse(value));
        }

        public FloatArrayYaField(final float[] value) {
            this.value = value;
        }

        public static float[] parse(final byte[] value) {
            List<String> values = PARSER.apply(
                StringDecoder.UTF_8.get()
                    .process(value, 0, value.length));
            int size = values.size();
            float[] result = new float[size];
            for (int i = 0; i < size; ++i) {
                result[i] = Float.parseFloat(values.get(i));
            }
            return result;
        }

        @Override
        public FieldType type() {
            return FieldType.FLOAT_ARRAY;
        }

        @Override
        public long sizeInBytes() {
            return value.length * Float.BYTES;
        }

        @Override
        public long longValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public float floatValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < value.length; ++i) {
                if (i != 0) {
                    sb.append('\n');
                }
                sb.append(value[i]);
            }
            return new String(sb);
        }

        @Override
        public int hashCode() {
            return Arrays.hashCode(value);
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof FloatArrayYaField
                && Arrays.equals(value, ((FloatArrayYaField) o).value);
        }

        @Override
        public int doCompareTo(final YaField other) {
            float[] otherValue = ((FloatArrayYaField) other).value;
            int cmp = Integer.compare(value.length, otherValue.length);
            if (cmp == 0) {
                for (int i = 0; cmp == 0 && i < value.length; ++i) {
                    cmp = Float.compare(value[i], otherValue[i]);
                }
            }
            return cmp;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startArray();
            for (float f: value) {
                writer.value(f);
            }
            writer.endArray();
        }

        public float dotProduct(final float[] otherValue) {
            int len = Math.min(value.length, otherValue.length);
            float product = 0f;
            for (int i = 0; i < len; ++i) {
                product += value[i] * otherValue[i];
            }
            return product;
        }
    }

    public static class IntArrayYaField implements YaField {
        private static final CollectionParser<
            String,
            List<String>,
            RuntimeException> PARSER = new CollectionParser<>(
                GenericFunction.identity(),
                ArrayList::new,
                ',');

        private final int[] value;

        public IntArrayYaField(final byte[] value) {
            this(parse(value));
        }

        public IntArrayYaField(final int[] value) {
            this.value = value;
        }

        public static int[] parse(final byte[] value) {
            List<String> values = PARSER.apply(
                StringDecoder.UTF_8.get()
                    .process(value, 0, value.length));
            int size = values.size();
            int[] result = new int[size];
            for (int i = 0; i < size; ++i) {
                result[i] = Integer.parseInt(values.get(i));
            }
            return result;
        }

        @Override
        public FieldType type() {
            return FieldType.INT_ARRAY;
        }

        @Override
        public long sizeInBytes() {
            return value.length * Integer.BYTES;
        }

        @Override
        public long longValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public float floatValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < value.length; ++i) {
                if (i != 0) {
                    sb.append(',');
                }
                sb.append(value[i]);
            }
            return new String(sb);
        }

        @Override
        public int hashCode() {
            return Arrays.hashCode(value);
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof IntArrayYaField
                && Arrays.equals(value, ((IntArrayYaField) o).value);
        }

        @Override
        public int doCompareTo(final YaField other) {
            int[] otherValue = ((IntArrayYaField) other).value;
            int cmp = Integer.compare(value.length, otherValue.length);
            if (cmp == 0) {
                for (int i = 0; cmp == 0 && i < value.length; ++i) {
                    cmp = Integer.compare(value[i], otherValue[i]);
                }
            }
            return cmp;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startArray();
            for (int i: value) {
                writer.value(i);
            }
            writer.endArray();
        }

        public float dotProduct(final int[] otherValue) {
            int len = Math.min(value.length, otherValue.length);
            long product = 0;
            for (int i = 0; i < len; ++i) {
                product += value[i] * otherValue[i];
            }
            return (float) product;
        }

        public int minHashDotProduct(final int[] otherValue) {
            final int len = Math.min(value.length, otherValue.length);
            int dp = len;
            for (int i = 0; i < len; i++) {
                if (value[i] == otherValue[i]) {
                    dp--;
                }
            }
            return dp;
        }
    }

    public static class ByteArrayYaField implements YaField {
        private final byte[] value;

        public ByteArrayYaField(final byte[] value) {
            this.value = value;
        }

        public static byte[] parse(final byte[] bytes) {
            return parse(bytes, 0, bytes.length);
        }

        public byte[] getValue() {
            return value;
        }

        public static byte[] parse(
            final byte[] bytes,
            final int off,
            final int len)
        {
            if ((len & 1) != 0) {
                throw new IllegalArgumentException(
                    "Odd length string: "
                    + StringDecoder.UTF_8.get().process(bytes, off, len));
            }
            byte[] buf = new byte[len >> 1];
            for (int i = 0, j = 0; i < len;) {
                buf[j++] =
                    UnhexStrings.unhex(bytes[off + i++], bytes[off + i++]);
            }
            return buf;
        }

        @Override
        public FieldType type() {
            return FieldType.BYTE_ARRAY;
        }

        @Override
        public long sizeInBytes() {
            return value.length;
        }

        @Override
        public long longValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public float floatValue() {
            throw new UnsupportedOperationException();
        }

        @Override
        public String toString() {
            return HexStrings.UPPER.toString(value);
        }

        @Override
        public int hashCode() {
            return Arrays.hashCode(value);
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof ByteArrayYaField
                && Arrays.equals(value, ((ByteArrayYaField) o).value);
        }

        @Override
        public int doCompareTo(final YaField other) {
            byte[] otherValue = ((ByteArrayYaField) other).value;
            int cmp = Integer.compare(value.length, otherValue.length);
            if (cmp == 0) {
                for (int i = 0; cmp == 0 && i < value.length; ++i) {
                    cmp = Byte.compare(value[i], otherValue[i]);
                }
            }
            return cmp;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startArray();
            for (byte b: value) {
                writer.value(b);
            }
            writer.endArray();
        }

        public int dotProduct(final byte[] otherValue) {
            int len = Math.min(value.length, otherValue.length);
            int product = 0;
            for (int i = 0; i < len; ++i) {
                product += ((int) value[i]) * ((int) otherValue[i]);
            }
            return product;
        }
    }
}

