package ru.yandex.msearch.collector.aggregate;

import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Set;

import org.apache.lucene.util.StringHelper;

import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;

public class CountFunc implements AggregateFunc {
    private final String outFieldName;
    private final String fieldName;
    private final int outFieldIndex;
    private final Integer fieldIndex;
    private YaField.StringYaField targetValue;
    private final YaField.LongYaField valueField;
    private int count = 0;

    public CountFunc(
        final String outFieldName,
        final String args,
        final FieldToIndex fieldToIndex)
        throws ParseException
    {
        this.outFieldName = StringHelper.intern(outFieldName);
        outFieldIndex = fieldToIndex.indexFor(this.outFieldName);
        if (args == null) {
            throw new ParseException("Null args supplied", 0);
        }
        String[] parts = args.trim().split(";");
        if (parts.length > 2) {
            throw new ParseException(
                "Expecting 1 or 2 args but got " + parts.length + Arrays.asList(parts),
                0);
        }
        fieldName = StringHelper.intern(parts[0].trim());
        fieldIndex = fieldToIndex.indexFor(fieldName);
        if (parts.length > 1) {
            targetValue = new YaField.StringYaField(parts[1].trim().getBytes(StandardCharsets.UTF_8));
        } else {
            targetValue = null;
        }
        // Original aggregate func should not be modified
        valueField = null;
    }

    private CountFunc(final CountFunc sample) {
        outFieldName = sample.outFieldName;
        fieldName = sample.fieldName;
        outFieldIndex = sample.outFieldIndex;
        fieldIndex = sample.fieldIndex;
        targetValue = sample.targetValue;
        valueField = new YaField.LongYaField(0L);
    }

    @Override
    public AggregateFunc clone() {
        return new CountFunc(this);
    }

    @Override
    public String outFieldName() {
        return outFieldName;
    }

    @Override
    public int outFieldIndex() {
        return outFieldIndex;
    }

    @Override
    public YaField aggField() {
        return valueField;
    }

    @Override
    public void updateValue(final YaDoc3 doc) {
        YaField docField = doc.getField(fieldIndex);
        if (docField != null && (targetValue == null || docField.compareTo(targetValue) == 0)) {
            count++;
            valueField.setValue(count);
        }
    }

    @Override
    public Set<String> loadFields() {
        return Collections.singleton(fieldName);
    }

    @Override
    public String toString() {
        return "COUNT(" + fieldName + ')';
    }
}
