package ru.yandex.msearch.collector.aggregate;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;

import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;

import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;

import ru.yandex.util.string.StringUtils;

public class LimitedSetAggFunc implements AggregateFunc {
    private final String outFieldName;
    private final String fieldName;
    private final int outFieldIndex;
    private final int fieldIndex;
    private final int limit;
    private final Set<String> set;
    private YaField.StringYaField valueField;

    protected LimitedSetAggFunc(
        final String outFieldName,
        final String args,
        final FieldToIndex fieldToIndex)
        throws ParseException
    {
        this.outFieldName = StringHelper.intern(outFieldName);
        outFieldIndex = fieldToIndex.indexFor(this.outFieldName);
        if (args == null) {
            throw new ParseException("Null args supplied", 0);
        }

        String[] parts = args.trim().split(";");
        if (parts.length != 2) {
            throw new ParseException(
                "Expecting 2 args but got "
                    + parts.length
                    + Arrays.asList(parts).toString(), 0);
        }

        fieldName = StringHelper.intern(parts[0].trim());
        fieldIndex = fieldToIndex.indexFor(fieldName);

        try {
            limit = Integer.parseInt(parts[1].trim());
        } catch (NumberFormatException nfe) {
            throw new ParseException(
                "Limit should be integer, but got "
                    + parts[1], parts[0].length());
        }

        // Original aggregate func should not be modified
        set = null;
        valueField = null;
    }

    private LimitedSetAggFunc(final LimitedSetAggFunc sample) {
        outFieldName = sample.outFieldName;
        fieldName = sample.fieldName;
        outFieldIndex = sample.outFieldIndex;
        fieldIndex = sample.fieldIndex;
        limit = sample.limit;
        set = new LinkedHashSet<>(limit << 1);
        valueField = YaField.StringYaField.EMPTY;
    }

    @Override
    public AggregateFunc clone() {
        return new LimitedSetAggFunc(this);
    }

    @Override
    public String outFieldName() {
        return outFieldName;
    }

    @Override
    public int outFieldIndex() {
        return outFieldIndex;
    }

    @Override
    public YaField aggField() {
        return valueField;
    }

    @Override
    public void updateValue(final YaDoc3 doc) {
        YaField docField = doc.getField(fieldIndex);
        if (docField != null && set.size() < limit) {
            String value = docField.toString();
            if (set.add(value.toLowerCase(Locale.ROOT))) {
                StringBuilder sb = new StringBuilder(valueField.toString());
                if (sb.length() > 0) {
                    sb.append('\n');
                }
                sb.append(value);

                valueField = new YaField.StringYaField(
                    StringUtils.getUtf8Bytes(new String(sb)));
            }
        }
    }

    @Override
    public Set<String> loadFields() {
        return Collections.singleton(fieldName);
    }

    @Override
    public String toString() {
        return "SET(" + fieldName + ')';
    }
}
