package ru.yandex.msearch.collector.cluster;

import java.io.IOException;

import java.util.List;
import java.util.Map;

import org.apache.lucene.util.BytesRef;

import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.writer.Utf8JsonValue;
import ru.yandex.json.writer.Utf8JsonWriter;

import ru.yandex.msearch.ClusteringConfig;
import ru.yandex.msearch.collector.YaField;

public abstract class AbstractCluster<T extends AbstractCluster> {
    protected final Map<String, Map<String, Counter>> counters;
    protected final ClusteringConfig config;
    protected int size = 1;
    protected long minDate;
    protected long maxDate;

    public AbstractCluster(
        final ClusteringConfig config,
        final Map<String, Map<String, Counter>> counters,
        final long date)
    {
        this.config = config;
        this.counters = counters;
        minDate = date;
        maxDate = date;
    }

    protected abstract void addDoc(final ClusterDoc doc);
    protected abstract void addDocs(final T other);
    protected abstract List<ClusterDoc> sortedDocs();

    public int size() {
        return size;
    }

    public long minDate() {
        return minDate;
    }

    public long maxDate() {
        return maxDate;
    }

    public void add(final ClusterDoc doc) {
        addDoc(doc);
        long date = doc.date();
        if (date > maxDate) {
            maxDate = date;
        } else if (date < minDate) {
            minDate = date;
        }
        ++size;
    }

    public void add(final T other) {
        addDocs(other);
        size += other.size;
        if (maxDate < other.maxDate) {
            maxDate = other.maxDate;
        }
        if (minDate > other.minDate) {
            minDate = other.minDate;
        }
        if (other.counters != null) {
            Map<String, Map<String, Counter>> otherCounters = other.counters;
            for (Map.Entry<String, Map<String, Counter>> entry
                : otherCounters.entrySet())
            {
                Map<String, Counter> counters =
                    this.counters.get(entry.getKey());
                for (Map.Entry<String, Counter> subentry
                    : entry.getValue().entrySet())
                {
                    String value = subentry.getKey();
                    Counter counter = counters.get(value);
                    if (counter == null) {
                        counters.put(value, subentry.getValue());
                    } else {
                        counter.add(subentry.getValue().get());
                    }
                }
            }
        }
    }

    public boolean isClose(
        final ClusterDoc remote,
        final long dateFrom,
        final long dateTo,
        final double distanceThreshold)
    {
        return false;
    }

    public void incrementCounter(final String name, final String value) {
        Map<String, Counter> counters = this.counters.get(name);
        Counter counter = counters.get(value);
        if (counter == null) {
            counters.put(value, new Counter());
        } else {
            counter.increment();
        }
    }

    public Map<String, Map<String, Counter>> counters() {
        return counters;
    }

    public void writeTo(final JsonWriterBase writer) throws IOException {
        String[] fields = config.fieldList();
        boolean skipNulls = config.getFieldsConfig().skipNulls();
        List<ClusterDoc> docs = sortedDocs();
        int len = Math.min(docs.size(), config.mergedLength());
        writer.startArray();
        for (int i = 0; i < len; ++i) {
            YaField[] docFields = docs.get(i).fields();
            writer.startObject();
            for (int j = 0; j < docFields.length; ++j) {
                YaField value = docFields[j];
                if (value == null) {
                    if (!skipNulls) {
                        writer.key(fields[j]);
                        writer.nullValue();
                    }
                } else {
                    writer.key(fields[j]);
                    value.writeValue(writer);
                }
            }
            writer.endObject();
        }
        writer.endArray();
    }

    public void writeTo(final Utf8JsonWriter writer) throws IOException {
        BytesRef[] fields = config.utf8FieldList();
        boolean skipNulls = config.getFieldsConfig().skipNulls();
        List<ClusterDoc> docs = sortedDocs();
        int len = Math.min(docs.size(), config.mergedLength());
        writer.startArray();
        for (int i = 0; i < len; ++i) {
            YaField[] docFields = docs.get(i).fields();
            writer.startObject();
            for (int j = 0; j < docFields.length; ++j) {
                YaField value = docFields[j];
                if (value == null) {
                    if (!skipNulls) {
                        writer.key(fields[j]);
                        writer.nullValue();
                    }
                } else {
                    writer.key(fields[j]);
                    writer.value((Utf8JsonValue) value);
                }
            }
            writer.endObject();
        }
        writer.endArray();
    }

    public static class Counter {
        private int value = 1;

        public int get() {
            return value;
        }

        public void increment() {
            ++value;
        }

        public void add(final int value) {
            this.value += value;
        }
    }
}

