package ru.yandex.msearch.collector.cluster;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import ru.yandex.msearch.ClusteringConfig;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;

public class ClusterDoc implements Comparable<ClusterDoc> {
    private static final Comparator<YaField> COMPARATOR =
        Comparator.nullsFirst(Comparator.naturalOrder());
    private static final float R = 6371e3f;

    private final YaField[] fields;
    private final float lat;
    private final float lon;
    private final long date;
//    private final float latSin;
//    private final float latCos;

//    private final float tlat;
//    private final float tlon;

    public ClusterDoc(final YaDoc3 yadoc, final ClusteringConfig config) {
        int[] fieldIndexes = config.fieldIndexes();
        fields = new YaField[fieldIndexes.length];
        for (int i = 0; i < fieldIndexes.length; ++i) {
            fields[i] = yadoc.getField(fieldIndexes[i]);
        }
//        float lat = extractFloat(yadoc, config.latitudeField());
//        if (lat != Float.MAX_VALUE) {
//            this.lat = (float) Math.toRadians(lat);
//        } else {
//            this.lat = lat;
//        }
        lat = extractFloat(yadoc, config.latitudeField());
        lon = extractFloat(yadoc, config.longitudeField());
/*        if (config.latitudeField() != -1
            && config.longitudeField() != -1
            && yadoc.getField(config.latitudeField()) != null
            && yadoc.getField(config.longitudeField()) != null)
        {
            tlat = yadoc.getField(config.latitudeField()).floatValue();
            tlon = yadoc.getField(config.longitudeField()).floatValue();
        } else {
            tlat = 0;
            tlon = 0;
        }
*/
//        latSin = (float) Math.sin(lat);
//        latCos = (float) Math.cos(lat);
//        latSin = (float) Math.sin(lat);
//        latCos = (float) Math.cos(lat);
        YaField date = yadoc.getField(config.dateField());
        if (date == null) {
            this.date = 0L;
        } else {
            this.date = date.longValue();
        }
    }

    private static float extractFloat(final YaDoc3 yadoc, final int field) {
        if (field == -1) {
            return Float.MAX_VALUE;
        } else {
            YaField fieldValue = yadoc.getField(field);
            if (fieldValue == null) {
                return Float.MAX_VALUE;
            } else {
                try {
//                    return fieldValue.floatValue();
                    return (float) Math.toRadians(fieldValue.floatValue());
                } catch (NumberFormatException e) {
                    return Float.MAX_VALUE;
                }
            }
        }
    }

    public YaField[] fields() {
        return fields;
    }

    public long date() {
        return date;
    }

    public boolean hasValidCoords() {
        return lat != Float.MAX_VALUE && lon != Float.MAX_VALUE;
    }

    public float distanceTo(final ClusterDoc other) {
//        float x = (lon - other.lon) * (float) Math.cos((lat + other.lat) / 2);
        float x = (lon - other.lon) * Riven.cos((lat + other.lat) / 2);
//        float x = (lon - other.lon) * Fast.cos((lat + other.lat) / 2);
        float y = lat - other.lat;
        float distance = (float) Math.sqrt(x * x + y * y);
        return distance;
    }

    @Override
    public int compareTo(final ClusterDoc other) {
        int cmp = Long.compare(date, other.date);
        if (cmp == 0) {
            cmp = Float.compare(lat, other.lat);
            if (cmp == 0) {
                cmp = Float.compare(lon, other.lon);
                if (cmp == 0) {
                    cmp = Integer.compare(
                        fields.length,
                        other.fields.length);
                    for (int i = 0; i < fields.length && cmp == 0; ++i) {
                        cmp = COMPARATOR.compare(fields[i], other.fields[i]);
                    }
                }
            }
        }
        return cmp;
    }

    private static final class Fast {
        private static final float PI = 3.1415927f;
        private static final float MINUS_PI = -PI;
        private static final float DOUBLE_PI = PI * 2f;
        private static final float PI_2 = PI / 2f;

        private static final float CONST_1 = 4f / PI;
        private static final float CONST_2 = 4f / (PI * PI);

        public static final float sin(float x) {
            if (x < MINUS_PI) {
                x += DOUBLE_PI;
            } else if (x > PI) {
                x -= DOUBLE_PI;
            }

            return (x < 0f) ? (CONST_1 * x + CONST_2 * x * x)
                    : (CONST_1 * x - CONST_2 * x * x);
        }

        public static final float cos(float x) {
            if (x < MINUS_PI) {
                x += DOUBLE_PI;
            } else if (x > PI) {
                x -= DOUBLE_PI;
            }

            x += PI_2;

            if (x > PI) {
                x -= DOUBLE_PI;
            }

            return (x < 0f) ? (CONST_1 * x + CONST_2 * x * x)
                    : (CONST_1 * x - CONST_2 * x * x);
        }
    }

    private static final class Riven {
        private static final int SIN_BITS, SIN_MASK, SIN_COUNT;
        private static final float RAD_FULL, RAD_TO_INDEX;
        private static final float DEG_FULL, DEG_TO_INDEX;
        private static final float[] SIN, COS;

        static {
            SIN_BITS = 12;
            SIN_MASK = ~(-1 << SIN_BITS);
            SIN_COUNT = SIN_MASK + 1;

            RAD_FULL = (float) (Math.PI * 2.0);
            DEG_FULL = (float) (360.0);
            RAD_TO_INDEX = SIN_COUNT / RAD_FULL;
            DEG_TO_INDEX = SIN_COUNT / DEG_FULL;

            SIN = new float[SIN_COUNT];
            COS = new float[SIN_COUNT];

            for (int i = 0; i < SIN_COUNT; i++) {
                SIN[i] = (float) Math.sin((i + 0.5f) / SIN_COUNT * RAD_FULL);
                COS[i] = (float) Math.cos((i + 0.5f) / SIN_COUNT * RAD_FULL);
            }

            // Four cardinal directions (credits: Nate)
            for (int i = 0; i < 360; i += 90) {
                SIN[(int) (i * DEG_TO_INDEX) & SIN_MASK] =
                    (float) Math.sin(i * Math.PI / 180.0);
                COS[(int) (i * DEG_TO_INDEX) & SIN_MASK] =
                    (float) Math.cos(i * Math.PI / 180.0);
            }
        }

        public static final float sin(float rad) {
            return SIN[(int) (rad * RAD_TO_INDEX) & SIN_MASK];
        }

        public static final float cos(float rad) {
            return COS[(int) (rad * RAD_TO_INDEX) & SIN_MASK];
        }
    }
}
