package ru.yandex.msearch.collector.docprocessor;

import java.io.IOException;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.logging.Level;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;

import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.Weight;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.ProcessorRequestContext;
import ru.yandex.msearch.Searcher;

import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.fieldscache.CacheInput;
import ru.yandex.msearch.fieldscache.FieldsCache;

import ru.yandex.msearch.collector.YaField;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.string.StringUtils;

public abstract class AbstractJoinDocProcessor implements DocProcessor {
    protected static final Map<String, YaField> EMPTY_DOC =
        Collections.emptyMap();
    private static final List<CacheInput> EMPTY_CACHE = Collections.emptyList();

    protected Searcher searcher;
    protected Prefix prefix;
    protected Map<String, Prefix> prefixes;
    protected final boolean multiprefix;
    protected final ProcessorRequestContext context;
    protected long subQueries = 0;
    protected HashMap<IndexReader, List<CacheInput>> cacheMap;
    protected FieldsCache fieldsCache;
    protected int reads = 0;
    protected int docsFromCache = 0;
    protected final int prefixFieldIndex;

    public AbstractJoinDocProcessor(
        final ProcessorRequestContext context)
        throws ParseException
    {
        this.context = context;

        if (context.prefix().size() < 1) {
            throw new ParseException(
                "unprefixed requests not supported by this doc processor",
                0);
        }

        if (context.prefix().size() > 1) {
            multiprefix = true;
            this.prefixFieldIndex =
                context.fieldToIndex().indexFor(Config.PREFIX_FIELD_KEY);
            this.prefix = null;
            this.searcher = null;
            Map<String, Prefix> prefixMap = new LinkedHashMap<>();
            for (Prefix prefix: context.prefix()) {
                prefixMap.put(prefix.toStringFast(), prefix);
            }
            this.prefixes = prefixMap;
        } else {
            multiprefix = false;
            this.prefixFieldIndex = -1;
            this.prefix = context.prefix().iterator().next();
            try {
                this.searcher = context.index().getSearcher(prefix, context.syncSearcher());
            } catch (IOException e) {
                ParseException pe = new ParseException("Unhandled error", 0);
                pe.initCause(e);
                throw pe;
            }
        }


        fieldsCache = context.fieldsCache();
    }

    protected boolean multiprefixSearcherUpdate(final YaDoc3 doc) throws IOException {
        YaField prefixField = doc.getField(prefixFieldIndex);
        if (prefixField == null) {
            if (context.debug()) {
                context.ctx().logger().warning(
                    "Not found prefix field in doc " + doc);
            }
            return false;
        }

        Prefix prefix = prefixes.get(prefixField.toString());
        if (prefix == null) {
            if (context.debug()) {
                context.ctx().logger().warning(
                    "Prefix from doc " + doc + " not found in request " + prefixField);
            }
            return false;
        }
        if (this.prefix == null || !this.prefix.equals(prefix)) {
            this.prefix = prefix;
            if (searcher != null) {
                try {
                    searcher.free();
                } catch (IOException e) {
                    context.ctx().logger().log(
                        Level.SEVERE,
                        "Searcher close error",
                        e);
                }
            }

            this.searcher = context.index().getSearcher(prefix, context.syncSearcher());
        }

        return true;
    }

    @Override
    public void after() {
        context.ctx().logger().info("JoinDocProcessor: docsRead: " + reads
            + ", docsFromCache: " + docsFromCache);
        if (searcher != null) {
            try {
                searcher.free();
            } catch (IOException e) {
                context.ctx().logger().log(
                    Level.SEVERE,
                    "Searcher close error",
                    e);
            }
            cacheMap = null;
            searcher = null;
        }
    }

    protected Map<String, YaField> readDocument(
        final IndexReader reader,
        final int docId,
        final FieldSelector fs)
        throws IOException
    {
        List<CacheInput> caches = null;
        if (fieldsCache != null) {
            if (cacheMap == null) {
                cacheMap = new HashMap<>();
            }
            caches = cacheMap.get(reader);
            if (caches == null) {
                caches = fieldsCache.getCachesFor(reader, loadFields());
                if (caches == null) {
                    caches = EMPTY_CACHE;
                }
                cacheMap.put(reader, caches);
            }
        } else {
            caches = EMPTY_CACHE;
        }
        Map<String, YaField> doc = null;
        if (caches != EMPTY_CACHE) {
            for (CacheInput oneFieldCache: caches) {
                YaField cachedField;
                if (oneFieldCache.seek(docId)) {
                    cachedField = oneFieldCache.field();
                } else {
                    doc = null;
                    break;
                }
                if (doc == null) {
                    doc = new IdentityHashMap<>();
                }
                doc.put(oneFieldCache.fieldname(), cachedField);
            }
        }
        if (doc == null) {
            doc = new IdentityHashMap<>();
            Document luceneDoc = reader.document(docId, fs);
            for (String field: loadFields()) {
                String value = luceneDoc.get(field);
                YaField yaValue;
                if (value != null) {
                    yaValue =
                        new YaField.StringYaField(
                            StringUtils.getUtf8Bytes(value));
                } else {
                    yaValue = null;
                }
                doc.put(field, yaValue);
            }
            reads++;
        } else {
            docsFromCache++;
        }
        return doc;
    }

    protected Map<String, YaField> extractDoc(
        final Query query,
        final FieldSelector selector)
        throws IOException
    {
        subQueries += 1;
        IndexSearcher indexSearcher = searcher.searcher();
        Weight weight = query.weight(indexSearcher);
        IndexReader.AtomicReaderContext[] leaves =
            indexSearcher.getTopReaderContext().leaves();
        if (leaves != null) {
            leaves = leaves.clone();
            Arrays.sort(
                leaves,
                (x, y) -> Long.compare(x.docBase, y.docBase));
            for (IndexReader.AtomicReaderContext context: leaves) {
                DocIdSetIterator docIds =
                    weight.scorer(context, Weight.ScorerContext.def());
                if (docIds != null) {
                    int docId = docIds.nextDoc();
                    if (docId == DocIdSetIterator.NO_MORE_DOCS) {
                        continue;
                    }

                    if (this.context.debug()) {
                        this.context.ctx().logger().info(
                            "Found docId " + (docId + context.docBase));
                    }

                    return readDocument(context.reader, docId, selector);
                }
            }
        }

        if (context.debug()) {
            context.ctx().logger().info("No doc foudn for " + query);
        }
        return EMPTY_DOC;
    }

    protected TermQuery buildJoinQuery(
        final String value,
        final String joinFieldRight)
    {
        Term term  = null;
        if(context.queryParser().getAnalyzer()
            instanceof PrefixingAnalyzerWrapper)
        {
            PrefixingAnalyzerWrapper pa =
                (PrefixingAnalyzerWrapper) context.queryParser().getAnalyzer();
            if (pa.prefixed(joinFieldRight)) {
                term = new Term(
                    joinFieldRight,
                    StringUtils.concat(
                        pa.getPrefix(),
                        pa.getSeparator(),
                        value));
            }
        }

        if (term == null) {
            term = new Term(joinFieldRight, value);
        }

        return new TermQuery(term);
    }

    public abstract Set<String> loadFields();

    protected static final class DocProcessorQueryCache<T>
        extends LinkedHashMap<String, T>
    {
        private final int cacheSize;

        protected DocProcessorQueryCache(final int cacheSize) {
            super(cacheSize, 0.75f, true);

            this.cacheSize = cacheSize;
        }

        @Override
        protected boolean removeEldestEntry(
            final Map.Entry<String, T> eldest)
        {
            boolean expire = size() > cacheSize;
            return expire;
        }
    }
}
