package ru.yandex.msearch.collector.docprocessor;

import java.io.IOException;
import java.util.List;

import ru.yandex.msearch.collector.YaDoc3;

public class CompositeDocProcessor implements DocProcessor {
    private final DocProcessor[] processors;

    private CompositeDocProcessor(final List<DocProcessor> processors) {
        this.processors =
            processors.toArray(new DocProcessor[processors.size()]);
    }

    @Override
    public void apply(final ModuleFieldsAggregator aggregator) {
        for (DocProcessor processor: processors) {
            processor.apply(aggregator);
        }
    }

    @Override
    public boolean processWithFilter(final YaDoc3 doc) throws IOException {
        boolean keep = true;
        for (DocProcessor processor: processors) {
            keep = processor.processWithFilter(doc);
            if (!keep) {
                break;
            }
        }

        return keep;
    }

    @Override
    public void after() {
        for (DocProcessor processor: processors) {
            processor.after();
        }
    }

    public static DocProcessor create(final List<DocProcessor> processors) {
        switch (processors.size()) {
            case 0:
                return NullDocProcessor.INSTANCE;
            case 1:
                return processors.get(0);
            case 2:
                return new DoubleDocProcessor(
                    processors.get(0),
                    processors.get(1));
            case 3:
                return new TripleDocProcessor(
                    processors.get(0),
                    processors.get(1),
                    processors.get(2));
            default:
                return new CompositeDocProcessor(processors);
        }
    }

    private static class DoubleDocProcessor implements DocProcessor {
        private final DocProcessor first;
        private final DocProcessor second;

        DoubleDocProcessor(
            final DocProcessor first,
            final DocProcessor second)
        {
            this.first = first;
            this.second = second;
        }

        @Override
        public void apply(final ModuleFieldsAggregator aggregator) {
            first.apply(aggregator);
            second.apply(aggregator);
        }

        @Override
        public boolean processWithFilter(final YaDoc3 doc) throws IOException {
            if (first.processWithFilter(doc)) {
                return second.processWithFilter(doc);
            } else {
                return false;
            }
        }

        @Override
        public void after() {
            first.after();
            second.after();
        }
    }

    private static class TripleDocProcessor implements DocProcessor {
        private final DocProcessor first;
        private final DocProcessor second;
        private final DocProcessor third;

        TripleDocProcessor(
            final DocProcessor first,
            final DocProcessor second,
            final DocProcessor third)
        {
            this.first = first;
            this.second = second;
            this.third = third;
        }

        @Override
        public void apply(final ModuleFieldsAggregator aggregator) {
            first.apply(aggregator);
            second.apply(aggregator);
            third.apply(aggregator);
        }

        @Override
        public boolean processWithFilter(final YaDoc3 doc) throws IOException {
            if (first.processWithFilter(doc)) {
                if (second.processWithFilter(doc)) {
                    return third.processWithFilter(doc);
                }
            }

            return false;
        }

        @Override
        public void after() {
            first.after();
            second.after();
            third.after();
        }
    }
}

