package ru.yandex.msearch.collector.docprocessor;

import java.text.ParseException;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.apache.lucene.util.StringHelper;

import org.joda.time.DateTimeField;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.GregorianChronology;

import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;

public class ExtractDateDocProcessor implements DocProcessor {
    private static final DateTimeField[] DATE_FIELDS = new DateTimeField[] {
        GregorianChronology.getInstanceUTC().year(),
        GregorianChronology.getInstanceUTC().monthOfYear(),
        GregorianChronology.getInstanceUTC().dayOfMonth()
    };

    private static final DateTimeZone TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");
    private static final YaField[] CACHE = new YaField[2039];

    static {
        for (int i = 1; i <= 31; ++i) {
            CACHE[i] = new YaField.IntegerYaField(i);
        }
        for (int i = 1970; i < CACHE.length; ++i) {
            CACHE[i] = new YaField.IntegerYaField(i);
        }
    }

    private final String[] outFields;
    private final String field;
    private final int[] outFieldIndexes;
    private final int fieldIndex;

    public ExtractDateDocProcessor(
        final String args,
        final FieldToIndex fieldToIndex)
        throws ParseException
    {
        if (args == null || args.isEmpty()) {
            throw new ParseException("Arguments required", 0);
        }
        int space = args.indexOf(' ');
        if (space <= 0 || space + 1 == args.length()) {
            throw new ParseException(
                "ExtractDateDocProcessor args expected to be in form: "
                + "field yearfield[,monthfield[,dayfield]]",
                0);
        }
        field = StringHelper.intern(args.substring(0, space));
        fieldIndex = fieldToIndex.indexFor(field);

        String[] outFields = args.substring(space + 1).split(",");
        if (outFields.length > DATE_FIELDS.length) {
            throw new ParseException(
                "At most " + DATE_FIELDS.length + " output fields allowed",
                0);
        }
        this.outFields = new String[outFields.length];
        outFieldIndexes = new int[outFields.length];
        for (int i = 0; i < outFields.length; ++i) {
            String outField = StringHelper.intern(outFields[i]);
            this.outFields[i] = outField;
            outFieldIndexes[i] = fieldToIndex.indexFor(outField);
        }
    }

    @Override
    public void apply(final ModuleFieldsAggregator aggregator) {
        aggregator.addWithSingleLoad(field, outFields);
    }

    @Override
    public void process(final YaDoc3 doc) {
        YaField field = doc.getField(fieldIndex);
        if (field != null) {
            long millis =
                TIMEZONE.convertUTCToLocal(field.longValue() * 1000L);
            for (int i = 0; i < outFieldIndexes.length; ++i) {
                int value = DATE_FIELDS[i].get(millis);
                if (value < CACHE.length) {
                    field = CACHE[value];
                    if (field == null) {
                        field = new YaField.IntegerYaField(value);
                    }
                } else {
                    field = new YaField.IntegerYaField(value);
                }
                doc.setField(outFieldIndexes[i], field);
            }
        }
    }
}

