package ru.yandex.msearch.collector.docprocessor;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import org.apache.lucene.util.StringHelper;
import ru.yandex.msearch.ProcessorRequestContext;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;
import ru.yandex.msearch.collector.postfilter.YaFieldToFloatPredicate;
import ru.yandex.msearch.collector.postfilter.YaFieldToLongPredicate;

public class FilterCompareDocProcessor implements DocProcessor {
    protected final Set<String> loadFields;

    protected final ProcessorRequestContext context;
    protected final Predicate<YaField> predicate;
    protected final String field;
    protected final int fieldIndex;
    protected long filtered = 0;

    public FilterCompareDocProcessor(
        final ProcessorRequestContext context,
        final Predicate<YaField> predicate,
        final String field)
    {
        this.context = context;
        this.predicate = predicate;

        this.field = StringHelper.intern(field);
        this.fieldIndex = context.fieldToIndex().indexFor(this.field);
        this.loadFields = Collections.singleton(this.field);
    }

    /**
     * FIXME This is dirty copy of postfilter
     * filter_cmp(field,operation,value)
     * agg_func concat,sum
     * @param args
     * @param context
     * @throws ParseException
     */
    public FilterCompareDocProcessor(
        final String args,
        final ProcessorRequestContext context)
        throws ParseException
    {
        this.context = context;
        String[] parts = args.split(",");
        if (parts.length != 3) {
            throw new ParseException("3 space separated parts expected", 0);
        }

        switch (parts[1]) {
            case "<=":
                Number upper = parse(parts[2]);
                if (upper instanceof Long) {
                    long value = upper.longValue();
                    predicate = new YaFieldToLongPredicate(x -> x <= value);
                } else {
                    float value = upper.floatValue();
                    predicate = new YaFieldToFloatPredicate(x -> x <= value);
                }
                break;
            case ">=":
                Number lower = parse(parts[2]);
                if (lower instanceof Long) {
                    long value = lower.longValue();
                    predicate = new YaFieldToLongPredicate(x -> x >= value);
                } else {
                    float value = lower.floatValue();
                    predicate = new YaFieldToFloatPredicate(x -> x >= value);
                }
                break;
            case "==":
                predicate = x -> parts[2].equals(Objects.toString(x, null));
                break;
            case "!=":
                predicate = x -> !parts[2].equals(Objects.toString(x, null));
                break;
            default:
                throw new ParseException("Unknown operation: " + parts[1], 0);
        }

        this.field = StringHelper.intern(parts[0]);
        this.fieldIndex = context.fieldToIndex().indexFor(this.field);
        this.loadFields = Collections.singleton(field);
    }

    private static Float parseFloat(final String value) throws ParseException {
        try {
            return Float.valueOf(value);
        } catch (NumberFormatException e) {
            ParseException ex =
                new ParseException("Can't parse numer from: " + value, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    private static Number parse(final String value) throws ParseException {
        if (value.indexOf('.') != -1) {
            return parseFloat(value);
        } else {
            try {
                return Long.valueOf(value);
            } catch (NumberFormatException e) {
                return parseFloat(value);
            }
        }
    }

    @Override
    public boolean processWithFilter(final YaDoc3 doc) throws IOException {
        boolean result = predicate.test(doc.getField(fieldIndex));
        filtered += result ? 0 : 1;
        return result;
    }

    @Override
    public void process(final YaDoc3 doc) throws IOException {
        throw new IOException("Unsupported method for filter docprocessor");
    }

    @Override
    public void apply(final ModuleFieldsAggregator aggregator) {
        aggregator.add(loadFields, Collections.emptySet());
    }

    @Override
    public void after() {
        context.ctx().logger().info("FilterDp, filtered:" + filtered);
    }
}
