package ru.yandex.msearch.collector.docprocessor;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.Weight;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;
import org.apache.lucene.util.Version;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.DefaultAnalyzerFactory;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.ProcessorRequestContext;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.YaField;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.util.string.StringUtils;

public class LeftJoinDocProcessor
    extends AbstractJoinDocProcessor
    implements DocProcessor
{
    private static final int CACHE_SIZE = 200;

    private static final String FORMAT =
        "left_join(field_join_on_left,field_join_on_right,extra_condition,"
            + "right_get_field1,right_get_field2 out_field1,out_field2)";
    private static final int FIRST_FIELD_INDEX = 3;

    protected final int joinLeftFieldIndex;
    protected final String extraQueryString;
    protected final DocProcessorQueryCache<YaField[]> cache;
    private final String joinFieldLeft;
    private final String joinFieldRight;
    private final Set<String> outFieldsSet;
    private final Set<String> getFieldsSet;
    private final Set<String> loadFieldsSet;
    private final String[] getFields;
    private final String[] outFields;
    private final String outScoreField;
    private final int outScoreFieldIndex;
    private final int[] outFieldsIndexes;
    private final int fieldsCount;
    private final FieldSelector fieldSelector;
    private PrefixingAnalyzerWrapper indexAnalyzer;
    //private final CachingDoubleAndQuery bq = new CachingDoubleAndQuery();
    protected Query extraQuery = null;
    private Selector extraQueryFields;
    private SimpleHashMapIndexPart localIndex;
    private int leftDocFound = 0;
    private long localSearchTime = 0;
    private long totalJoinWeightTime = 0;
    private long fastFirstDocs = 0;
    private long fastTotalTime = 0;
    private long extractTotalTime = 0L;
    int fstSearchSkipped = 0;
    int fstSearchScanned = 0;
    private enum ParseState {
        BEGIN,
        JOIN_ON_LEFT,
        JOIN_ON_RIGHT,
        EXTRA_CONDITION,
        GET_FIELD,
        OUT_FIELD
    }

    /**
     * left_join(field_join_on_left,field_join_on_right,extra_condition,get_field1+out_field1,get_field2+out_field2)
     * @param args
     * @param context
     * @throws ParseException
     */
    public LeftJoinDocProcessor(
        final String args,
        final ProcessorRequestContext context)
        throws ParseException
    {
        super(context);

        if (args == null || args.isEmpty()) {
            throw new ParseException("Arguments required", 0);
        }

        int commas = 0;
        for (int i = 0; i < args.length(); i++) {
            if (args.charAt(i) == ',') {
                commas += 1;
            }
        }

        if (commas < FIRST_FIELD_INDEX) {
            throw new ParseException(
                "Not enough arguments, format is:" + FORMAT, 0);
        }

        if (context.subQueryCacheSize() > 0) {
            this.cache = new DocProcessorQueryCache<>(CACHE_SIZE);
        } else {
            this.cache = null;
        }

        getFields = new String[commas - FIRST_FIELD_INDEX + 1];
        outFields = new String[commas - FIRST_FIELD_INDEX + 1];
        outFieldsIndexes = new int[commas - FIRST_FIELD_INDEX + 1];

        String leftJoinOn = null;
        String rightJoinOn = null;
        String extraCondition = null;

        int argStart = 0;
        boolean escape = false;
        int fieldsCnt = 0;
        ParseState state = ParseState.JOIN_ON_LEFT;
        String field;
        for (int i = 0; i < args.length(); i++) {
            char c = args.charAt(i);
            if (c == '\\') {
                escape = true;
                continue;
            }

            if (c == ' '
                && !escape
                && state == ParseState.GET_FIELD)
            {
                field =
                    StringHelper.intern(args.substring(argStart, i));
                getFields[fieldsCnt] = field;

                state = ParseState.OUT_FIELD;
                fieldsCnt += 1;
                argStart = i + 1;
            } else if (c == ',' && !escape) {
                switch (state) {
                    case JOIN_ON_LEFT:
                        leftJoinOn = args.substring(argStart, i);
                        state = ParseState.JOIN_ON_RIGHT;
                        break;
                    case JOIN_ON_RIGHT:
                        rightJoinOn = args.substring(argStart, i);
                        state = ParseState.EXTRA_CONDITION;
                        break;
                    case EXTRA_CONDITION:
                        extraCondition = args.substring(argStart, i);
                        state = ParseState.GET_FIELD;
                        break;
                    case GET_FIELD:
                        field =
                            StringHelper.intern(args.substring(argStart, i));
                        getFields[fieldsCnt] = field;
                        outFields[fieldsCnt] = field;
                        outFieldsIndexes[fieldsCnt] =
                            context.fieldToIndex().indexFor(field);

                        state = ParseState.GET_FIELD;
                        fieldsCnt += 1;
                        break;
                    case OUT_FIELD:
                        field =
                            StringHelper.intern(args.substring(argStart, i));
                        outFields[fieldsCnt - 1] = field;
                        outFieldsIndexes[fieldsCnt - 1] =
                            context.fieldToIndex().indexFor(field);

                        state = ParseState.GET_FIELD;
                        break;
                }

                argStart = i + 1;
            }

            escape = false;
        }

        if (argStart < args.length()) {
            switch (state) {
                case GET_FIELD:
                    field =
                        StringHelper.intern(args.substring(argStart, args.length()));
                    getFields[fieldsCnt] = field;
                    outFields[fieldsCnt] = field;
                    outFieldsIndexes[fieldsCnt] =
                        context.fieldToIndex().indexFor(field);

                    fieldsCnt += 1;
                case OUT_FIELD:
                    field =
                        StringHelper.intern(args.substring(argStart, args.length()));
                    outFields[fieldsCnt - 1] = field;
                    outFieldsIndexes[fieldsCnt - 1] =
                        context.fieldToIndex().indexFor(field);
                    break;
                default:
                    throw new ParseException("Invalid state " + state, 0);
            }
        }

        this.fieldsCount = fieldsCnt;
        this.extraQueryString = extraCondition;
        this.joinFieldLeft = StringHelper.intern(leftJoinOn);
        this.joinLeftFieldIndex =
            context.fieldToIndex().indexFor(joinFieldLeft);
        this.joinFieldRight = rightJoinOn;

        Set<String> outFieldsSet = new LinkedHashSet<>(outFields.length);
        for (int i = 0; i < fieldsCount; i++) {
            outFieldsSet.add(outFields[i]);
        }

        if (context.outScoreField() != null) {
            outScoreField = StringHelper.intern(context.outScoreField());
            outScoreFieldIndex = context.fieldToIndex().indexFor(outScoreField);
            outFieldsSet.add(outScoreField);
        } else {
            outScoreField = null;
            outScoreFieldIndex = -1;
        }

        this.outFieldsSet = Collections.unmodifiableSet(outFieldsSet);
        this.fieldSelector = new MapFieldSelector(getFields);
        this.getFieldsSet = new HashSet<>();
        for (int i = 0; i < getFields.length; i++) {
            getFieldsSet.add(getFields[i]);
        }

        if (multiprefix) {
            loadFieldsSet = new LinkedHashSet<>();
            loadFieldsSet.add(joinFieldLeft);
            loadFieldsSet.add(Config.PREFIX_FIELD_KEY);
        } else {
            loadFieldsSet = Collections.singleton(joinFieldLeft);
        }

        if (context.experimental()) {
            context.ctx().logger().info("Experimental join");
        }
    }

    public LeftJoinDocProcessor(
        final ProcessorRequestContext context,
        final String joinFieldLeft,
        final String joinFieldRight,
        final String extraQueryString,
        final String[] getFields,
        final String[] outFields,
        final String[] loadFields)
        throws ParseException
    {
        super(context);
        this.extraQueryString = extraQueryString;
        this.joinFieldLeft = joinFieldLeft;
        this.joinFieldRight = joinFieldRight;
        this.getFields = getFields;
        this.outFields = outFields;

        this.fieldsCount = outFields.length;
        this.joinLeftFieldIndex =
            context.fieldToIndex().indexFor(joinFieldLeft);

        this.outFieldsIndexes = new int[fieldsCount];
        Set<String> outFieldsSet = new LinkedHashSet<>(outFields.length);
        for (int i = 0; i < fieldsCount; i++) {
            outFieldsSet.add(outFields[i]);
            outFieldsIndexes[i] = context.fieldToIndex().indexFor(outFields[i]);
        }

        if (context.outScoreField() != null) {
            outScoreField = StringHelper.intern(context.outScoreField());
            outScoreFieldIndex = context.fieldToIndex().indexFor(outScoreField);
            outFieldsSet.add(outScoreField);
        } else {
            outScoreField = null;
            outScoreFieldIndex = -1;
        }

        this.outFieldsSet = Collections.unmodifiableSet(outFieldsSet);
        this.fieldSelector = new MapFieldSelector(getFields);
        this.getFieldsSet = new HashSet<>();
        for (int i = 0; i < getFields.length; i++) {
            getFieldsSet.add(getFields[i]);
        }

        loadFieldsSet = new LinkedHashSet<>();
        Collections.addAll(loadFieldsSet, loadFields);
        if (multiprefix) {
            loadFieldsSet.add(joinFieldLeft);
            loadFieldsSet.add(Config.PREFIX_FIELD_KEY);
        } else {
            loadFieldsSet.add(joinFieldLeft);
        }

        if (context.experimental()) {
            context.ctx().logger().info("Experimental join");
        }

        if (context.subQueryCacheSize() > 0) {
            this.cache = new DocProcessorQueryCache<>(CACHE_SIZE);
        } else {
            this.cache = null;
        }
    }

    @Override
    public Set<String> loadFields() {
        return getFieldsSet;
    }

    @Override
    public void apply(final ModuleFieldsAggregator aggregator) {
        aggregator.add(loadFieldsSet, outFieldsSet);
    }

    private class Selector implements FieldSelector {
        private final Set<String> stored;
        private final Map<String, String> storedAlias;
        private final Set<String> extraQueryFields;

        public Selector(
            final Set<String> extraQueryFields,
            final Set<String> stored,
            final Map<String, String> storedAlias)
        {
            this.stored = stored;
            this.storedAlias = storedAlias;
            this.extraQueryFields = extraQueryFields;
        }

        public Map<String, String> storedAlias() {
            return storedAlias;
        }

        public Set<String> extraQueryFields() {
            return extraQueryFields;
        }

        @Override
        public FieldSelectorResult accept(final String fieldName) {
            if (stored.contains(fieldName) || storedAlias.containsKey(fieldName)) {
                return FieldSelectorResult.LOAD;
            }

            if (outFieldsSet.contains(fieldName)) {
                return FieldSelectorResult.LAZY_LOAD;
            }

            return FieldSelectorResult.NO_LOAD;
        }

        public Set<String> stored() {
            return stored;
        }

        @Override
        public int maxFieldCount() {
            // unite , real less ?
            return storedAlias.size() + outFieldsSet.size() + stored.size();
        }
    }

    protected Map<String, YaField> extractDocExperimental(
        final String value)
        throws IOException, org.apache.lucene.queryParser.ParseException
    {
        if (extraQueryString.isEmpty()) {
            return extractDocNormal(value, extraQuery);
        }

        if (extraQuery == null) {
            YandexKeepFieldsQueryParser parser =
                new YandexKeepFieldsQueryParser(
                    Version.LUCENE_40,
                    null,
                    context.queryParser().getAnalyzer(),
                    context.index().config());
            parser.clearFields();
            extraQuery = parser.parse(extraQueryString);

            indexAnalyzer = DefaultAnalyzerFactory.INDEX.apply(context.index().config());
            if (context.isPrefixedAnalyzer()) {
                indexAnalyzer.setPrefix(context.prefixedAnalyzer().getPrefix());
            }

            Set<String> stored = new LinkedHashSet<>(parser.fields().size() << 1);
            Map<String, String> storedAlias = Collections.emptyMap();

            for (String field: parser.fields()) {
                FieldConfig config = context.index().config().fieldConfig(field);
                if (config.store()) {
                    stored.add(field);
                    continue;
                }

                List<FieldConfig> aliasChain =
                    context.index().config().aliasChain(field);
                config = null;
                if (aliasChain != null) {
                    for (FieldConfig fieldConfig: aliasChain) {
                        if (fieldConfig.store()) {
                            config = fieldConfig;
                            break;
                        }
                    }
                }

                if (config == null) {
                    context.ctx().logger().info(
                        "Fallback to regular, we have not stored field / alias chain in query, " + field);
                    return extractDocNormal(value, extraQuery);
                }

                if (storedAlias.isEmpty()) {
                    storedAlias = new LinkedHashMap<>(parser.fields().size() << 1);
                }

                stored.add(config.field());
                storedAlias.put(field, config.field());
            }

            if (context.debug()) {
                context.ctx().logger().info("extraQuery " + extraQuery);
                context.ctx().logger().info(
                "Stored: " + stored + " alias "
                    + storedAlias + " fields "
                    + parser.fields());
            }
            extraQueryFields = new Selector(parser.fields(), stored, storedAlias);
            if (context.debug()) {
                localIndex = new SimpleHashMapIndexPart(context.ctx().logger());
            } else {
                localIndex = new SimpleHashMapIndexPart();
            }
            //bq.second(extraQuery);
        }

        IndexSearcher indexSearcher = searcher.searcher();
        List<Document> docs = null;
        Query joinQuery = buildJoinQuery(value, joinFieldRight);
        if (joinQuery instanceof TermQuery) {
            docs = fastDiskSearch((TermQuery) joinQuery, indexSearcher);
        } else {
            long weightStartTs = System.nanoTime();
            Weight weight = joinQuery.weight(indexSearcher);
            totalJoinWeightTime += System.nanoTime() - weightStartTs;

            IndexReader.AtomicReaderContext[] leaves =
                indexSearcher.getTopReaderContext().leaves();
            if (leaves != null) {
                leaves = leaves.clone();
                Arrays.sort(
                    leaves,
                    (x, y) -> Long.compare(x.docBase, y.docBase));
                for (IndexReader.AtomicReaderContext context : leaves) {
                    DocIdSetIterator docIds =
                        weight.scorer(context, Weight.ScorerContext.def());

                    if (docIds != null) {
                        int docId = docIds.nextDoc();
                        if (docId == DocIdSetIterator.NO_MORE_DOCS) {
                            continue;
                        }

                        if (this.context.debug()) {
                            this.context.ctx().logger().info(
                                "Found docId " + (docId + context.docBase));
                        }

                        Document document = context.reader.document(docId, extraQueryFields);
                        if (!extraQueryFields.storedAlias().isEmpty()) {
                            for (Map.Entry<String, String> fieldPair :
                                extraQueryFields.storedAlias().entrySet()) {
                                String docValue = document.get(fieldPair.getValue());
                                if (docValue != null) {
                                    document.add(
                                        new Field(
                                            fieldPair.getKey(),
                                            docValue,
                                            Field.Store.YES,
                                            Field.Index.ANALYZED_NO_NORMS));
                                }
                            }
                        }

                        if (docs == null) {
                            docs = new ArrayList<>();
                        }
                        docs.add(document);
                    }
                }
            }
        }

        if (docs != null && !docs.isEmpty()) {
            long startLocal = System.nanoTime();
            Document[] docArray = new Document[docs.size()];
            docs.toArray(docArray);
            localIndex.setDocuments(
                indexAnalyzer,
                docArray,
                extraQueryFields.stored(),
                extraQueryFields.storedAlias());

            if (context.debug()) {
                this.context.ctx().logger().info("JoinValue " + value);
                this.context.ctx().logger().info("Local Index on " + value  + " " + localIndex.toString());
            }

            IndexSearcher searcher = new IndexSearcher(localIndex);
            Weight extraWeight = extraQuery.weight(searcher);
            IndexReader.AtomicReaderContext atomicReaderContext =
                localIndex.getTopReaderContext().leaves()[0];
            Scorer scorer =
                extraWeight.scorer(atomicReaderContext, Weight.ScorerContext.def());

            Document luceneDoc = null;
            if (scorer != null) {
                int docId = scorer.nextDoc();
                if (docId != DocIdSetIterator.NO_MORE_DOCS) {
                    luceneDoc = docArray[docId];
                }
            }

            if (luceneDoc != null) {
                if (context.debug()) {
                    this.context.ctx().logger().info(
                        "Doc found " + luceneDoc
                            + " score " + scorer.score());
                }
                Map<String, YaField> doc = new IdentityHashMap<>();
                for (String field: loadFields()) {
                    String dvalue = luceneDoc.get(field);
                    YaField yaValue;
                    if (dvalue != null) {
                        yaValue =
                            new YaField.StringYaField(
                                StringUtils.getUtf8Bytes(dvalue));
                    } else {
                        yaValue = null;
                    }
                    doc.put(field, yaValue);
                }

                if (outScoreField != null) {
                    doc.put(
                        outScoreField,
                        new YaField.FloatYaField(scorer.score()));
                }

                localSearchTime += System.nanoTime() - startLocal;
                return doc;
            }

            localSearchTime += System.nanoTime() - startLocal;
        }

        return Collections.emptyMap();
    }

    public List<Document> fastDiskSearch(
        final TermQuery query,
        final IndexSearcher searcher)
        throws IOException
    {
        long start = System.nanoTime();
        final Term keyTerm = query.getTerm();
        final String field = keyTerm.field();
        boolean needOnlyFirst = false;
        if (context.index().config().primaryKey().size() == 1
            && context.index().config().primaryKey().contains(field))
        {
            needOnlyFirst = true;
        }

        final BytesRef needle = keyTerm.bytes();

        IndexReader.AtomicReaderContext[] atomicReaders =
            searcher.getIndexReader().getTopReaderContext().leaves();

        ArrayList<Document> docs = null;
        BytesRef term = null;

        for (int i = atomicReaders.length - 1; i >= 0; i--) {
            IndexReader.AtomicReaderContext arc = atomicReaders[i];
        //for (IndexReader.AtomicReaderContext arc : atomicReaders) {

            IndexReader reader = arc.reader;

            Terms terms = reader.fields().terms(field);
            if (terms == null) {
                continue;
            }

            TermsEnum te = terms.getThreadTermsEnum(true);
            if (te == null) {
                continue;
            }
            try {
                DocsEnum de = null;
                int sr = te.seekExactEx(needle, true);
                if (sr == -1) {
                    fstSearchSkipped++;
                } else {
                    fstSearchScanned++;
                }
                if (sr < 1) {
                    continue;
                }
                Bits deletedDocs = reader.getDeletedDocs();
                term = te.term();
                if (term == null) {
                    continue;
                }
                de = te.docs(deletedDocs, de);
                int docId;
                while ((docId = de.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                    if (needOnlyFirst) {
                        fastTotalTime += System.nanoTime() - start;
                        return Collections.singletonList(reader.document(docId));
                    }

                    if (docs == null) {
                        fastFirstDocs += System.nanoTime() - start;
                        docs = new ArrayList<>(1);
                    }

                    docs.add(reader.document(docId));
                }
            } finally {
                te.close();
            }
        }
//        if (logger.isLoggable(Level.FINEST)) {
//            logger.finest("key: <" + needle.utf8ToString()
//                + ">: scans: " + scanned
//                + ", skips: " + skipped);
//        }

        fastTotalTime += System.nanoTime() - start;
//        if (docs == null) {
//            return Collections.emptyList();
//        }
        return docs;
    }

    protected Map<String, YaField> extractDocNormal(
        final String value)
        throws IOException, org.apache.lucene.queryParser.ParseException
    {
        if (!extraQueryString.isEmpty() && extraQuery == null) {
            extraQuery = context.queryParser().parse(extraQueryString);
        }

        return extractDocNormal(value, extraQuery);
    }

    protected Map<String, YaField> extractDocNormal(
        final String value,
        final Query extraQuery)
        throws IOException, org.apache.lucene.queryParser.ParseException
    {
        Query joinQuery = buildJoinQuery(value, joinFieldRight);

        if (extraQuery != null) {
            BooleanQuery bq = new BooleanQuery();
            bq.add(joinQuery, BooleanClause.Occur.MUST);
            bq.add(extraQuery, BooleanClause.Occur.MUST);
            //bq.first(joinQuery);
            joinQuery = bq;
        }

        return extractDoc(joinQuery, fieldSelector);
    }

    protected Map<String, YaField> extractDoc(final String value)
        throws IOException, org.apache.lucene.queryParser.ParseException
    {
        leftDocFound += 1;
        if (context.experimental()) {
            return extractDocExperimental(value);
        } else {
            return extractDocNormal(value);
        }
    }

    @Override
    public boolean processWithFilter(final YaDoc3 doc) throws IOException {
        if (multiprefix) {
            if (!multiprefixSearcherUpdate(doc)) {
                return context.keepRightNull();
            }
        }

        YaField field = doc.getField(joinLeftFieldIndex);
        if (field == null) {
            if (context.debug()) {
                context.ctx().logger().warning(
                    "Left Field not found in doc " + doc);
            }
            return context.keepRightNull();
        }

        String value = field.toString();
        YaField[] yaFields = null;
        if (cache != null) {
            yaFields = cache.get(value);
        }

        if (yaFields == null) {
            try {
                long startTs = System.nanoTime();
                Map<String, YaField> document = extractDoc(value);
                extractTotalTime += System.nanoTime() - startTs;
                if (document == EMPTY_DOC && !context.keepRightNull()) {
                    return false;
                }

                if (outScoreField != null) {
                    yaFields = new YaField[fieldsCount + 1];
                    YaField sdata = document.get(outScoreField);
                    if (sdata != null) {
                        yaFields[fieldsCount] = sdata;
                    }

                    doc.setField(outScoreFieldIndex, sdata);
                } else {
                    yaFields = new YaField[fieldsCount];
                }

                for (int i = 0; i < fieldsCount; i++) {
                    YaField sdata = document.get(getFields[i]);

                    if (sdata != null) {
                        yaFields[i] = sdata;
                        doc.setField(outFieldsIndexes[i], sdata);
                    }
                }

//                extractDoc(value);
                if (cache != null) {
                    cache.put(value, yaFields);
                }
            } catch (org.apache.lucene.queryParser.ParseException pe) {
                context.ctx().logger().log(
                    Level.WARNING,
                    "Subquery failed " + extraQueryString,
                    pe);
                throw new IOException("Bad query " + extraQueryString, pe);
            }
        } else {
            for (int i = 0; i < fieldsCount; i++) {
                if (yaFields[i] != null) {
                    doc.setField(outFieldsIndexes[i], yaFields[i]);
                }
            }

            if (outScoreField != null && yaFields[fieldsCount] != null) {
                doc.setField(outScoreFieldIndex, yaFields[fieldsCount]);
            }
        }

        return true;
    }

    @Override
    public void after() {
        super.after();
        context.ctx().logger().info("ExtractTotalTime: " + extractTotalTime);
        context.ctx().logger().info("leftDocFound: " + leftDocFound);

        if (context.experimental()) {
            context.ctx().logger().info("localSearchTime: " + localSearchTime);
            context.ctx().logger().info("totalJoinWeightTime: " + totalJoinWeightTime);
            context.ctx().logger().info("fastFirstDocs: " + fastFirstDocs);
            context.ctx().logger().info("fastAllDocs: " + fastTotalTime);
            context.ctx().logger().info("fstScanned: " + fstSearchScanned);
            context.ctx().logger().info("fstSkipped: " + fstSearchSkipped);
        }
    }
}
