package ru.yandex.msearch.collector.docprocessor;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.TermToBytesRefAttribute;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.FieldVisitor;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.FieldsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.TermFreqVector;
import org.apache.lucene.index.TermVectorMapper;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.DefaultAnalyzerFactory;
import ru.yandex.msearch.PrefixingAnalyzerWrapper;
import ru.yandex.msearch.config.DatabaseConfig;

/**
 * Allows to index document in structure in memory to apply search query
 * Not thread safe as MemoryIndex. Simple, so use it for tests or experiments
 *
 */
public class SimpleHashMapIndexPart extends IndexReader {
    private final Map<String, Terms> fieldTermsMap;
    private final FakeFields fields;

    private final ReaderContext readerContext;
    private Document[] documents;
    private Analyzer analyzer;
    private Logger logger;

    public SimpleHashMapIndexPart(final Logger logger) {
        this.readerContext = new TopLevelReaaderContext();
        this.fieldTermsMap = new LinkedHashMap<>();
        this.fields = new FakeFields();
        this.documents = new Document[0];
        this.logger = logger;
    }
    public SimpleHashMapIndexPart() {
        this(null);
    }

    public SimpleHashMapIndexPart(
        final DatabaseConfig config,
        final Document[] documents,
        final Set<String> storedFields,
        final Map<String, String> aliases)
    {
        this(DefaultAnalyzerFactory.INDEX.apply(config), documents, storedFields, aliases);
    }

    public SimpleHashMapIndexPart(
        final PrefixingAnalyzerWrapper analyzer,
        final Document[] documents,
        final Set<String> storedFields,
        final Map<String, String> aliases)
    {
        this();

        this.setDocuments(analyzer, documents, storedFields, aliases);
    }

    /**
     *
     * @param analyzer
     * @param documents
     * @param storedFields - list of stored fields
     * @param aliases - map (indexed, stored) fields, due to aliases things
     */
    public void setDocuments(
        final PrefixingAnalyzerWrapper analyzer,
        final Document[] documents,
        final Collection<String> storedFields,
        final Map<String, String> aliases)
    {
        this.analyzer = analyzer;
        this.fieldTermsMap.clear();
        this.documents = documents;
        if (logger != null) {
            logger.info("Indexing to local index docs:" + documents.length + " analyzer prefix: " + analyzer.getPrefix());
        }
        Map<String, Map<BytesRef, List<Document>>> index = new LinkedHashMap<>();
        for (Document document: documents) {
            if (storedFields == null && aliases == null) {
                for (Fieldable fieldable: document.getFields()) {
                    if (fieldable != null) {
                        add(fieldable.name(), fieldable, document, index);
                    }
                }
            } else {
                if (storedFields != null) {
                    for (String field: storedFields) {
                        Fieldable value = document.getFieldable(field);
                        if (value != null) {
                            add(field, value, document, index);
                        }
                    }
                }

                if (aliases != null) {
                    for (Map.Entry<String, String> entry: aliases.entrySet()) {
                        Fieldable value = document.getFieldable(entry.getValue());
                        if (value != null) {
                            add(entry.getKey(), value, document, index);
                        }
                    }
                }
            }
        }

        for (Map.Entry<String, Map<BytesRef, List<Document>>> entry: index.entrySet()) {
            fieldTermsMap.put(entry.getKey(), new FakeTerms(new FakeTermsEnum(entry.getKey(), entry.getValue())));
        }
    }

    @Override
    public String toString() {
        return "SimpleHashMapIndexPart{" +
            "fieldTermsMap=" + fieldTermsMap.toString() +
            '}';
    }

    private void add(
        final String name,
        final Fieldable value,
        final Document document,
        final Map<String, Map<BytesRef, List<Document>>> index)
    {
        Reader reader = value.readerValue();
        if (reader == null) {
            reader = new StringReader(value.stringValue());
        }
        try {
            TokenStream stream = analyzer.tokenStream(name, reader);
            if (stream != null) {
                boolean hasMoreTokens = false;
                stream.reset();
                final TermToBytesRefAttribute termAtt =
                    stream.getAttribute(TermToBytesRefAttribute.class);

                hasMoreTokens = stream.incrementToken();
                while (hasMoreTokens) {
                    BytesRef bytes = new BytesRef();
                    termAtt.toBytesRef(bytes);
                    index.computeIfAbsent(name, (x) -> new LinkedHashMap<>())
                        .computeIfAbsent(bytes, (x) -> new ArrayList<>()).add(document);
                    hasMoreTokens = stream.incrementToken();
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public TermFreqVector[] getTermFreqVectors(final int docNumber) throws IOException {
        return new TermFreqVector[0];
    }

    @Override
    public TermFreqVector getTermFreqVector(final int docNumber, final String field) throws IOException {
        return null;
    }

    @Override
    public void getTermFreqVector(final int docNumber, final String field, final TermVectorMapper mapper) throws IOException {

    }

    @Override
    public void getTermFreqVector(final int docNumber, final TermVectorMapper mapper) throws IOException {

    }

    @Override
    public int numDocs() {
        return documents.length;
    }

    @Override
    public int maxDoc() {
        return documents.length;
    }

    @Override
    public void readDocument(final int n, final FieldVisitor visitor) throws CorruptIndexException, IOException {
    }

    @Override
    public Document document(final int n, final FieldSelector fieldSelector) throws CorruptIndexException, IOException {
        return documents[n];
    }

    @Override
    public boolean hasDeletions() {
        return false;
    }

    @Override
    public byte[] norms(final String field) throws IOException {
        return null;
    }

    @Override
    protected void doSetNorm(final int doc, final String field, final byte value) throws CorruptIndexException, IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    public Fields fields() throws IOException {
        return fields;
    }

    @Override
    protected void doDelete(final int docNum) throws CorruptIndexException, IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    protected void doUndeleteAll() throws CorruptIndexException, IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    protected void doCommit(final Map<String, String> commitUserData) throws IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    protected void doClose() throws IOException {
    }

    @Override
    public Collection<String> getFieldNames(final FieldOption fldOption) {
        return fieldTermsMap.keySet();
    }

    @Override
    public Bits getDeletedDocs() {
        return null;
    }

    @Override
    public ReaderContext getTopReaderContext() {
        return readerContext;
    }

    private class FakeFields extends Fields {
        @Override
        public FieldsEnum iterator() throws IOException {
            return new FieldsIterator();
        }

        @Override
        public Terms terms(final String field) throws IOException {
            return fieldTermsMap.get(field);
        }
    }

    public class TopLevelReaaderContext extends ReaderContext {
        private final AtomicReaderContext[] atomicReaderContext;

        public TopLevelReaaderContext() {
            super(null, SimpleHashMapIndexPart.this, true, 0, 0);

            atomicReaderContext = new AtomicReaderContext[1];
            atomicReaderContext[0] = new AtomicReaderContext(this, SimpleHashMapIndexPart.this, 0, 0, 0, 0);
        }

        @Override
        public AtomicReaderContext[] leaves() {
            return atomicReaderContext;
        }
    }

    private final class FieldsIterator extends FieldsEnum {
        private final Iterator<String> it;
        private String current;

        public FieldsIterator() {
            it = fieldTermsMap.keySet().iterator();
        }

        @Override
        public String next() {
            if (it.hasNext()) {
                current = it.next();
            } else {
                current = null;
            }

            return current;
        }

        @Override
        public TermsEnum terms(final boolean buffered) throws IOException {
            Terms terms = fieldTermsMap.get(current);
            if (terms != null) {
                return terms.iterator(buffered);
            } else {
                return TermsEnum.EMPTY;
            }
        }
    }

    private static class FakeTerms extends Terms {
        private final TermsEnum termsEnum;

        public FakeTerms(final TermsEnum termsEnum) {
            this.termsEnum = termsEnum;
        }

        @Override
        public TermsEnum iterator(final boolean buffered) throws IOException {
            return termsEnum;
        }

        @Override
        public Comparator<BytesRef> getComparator() throws IOException {
            return BytesRef.getUTF8SortedAsUTF16Comparator();
        }

        @Override
        public long getSumTotalTermFreq() throws IOException {
            return 0;
        }

        @Override
        public String toString() {
            return termsEnum.toString();
        }
    }

    private class FakeTermsEnum extends TermsEnum {
        private final BytesRef[] array;
        private final String name;
        private int index = 0;
        private final Map<BytesRef, List<Document>> map;

        public FakeTermsEnum(final String name, final Map<BytesRef, List<Document>> docs) {
            this.map = docs;
            this.array = new BytesRef[docs.size()];
            docs.keySet().toArray(array);
            Arrays.sort(array, getComparator());
            this.name = name;
            if (logger != null) {
                logger.info("TermsEnum for  " + name + " array " + Arrays.toString(array));
            }
        }

        @Override
        public SeekStatus seek(final BytesRef text, final boolean useCache) throws IOException {
            int insIndex = Arrays.binarySearch(array, text, getComparator());
            if (insIndex < 0) {
                index = -insIndex - 1;
                if (index < array.length) {
                    if (logger != null) {
                        logger.info("Seek field " + name + " text " + text + " NOT FOUND " + index);
                    }
                    return SeekStatus.NOT_FOUND;
                } else {
                    if (logger != null) {
                        logger.info("Seek field " + name + " text " + text +  " END " + index);
                    }
                    return SeekStatus.END;
                }
            } else {
                index = insIndex;
            }

            if (logger != null) {
                logger.info("Seek field " + name + " text " + text + " FOUND " + index);
            }

            return SeekStatus.FOUND;
        }

        @Override
        public SeekStatus seek(final long ord) throws IOException {
            index = (int) ord;
            return SeekStatus.FOUND;
        }

        @Override
        public BytesRef next() throws IOException {
            if (index + 1 < array.length) {
                //if (logger != null) {
                //logger.info("Next on field " + name + " index " + index + " FOUND " + array[index + 1]);
                //}
                return array[++index];
            }

            //if (logger != null) {
            //logger.info("Next on field " + name + " index " + index + " NOT FOUND ");
            //}
            return null;
        }

        @Override
        public BytesRef term() throws IOException {
            if (index < array.length) {
//                if (logger != null) {
//                    logger.info("Term " + index + " FOUND " + array[index]);
//                }

                return array[index];
            }

//            if (logger != null) {
//                logger.info("Term " + index + " NOT FOUND ");
//            }

            return null;
        }

        @Override
        public long ord() throws IOException {
            return index;
        }

        @Override
        public int docFreq() throws IOException {
            return 0;
        }

        @Override
        public long totalTermFreq() throws IOException {
            return 0;
        }

        @Override
        public DocsEnum docs(final Bits skipDocs, final DocsEnum reuse) throws IOException {
            return new FakeDocsEnum(map.get(array[index]));
        }

        @Override
        public DocsAndPositionsEnum docsAndPositions(final Bits skipDocs, final DocsAndPositionsEnum reuse) throws IOException {
            return new FakeDocsAndPositionsEnum(map.get(array[index]));
        }

        @Override
        public Comparator<BytesRef> getComparator() {
            return BytesRef.getUTF8SortedAsUTF16Comparator();
        }

        @Override
        public void close() throws IOException {
            index = 0;
        }

        @Override
        public String toString() {
            return map.toString();
        }
    }

    private static class FakeDocsAndPositionsEnum extends DocsAndPositionsEnum {
        private final List<Document> documents;
        private int index = 0;

        public FakeDocsAndPositionsEnum(final List<Document> documents) {
            this.documents = documents;
        }

        @Override
        public int freq() {
            return 0;
        }

        @Override
        public int docID() {
            return index;
        }

        @Override
        public int nextDoc() throws IOException {
            if (index >= documents.size()) {
                return NO_MORE_DOCS;
            }

            return index++;
        }

        @Override
        public int advance(final int target) throws IOException {
            return target;
        }

        @Override
        public int nextPosition() throws IOException {
            return 0;
        }

        @Override
        public BytesRef getPayload() throws IOException {
            return null;
        }

        @Override
        public boolean hasPayload() {
            return false;
        }
    }

    private static class FakeDocsEnum extends DocsEnum {
        private final List<Document> documents;
        private int index = 0;

        public FakeDocsEnum(final List<Document> documents) {
            this.documents = documents;
        }

        @Override
        public int freq() {
            return 0;
        }

        @Override
        public int docID() {
            return index;
        }

        @Override
        public int nextDoc() throws IOException {
            if (index >= documents.size()) {
                return NO_MORE_DOCS;
            }

            return index++;
        }

        @Override
        public int advance(final int target) throws IOException {
            return target;
        }
    }
}
