package ru.yandex.msearch.collector.postfilter;

import java.text.ParseException;

import java.util.Objects;
import java.util.function.Predicate;

import ru.yandex.function.GenericBiFunction;

import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.YaField;

public enum PostFilterFactory
    implements GenericBiFunction<
        String,
        FieldToIndex,
        PostFilter,
        ParseException>
{
    INSTANCE;

    private static Float parseFloat(final String value) throws ParseException {
        try {
            return Float.valueOf(value);
        } catch (NumberFormatException e) {
            ParseException ex =
                new ParseException("Can't parse numer from: " + value, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    private static Number parse(final String value) throws ParseException {
        if (value.indexOf('.') != -1) {
            return parseFloat(value);
        } else {
            try {
                return Long.valueOf(value);
            } catch (NumberFormatException e) {
                return parseFloat(value);
            }
        }
    }

    @Override
    public PostFilter apply(
        final String condition,
        final FieldToIndex fieldToIndex)
        throws ParseException
    {
        if (condition.isEmpty()) {
            return NullPostFilter.INSTANCE;
        }
        String[] parts = condition.split(" ");
        if (parts.length != 3) {
            throw new ParseException("3 space separated parts expected", 0);
        }

        Predicate<YaField> predicate;
        switch (parts[1]) {
            case "<=":
                Number upper = parse(parts[2]);
                if (upper instanceof Long) {
                    long value = upper.longValue();
                    predicate = new YaFieldToLongPredicate(x -> x <= value);
                } else {
                    float value = upper.floatValue();
                    predicate = new YaFieldToFloatPredicate(x -> x <= value);
                }
                break;
            case ">=":
                Number lower = parse(parts[2]);
                if (lower instanceof Long) {
                    long value = lower.longValue();
                    predicate = new YaFieldToLongPredicate(x -> x >= value);
                } else {
                    float value = lower.floatValue();
                    predicate = new YaFieldToFloatPredicate(x -> x >= value);
                }
                break;
            case "==":
                predicate = x -> parts[2].equals(Objects.toString(x, null));
                break;
            case "!=":
                predicate = x -> !parts[2].equals(Objects.toString(x, null));
                break;
            case "equals":
                return new EqualFieldsFilter(parts[0], parts[2], fieldToIndex);
            case "notequals":
                return new NotEqualFieldsFilter(
                    parts[0],
                    parts[2],
                    fieldToIndex);
            case "greaterthan":
                return new GreaterThanFieldsFilter(
                    parts[0],
                    parts[2],
                    fieldToIndex);
            case "lessthan":
                return new LessThanFieldsFilter(
                    parts[0],
                    parts[2],
                    fieldToIndex);
            case "in":
                return new InSetFieldsFilter(
                    parts[0],
                    fieldToIndex,
                    parts[2].split(","));
            default:
                throw new ParseException("Unknown operation: " + parts[1], 0);
        }
        return new SimplePostFilter(parts[0], predicate, fieldToIndex);
    }
}

