package ru.yandex.msearch.config;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.lucene.index.codecs.yandex.YandexCodec;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.util.StringHelper;

import ru.yandex.analyzer.TokenizerType;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.FieldsConfig;
import ru.yandex.msearch.UnsharingPrefixParser;
import ru.yandex.msearch.collector.docprocessor.DefaultDocProcessorFactory;
import ru.yandex.msearch.collector.docprocessor.DocProcessorFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.IntegerMemorySizeParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NotLessDoubleValidator;
import ru.yandex.parser.string.NotLessIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.search.prefix.PrefixParser;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.stater.SimpleHistogramMetric;

public class DatabaseConfigBuilder implements DatabaseConfig {
    private static long DEFAULT_PREFIX_ACTIVITY_TIMEOUT =
        30L * 86400L * 1000L;
    private int indexThreads;
    private long maxMemDocs;
    private double reclaimDeletesWeight;
    private double autoExpungePct;
    private int indexDivisor;
    private int shards;
    private int inMemoryParsingLimit;
    private boolean useJournal;
    private boolean autoFixIndex;
    private boolean lowercaseExpandedTerms;
    private final int yandexCodecCompressorLevel;
    private File indexPath;
    private PrefixParser prefixParser;
    private PrefixParser indexPrefixParser;
    private QueryParser.Operator defaultOperator;
    private Set<String> primaryKey;
    private int yandexPostingsWriterBlockSize;
    private int yandexTermsWriterBlockSize;
    private Set<String> yandexCodecGroupFields;
    private int yandexFieldsWriterBufferSize;
    private int memoryCodecCompressorLevel;
    private boolean primaryKeyPartCacheDocuments;
    private boolean prefixedPrimaryKey;
    private int partsExecutorThreadCount;
    private long maxPartJournalSize;
    private int limitIndexRequests;
    private boolean logParsedRequest;
    private int maxMergeThreads;
    private int indexCopyRateLimitMb;
    private boolean checkCopyness;
    private boolean useFastCommitCodec;
    private boolean autoPruning;
    private int fieldIndexReadBufferSize;
    private boolean inMemoryFieldsIndex;
    private String defaultFieldCodec;
    private boolean queueIdServiceFallback;
    private int warmerThreads;
    private boolean returnMaxQueueId;
    private boolean fakeQueueIdsPush;
    private int commonTaskExecutorThreads;
    private int segmentsPerTier;
    private long maxSegmentSize;
    private boolean copyIndexUpdateQueueIdsOnEmptyDump;
    private boolean updatePrefixActivity;
    private long prefixActivityTimeout;
    private int partFlushThreshold;
    private boolean warmOnInit;
    private DocProcessorFactory docProcessorFactory;
    private boolean syncSearcherDefault;
    private boolean earlyInterruptDefault;
    private SimpleHistogramMetric indexLagHistogram;
    private boolean oDirectWrite;
    private Set<String> services;

    private String name;
    private FieldsConfig fieldsConfig;

    private boolean orderIndependentUpdate;

    public DatabaseConfigBuilder(final IniConfig config, final String name) throws ConfigException {
        this.name = name;
        syncSearcherDefault =
            config.getBoolean(
                "sync-searcher-default",
                true);
        earlyInterruptDefault =
            config.getBoolean(
                "early-interrupt-default",
                false);
        oDirectWrite = config.getBoolean(
            "odirect-write",
            false);
        warmOnInit = config.getBoolean(
            "warm-on-init",
            false);
        partFlushThreshold = config.getInt(
            "part-flush-threshold",
            150000);
        updatePrefixActivity = config.getBoolean(
            "update-prefix-activity",
            false);
        prefixActivityTimeout = config.getLongDuration(
            "prefix-activity-timeout",
            DEFAULT_PREFIX_ACTIVITY_TIMEOUT);

        services = config.get(
            "services",
            Collections.emptySet(),
            new CollectionParser<>(String::trim, LinkedHashSet::new, ','));
        segmentsPerTier = config.get(
            "segments-per-tier",
            10,
            new NotLessIntegerValidator(2));
        maxSegmentSize = config.get("max-segment-size",
            5 * 1024 * 1024 * 1024L,
            LongMemorySizeParser.INSTANCE);
        commonTaskExecutorThreads = config.get(
            "common-task-executor-threads",
            Runtime.getRuntime().availableProcessors(),
            PositiveIntegerValidator.INSTANCE);
        returnMaxQueueId = config.getBoolean("return-max-queue-id", false);
        fakeQueueIdsPush = config.getBoolean("fake-queue-ids-push", true);
        useFastCommitCodec = config.getBoolean("use-fast-commit-codec", true);
        checkCopyness = config.getBoolean("check-copyness", false);
        indexThreads = config.get("index_threads", 10,
            PositiveIntegerValidator.INSTANCE);
        warmerThreads = config.get("warmer-threads", indexThreads,
            PositiveIntegerValidator.INSTANCE);
        partsExecutorThreadCount = config.get("parts_threads", indexThreads,
            new NotLessIntegerValidator(0));
        maxMemDocs = config.get("maxmemdocs", 2000000L,
            PositiveLongValidator.INSTANCE);

        maxMergeThreads = config.get(
            "max-merge-threads",
            Runtime.getRuntime().availableProcessors(),
            new NotLessIntegerValidator(1));
        reclaimDeletesWeight = config.get("reclaim_deletes_weight", 8.0,
            new NotLessDoubleValidator(4));
        autoExpungePct = config.get("auto_expunge_pct", 5.0,
            new NotLessDoubleValidator(0));
        indexDivisor = config.get("index_divisor", 1,
            PositiveIntegerValidator.INSTANCE);
        shards = config.get("shards", 1000, PositiveIntegerValidator.INSTANCE);
        inMemoryParsingLimit = config.get(
            "in_memory_parsing_limit",
            131072,
            NonNegativeIntegerValidator.INSTANCE);
        useJournal = config.getBoolean("use_journal", true);
        maxPartJournalSize = config.get("max_part_journal_size_mb", 128,
            new NotLessIntegerValidator(1)) * 1024 * 1024;
        indexCopyRateLimitMb = config.get("index_copy_rate_limit_mb", 128,
            new NotLessIntegerValidator(1));
        lowercaseExpandedTerms =
            config.getBoolean("lowercase_expanded_terms", false);
        queueIdServiceFallback =
            config.getBoolean("queueid-service-fallback", false);
        indexPath = config.getDir("index_path", new File("/u0/index/"));
        logParsedRequest = config.getBoolean("log_parsed_request", true);
        defaultOperator = config.getEnum(QueryParser.Operator.class,
            "default_operator", QueryParser.Operator.OR);

        yandexCodecCompressorLevel = config.get(
            "yandex_codec.compressor-level",
            9,
            PositiveIntegerValidator.INSTANCE);
        limitIndexRequests = config.section("indexer").get(
            "limit_index_requests", 0, NonNegativeIntegerValidator.INSTANCE);

        memoryCodecCompressorLevel = config.get(
            "yandex_codec.memory-compressor-level",
            1,
            PositiveIntegerValidator.INSTANCE);
        yandexPostingsWriterBlockSize = config.get("yandex_codec.postings_writer_block_size", 4096,
            PositiveIntegerValidator.INSTANCE);
        yandexTermsWriterBlockSize = config.get("yandex_codec.terms_writer_block_size", 4096,
            PositiveIntegerValidator.INSTANCE);
        yandexFieldsWriterBufferSize = config.get("yandex_codec.fields_writer_buffer_size", 6*1024,
            PositiveIntegerValidator.INSTANCE);
        fieldIndexReadBufferSize =
            config.get(
                "yandex_codec.field-index-read-buffer-size",
                YandexCodec.DEFAULT_FIELD_INDEX_READ_BUFFER_SIZE,
                IntegerMemorySizeParser.INSTANCE);
        defaultFieldCodec = config.getString("default-field-codec", "Yandex");

        inMemoryFieldsIndex =
            config.getBoolean(
                "yandex_codec.in-memory-fields-index",
                YandexCodec.DEFAULT_IN_MEMORY_FIELDS_INDEX);

        yandexCodecGroupFields =
            config.get(
                "yandex_codec.group_fields",
                new LinkedHashSet(),
                new CollectionParser<>(String::trim, LinkedHashSet::new, ','));
        if (yandexCodecGroupFields.size() == 0) {
            yandexCodecGroupFields.add(
                config.get(
                    "yandex_codec.group_field",
                    Config.PREFIX_FIELD_KEY,
                    NonEmptyValidator.INSTANCE));
        }
        // will always have the prefix field

        FieldsConfig fieldsConfig;
        try {
            fieldsConfig = new FieldsConfig(config, indexDivisor);
        } catch (IOException ioe) {
            throw new ConfigException(ioe.getMessage());
        }

        this.fieldsConfig = fieldsConfig;
        String primaryKeyString = config.getString("primary_key", null);
        if (primaryKeyString == null) {
            primaryKey = null;
            prefixedPrimaryKey = false;
            autoFixIndex = false;
            primaryKeyPartCacheDocuments = false;
        } else {
            Set<String> primaryKey = new LinkedHashSet<>();
            boolean prefixed = false;
            for (String key: primaryKeyString.split("[|,]")) {
                String trimmed = key.trim();
                if (!trimmed.isEmpty()) {
                    FieldConfig fieldConfig = fieldsConfig.fields().get(trimmed);
                    if (fieldConfig == null) {
                        throw new ConfigException("Field " + trimmed
                            + " is a part of primary key, but it is not "
                            + "defined in config");
                    }
                    if (!fieldConfig.index()) {
                        throw new ConfigException("Field " + trimmed
                            + " is a part of primary key, but it is not "
                            + "indexed");
                    }
                    if (fieldConfig.tokenizer() != TokenizerType.KEYWORD) {
                        throw new ConfigException("Field " + trimmed
                            + " is a part of primary key, "
                            + "it must have 'keyword' tokenizer, not '"
                            + fieldConfig.tokenizer().toString().toLowerCase()
                            + "'");

                    }
                    if (fieldConfig.searchFilters().size() > 0 || fieldConfig.indexFilters().size() > 0) {
                        throw new ConfigException("Field " + trimmed
                            + " is a part of primary key, "
                            + "it shouldn't have any filters");
                    }
                    if (!fieldConfig.store()) {
                        throw new ConfigException("Field " + trimmed
                            + " is a part o primary key, "
                            + "it should be stored");
                    }
                    if (fieldConfig.prefixed()) {
                        prefixed = true;
                    }
                    primaryKey.add(StringHelper.intern(trimmed));
                }
            }
            prefixedPrimaryKey = prefixed;
            if (primaryKey.isEmpty()) {
                this.primaryKey = null;
                autoFixIndex = false;
            } else {
                this.primaryKey = Collections.unmodifiableSet(primaryKey);
                autoFixIndex = config.getBoolean("auto_fix_index", false);
            }
            primaryKeyPartCacheDocuments =
                config.getBoolean("primary_key_part_cache_documents", false);
        }
        prefixParser =
            config.getEnum(PrefixType.class, "prefix_type", PrefixType.LONG);
        if (prefixParser == PrefixType.STRING
            && !primaryKeyPartCacheDocuments)
        {
            indexPrefixParser = new UnsharingPrefixParser(prefixParser);
        } else {
            indexPrefixParser = prefixParser;
        }

        autoPruning = config.getBoolean("auto-pruning", false);

        copyIndexUpdateQueueIdsOnEmptyDump =
            config.getBoolean(
                "update-queueid-on-empty-dumpindex",
                false);

        orderIndependentUpdate = config.getBoolean("order-independent-update", false);

        String docProcessorFactoryName = config.getString("doc-processor-factory", null);
        if (docProcessorFactoryName == null) {
            docProcessorFactory = new DefaultDocProcessorFactory();
        } else {
            try {
                Class<? extends DocProcessorFactory> factoryClass =
                    Class.forName(docProcessorFactoryName).asSubclass(DocProcessorFactory.class);
                docProcessorFactory = factoryClass.getConstructor().newInstance();
            } catch (ClassNotFoundException | NoSuchMethodException
                | InstantiationException | IllegalAccessException | InvocationTargetException e) {
                throw new ConfigException(
                    "Unable to load doc processor factory " + docProcessorFactoryName, e);
            }
        }

        IniConfig statLagConfig = config.sectionOrNull("index-stat-lag");
        if (statLagConfig != null) {
            indexLagHistogram = new SimpleHistogramMetric(statLagConfig);
        } else {
            indexLagHistogram = null;
        }

        config.checkUnusedKeys();
    }


    @Override
    public int indexThreads() {
        return indexThreads;
    }

    public DatabaseConfigBuilder indexThreads(final int indexThreads) {
        this.indexThreads = indexThreads;
        return this;
    }

    @Override
    public long maxMemDocs() {
        return maxMemDocs;
    }

    public DatabaseConfigBuilder maxMemDocs(final long maxMemDocs) {
        this.maxMemDocs = maxMemDocs;
        return this;
    }

    @Override
    public double reclaimDeletesWeight() {
        return reclaimDeletesWeight;
    }

    public DatabaseConfigBuilder reclaimDeletesWeight(final double reclaimDeletesWeight) {
        this.reclaimDeletesWeight = reclaimDeletesWeight;
        return this;
    }

    @Override
    public double autoExpungePct() {
        return autoExpungePct;
    }

    public DatabaseConfigBuilder autoExpungePct(final double autoExpungePct) {
        this.autoExpungePct = autoExpungePct;
        return this;
    }

    @Override
    public int indexDivisor() {
        return indexDivisor;
    }

    public DatabaseConfigBuilder indexDivisor(final int indexDivisor) {
        this.indexDivisor = indexDivisor;
        return this;
    }

    @Override
    public int shards() {
        return shards;
    }

    public DatabaseConfigBuilder shards(final int shards) {
        this.shards = shards;
        return this;
    }

    @Override
    public int inMemoryParsingLimit() {
        return inMemoryParsingLimit;
    }

    public DatabaseConfigBuilder inMemoryParsingLimit(final int inMemoryParsingLimit) {
        this.inMemoryParsingLimit = inMemoryParsingLimit;
        return this;
    }

    @Override
    public boolean useJournal() {
        return useJournal;
    }

    public DatabaseConfigBuilder useJournal(final boolean useJournal) {
        this.useJournal = useJournal;
        return this;
    }

    @Override
    public boolean autoFixIndex() {
        return autoFixIndex;
    }

    public DatabaseConfigBuilder autoFixIndex(final boolean autoFixIndex) {
        this.autoFixIndex = autoFixIndex;
        return this;
    }

    @Override
    public boolean lowercaseExpandedTerms() {
        return lowercaseExpandedTerms;
    }

    public DatabaseConfigBuilder lowercaseExpandedTerms(final boolean lowercaseExpandedTerms) {
        this.lowercaseExpandedTerms = lowercaseExpandedTerms;
        return this;
    }

    @Override
    public File indexPath() {
        return indexPath;
    }

    public DatabaseConfigBuilder indexPath(final File indexPath) {
        this.indexPath = indexPath;
        return this;
    }

    @Override
    public PrefixParser prefixParser() {
        return prefixParser;
    }

    public DatabaseConfigBuilder prefixParser(final PrefixParser prefixParser) {
        this.prefixParser = prefixParser;
        return this;
    }

    @Override
    public PrefixParser indexPrefixParser() {
        return indexPrefixParser;
    }

    public DatabaseConfigBuilder indexPrefixParser(final PrefixParser indexPrefixParser) {
        this.indexPrefixParser = indexPrefixParser;
        return this;
    }

    @Override
    public QueryParser.Operator defaultOperator() {
        return defaultOperator;
    }

    public DatabaseConfigBuilder defaultOperator(final QueryParser.Operator defaultOperator) {
        this.defaultOperator = defaultOperator;
        return this;
    }

    @Override
    public Set<String> primaryKey() {
        return primaryKey;
    }

    public DatabaseConfigBuilder primaryKey(final Set<String> primaryKey) {
        this.primaryKey = primaryKey;
        return this;
    }

    @Override
    public int yandexPostingsWriterBlockSize() {
        return yandexPostingsWriterBlockSize;
    }

    public DatabaseConfigBuilder yandexPostingsWriterBlockSize(final int yandexPostingsWriterBlockSize) {
        this.yandexPostingsWriterBlockSize = yandexPostingsWriterBlockSize;
        return this;
    }

    @Override
    public int yandexTermsWriterBlockSize() {
        return yandexTermsWriterBlockSize;
    }

    public DatabaseConfigBuilder yandexTermsWriterBlockSize(final int yandexTermsWriterBlockSize) {
        this.yandexTermsWriterBlockSize = yandexTermsWriterBlockSize;
        return this;
    }

    @Override
    public Set<String> yandexCodecGroupFields() {
        return yandexCodecGroupFields;
    }

    public DatabaseConfigBuilder yandexCodecGroupFields(final Set<String> yandexCodecGroupFields) {
        this.yandexCodecGroupFields = yandexCodecGroupFields;
        return this;
    }

    @Override
    public int yandexFieldsWriterBufferSize() {
        return yandexFieldsWriterBufferSize;
    }

    public DatabaseConfigBuilder yandexFieldsWriterBufferSize(final int yandexFieldsWriterBufferSize) {
        this.yandexFieldsWriterBufferSize = yandexFieldsWriterBufferSize;
        return this;
    }

    @Override
    public int memoryCodecCompressorLevel() {
        return memoryCodecCompressorLevel;
    }

    public DatabaseConfigBuilder memoryCodecCompressorLevel(final int memoryCodecCompressorLevel) {
        this.memoryCodecCompressorLevel = memoryCodecCompressorLevel;
        return this;
    }

    @Override
    public boolean primaryKeyPartCacheDocuments() {
        return primaryKeyPartCacheDocuments;
    }

    public DatabaseConfigBuilder primaryKeyPartCacheDocuments(final boolean primaryKeyPartCacheDocuments) {
        this.primaryKeyPartCacheDocuments = primaryKeyPartCacheDocuments;
        return this;
    }

    @Override
    public boolean prefixedPrimaryKey() {
        return prefixedPrimaryKey;
    }

    public DatabaseConfigBuilder prefixedPrimaryKey(final boolean prefixedPrimaryKey) {
        this.prefixedPrimaryKey = prefixedPrimaryKey;
        return this;
    }

    @Override
    public int partsExecutorThreadCount() {
        return partsExecutorThreadCount;
    }

    public DatabaseConfigBuilder partsExecutorThreadCount(final int partsExecutorThreadCount) {
        this.partsExecutorThreadCount = partsExecutorThreadCount;
        return this;
    }

    @Override
    public long maxPartJournalSize() {
        return maxPartJournalSize;
    }

    public DatabaseConfigBuilder maxPartJournalSize(final long maxPartJournalSize) {
        this.maxPartJournalSize = maxPartJournalSize;
        return this;
    }

    @Override
    public int limitIndexRequests() {
        return limitIndexRequests;
    }

    public DatabaseConfigBuilder limitIndexRequests(final int limitIndexRequests) {
        this.limitIndexRequests = limitIndexRequests;
        return this;
    }

    @Override
    public boolean logParsedRequest() {
        return logParsedRequest;
    }

    public DatabaseConfigBuilder logParsedRequest(final boolean logParsedRequest) {
        this.logParsedRequest = logParsedRequest;
        return this;
    }

    @Override
    public int maxMergeThreads() {
        return maxMergeThreads;
    }

    public DatabaseConfigBuilder maxMergeThreads(final int maxMergeThreads) {
        this.maxMergeThreads = maxMergeThreads;
        return this;
    }

    @Override
    public int indexCopyRateLimitMb() {
        return indexCopyRateLimitMb;
    }

    public DatabaseConfigBuilder indexCopyRateLimitMb(final int indexCopyRateLimitMb) {
        this.indexCopyRateLimitMb = indexCopyRateLimitMb;
        return this;
    }

    @Override
    public boolean checkCopyness() {
        return checkCopyness;
    }

    public DatabaseConfigBuilder checkCopyness(final boolean checkCopyness) {
        this.checkCopyness = checkCopyness;
        return this;
    }

    @Override
    public boolean useFastCommitCodec() {
        return useFastCommitCodec;
    }

    public DatabaseConfigBuilder useFastCommitCodec(final boolean useFastCommitCodec) {
        this.useFastCommitCodec = useFastCommitCodec;
        return this;
    }

    @Override
    public boolean autoPruning() {
        return autoPruning;
    }

    public DatabaseConfigBuilder autoPruning(final boolean autoPruning) {
        this.autoPruning = autoPruning;
        return this;
    }

    @Override
    public int fieldIndexReadBufferSize() {
        return fieldIndexReadBufferSize;
    }

    public DatabaseConfigBuilder fieldIndexReadBufferSize(final int fieldIndexReadBufferSize) {
        this.fieldIndexReadBufferSize = fieldIndexReadBufferSize;
        return this;
    }

    @Override
    public boolean inMemoryFieldsIndex() {
        return inMemoryFieldsIndex;
    }

    public DatabaseConfigBuilder inMemoryFieldsIndex(final boolean inMemoryFieldsIndex) {
        this.inMemoryFieldsIndex = inMemoryFieldsIndex;
        return this;
    }

    @Override
    public String defaultFieldCodec() {
        return defaultFieldCodec;
    }

    public DatabaseConfigBuilder defaultFieldCodec(final String defaultFieldCodec) {
        this.defaultFieldCodec = defaultFieldCodec;
        return this;
    }

    @Override
    public boolean queueIdServiceFallback() {
        return queueIdServiceFallback;
    }

    public DatabaseConfigBuilder queueIdServiceFallback(final boolean queueIdServiceFallback) {
        this.queueIdServiceFallback = queueIdServiceFallback;
        return this;
    }

    @Override
    public int warmerThreads() {
        return warmerThreads;
    }

    public DatabaseConfigBuilder warmerThreads(final int warmerThreads) {
        this.warmerThreads = warmerThreads;
        return this;
    }

    @Override
    public boolean returnMaxQueueId() {
        return returnMaxQueueId;
    }

    public DatabaseConfigBuilder returnMaxQueueId(final boolean returnMaxQueueId) {
        this.returnMaxQueueId = returnMaxQueueId;
        return this;
    }

    @Override
    public boolean fakeQueueIdsPush() {
        return fakeQueueIdsPush;
    }

    public DatabaseConfigBuilder fakeQueueIdsPush(final boolean fakeQueueIdsPush) {
        this.fakeQueueIdsPush = fakeQueueIdsPush;
        return this;
    }

    @Override
    public int commonTaskExecutorThreads() {
        return commonTaskExecutorThreads;
    }

    public DatabaseConfigBuilder commonTaskExecutorThreads(final int commonTaskExecutorThreads) {
        this.commonTaskExecutorThreads = commonTaskExecutorThreads;
        return this;
    }

    @Override
    public int segmentsPerTier() {
        return segmentsPerTier;
    }

    public DatabaseConfigBuilder segmentsPerTier(final int segmentsPerTier) {
        this.segmentsPerTier = segmentsPerTier;
        return this;
    }

    @Override
    public long maxSegmentSize() {
        return maxSegmentSize;
    }

    public DatabaseConfigBuilder maxSegmentSize(final long maxSegmentSize) {
        this.maxSegmentSize = maxSegmentSize;
        return this;
    }

    @Override
    public boolean copyIndexUpdateQueueIdsOnEmptyDump() {
        return copyIndexUpdateQueueIdsOnEmptyDump;
    }

    public DatabaseConfigBuilder copyIndexUpdateQueueIdsOnEmptyDump(final boolean copyIndexUpdateQueueIdsOnEmptyDump) {
        this.copyIndexUpdateQueueIdsOnEmptyDump = copyIndexUpdateQueueIdsOnEmptyDump;
        return this;
    }

    @Override
    public boolean updatePrefixActivity() {
        return updatePrefixActivity;
    }

    public DatabaseConfigBuilder updatePrefixActivity(final boolean updatePrefixActivity) {
        this.updatePrefixActivity = updatePrefixActivity;
        return this;
    }

    @Override
    public long prefixActivityTimeout() {
        return prefixActivityTimeout;
    }

    public DatabaseConfigBuilder prefixActivityTimeout(final long prefixActivityTimeout) {
        this.prefixActivityTimeout = prefixActivityTimeout;
        return this;
    }

    @Override
    public int partFlushThreshold() {
        return partFlushThreshold;
    }

    public DatabaseConfigBuilder partFlushThreshold(final int partFlushThreshold) {
        this.partFlushThreshold = partFlushThreshold;
        return this;
    }

    @Override
    public boolean warmOnInit() {
        return warmOnInit;
    }

    public DatabaseConfigBuilder warmOnInit(final boolean warmOnInit) {
        this.warmOnInit = warmOnInit;
        return this;
    }

    @Override
    public DocProcessorFactory docProcessorFactory() {
        return docProcessorFactory;
    }

    public DatabaseConfigBuilder docProcessorFactory(final DocProcessorFactory docProcessorFactory) {
        this.docProcessorFactory = docProcessorFactory;
        return this;
    }

    @Override
    public boolean syncSearcherDefault() {
        return syncSearcherDefault;
    }

    public DatabaseConfigBuilder syncSearcherDefault(final boolean syncSearcherDefault) {
        this.syncSearcherDefault = syncSearcherDefault;
        return this;
    }

    @Override
    public boolean earlyInterruptDefault() {
        return earlyInterruptDefault;
    }

    public DatabaseConfigBuilder earlyInterruptDefault(final boolean earlyInterruptDefault) {
        this.earlyInterruptDefault = earlyInterruptDefault;
        return this;
    }

    @Override
    public SimpleHistogramMetric indexLagHistogram() {
        return indexLagHistogram;
    }

    public DatabaseConfigBuilder indexLagHistogram(final SimpleHistogramMetric indexLagHistogram) {
        this.indexLagHistogram = indexLagHistogram;
        return this;
    }

    @Override
    public boolean oDirectWrite() {
        return oDirectWrite;
    }

    public DatabaseConfigBuilder oDirectWrite(final boolean oDirectWrite) {
        this.oDirectWrite = oDirectWrite;
        return this;
    }

    @Override
    public FieldsConfig fieldsConfig() {
        return fieldsConfig;
    }

    public DatabaseConfigBuilder fieldsConfig(final FieldsConfig fieldsConfig) {
        this.fieldsConfig = fieldsConfig;
        return this;
    }

    @Override
    public void reloadDynamicConfig(final String name, final String configStr) throws ConfigException, IOException {
        this.fieldsConfig = new FieldsConfig(this.fieldsConfig, name, configStr);
    }


    @Override
    public int yandexCodecCompressorLevel() {
        return yandexCodecCompressorLevel;
    }

    public String name() {
        return name;
    }

    @Override
    public Set<String> services() {
        return services;
    }

    public DatabaseConfigBuilder services(final Set<String> services) {
        this.services = new LinkedHashSet<>(services);
        return this;
    }

    @Override
    public boolean orderIndependentUpdate() {
        return orderIndependentUpdate;
    }

    public DatabaseConfigBuilder orderIndependentUpdate(final boolean orderIndependentUpdate) {
        this.orderIndependentUpdate = orderIndependentUpdate;
        return this;
    }
}
