package ru.yandex.msearch.config;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.lucene.analysis.LengthFilter;
import org.apache.lucene.analysis.LowerCaseFilter;

import ru.yandex.analyzer.AazFilter;
import ru.yandex.analyzer.FilterFactory;
import ru.yandex.analyzer.HeadersFilter;
import ru.yandex.analyzer.IgnorePrefixFilter;
import ru.yandex.analyzer.Int2HexFilter;
import ru.yandex.analyzer.MafFilter;
import ru.yandex.analyzer.MultiTokenPermuteFilter;
import ru.yandex.analyzer.PaddingFilter;
import ru.yandex.analyzer.PermuteFilter;
import ru.yandex.analyzer.PositionSuffixFilter;
import ru.yandex.analyzer.PrefixSubstringsFilter;
import ru.yandex.analyzer.PrefixSubstringsFilterFactory;
import ru.yandex.analyzer.ReplaceCharFilter;
import ru.yandex.analyzer.ReplaceStringFilter;
import ru.yandex.analyzer.SedFilter;
import ru.yandex.analyzer.SetFilter;
import ru.yandex.analyzer.TruncateFilter;
import ru.yandex.analyzer.UpperCaseFilter;
import ru.yandex.analyzer.YandexLemmFilter;
import ru.yandex.function.GenericFunction;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.string.CollectionParser;

public class FilterParser implements GenericFunction<String, FilterFactory, Exception> {
    private static final String ARGS_SEPARATOR = ":";

    private final String field;
    private final char argsDelimiter;
    private final CollectionParser<String, List<String>, Exception> argsParser;

    public FilterParser(final String field, final char argsDelimiter) {
        this.field = field;
        this.argsDelimiter = argsDelimiter;
        this.argsParser = new CollectionParser<>(String::trim, ArrayList::new, argsDelimiter);
    }

    @Override
    public FilterFactory apply(final String filter) throws Exception {
        List<String> args = argsParser.apply(filter);

        FilterFactory result = null;
        switch (args.size()) {
            case 1:
                result = parseFilter(args.get(0));
                break;
            case 2:
                result = parseFilterOneArg(args.get(0), args.get(1));
                break;
            case 3:
                result = parseFilterTwoarg(args.get(0), args.get(1), args.get(2));
                break;
            default:
                throw new ConfigException(
                        "Too many parameters in filter " + filter
                                + " for field " + field);
        }

        if (result == null) {
            throw new IllegalArgumentException("Filter not found: " + filter);
        }

        return result;
    }

    private FilterFactory parseFilter(final String filter) {
        switch (filter) {
            case "lowercase":
                return LowerCaseFilter::new;

            case "uppercase":
                return UpperCaseFilter::new;

            case "lemmer":
                return YandexLemmFilter::new;

            case "ps":
                return PositionSuffixFilter::new;

            case "i2h":
                return Int2HexFilter::new;

            case "yo":
            case "io":
                return (prev) ->
                        new ReplaceCharFilter(
                                new ReplaceCharFilter(prev, '\u0451', '\u0435'),
                                '\u0401',
                                '\u0415');
            case "headers":
                return HeadersFilter::new;

            case "set":
                return SetFilter::new;

            default:
                return null;
        }
    }

    private FilterFactory parseFilterOneArg(final String filter, final String param) {
        switch (filter) {
            case "ignoreprefix":
                return (prev) -> new IgnorePrefixFilter(prev, param);

            case "maf":
                return (prev) -> new MafFilter(prev, Double.valueOf(param));

            case "daf":
                return (prev) -> new MafFilter(prev, 1d / Double.valueOf(param));

            case "padding":
                return (prev) -> new PaddingFilter(prev, Integer.parseInt(param));
            case "aaz":
            case "paz":
                return (prev) -> new AazFilter(prev, Double.valueOf(param));

            case "truncate":
                return (prev) -> new TruncateFilter(prev, Integer.parseInt(param));
            case "prefixes":
                return new PrefixSubstringsFilterFactory(Integer.parseInt(param));

            default:
                return null;
        }
    }

    private FilterFactory parseFilterTwoarg(
            final String filter,
            final String param1,
            final String param2)
    {
        switch (filter) {
            case "length":
                return (prev) -> new LengthFilter(prev, Integer.parseInt(param1), Integer.parseInt(param2));
            case "replace":
                if (param1.length() == 1 && param2.length() == 1) {
                    return (prev) -> new ReplaceCharFilter(prev, param1.charAt(0), param2.charAt(0));
                } else {
                    return (prev) -> new ReplaceStringFilter(prev, param1, param2);
                }
            case "permute":
                return (prev) -> new PermuteFilter(prev, Integer.parseInt(param1), Integer.parseInt(param2));
            case "mt_permute":
                return (prev) -> new MultiTokenPermuteFilter(
                        prev,
                        Integer.parseInt(param1),
                        Integer.parseInt(param2));
            case "sed":
                final Pattern pattern = Pattern.compile(param1);
                return (prev) -> new SedFilter(prev, pattern, param2);

            default:
                return null;
        }
    }
}
