package ru.yandex.msearch.fieldscache;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;

import java.util.logging.Level;
import java.util.logging.Logger;

import java.io.IOException;

import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.BytesRef;

import ru.yandex.msearch.ActivityProvider;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.collector.YaField.FieldType;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.Prefix;

import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

import ru.yandex.msearch.util.JavaAllocator;

public class FieldsCache implements Stater {
    public static final String CACHED_FILES_EXTENSION = "fc";
    private final Map<Object, ReaderFieldsCache> cache =
        Collections.synchronizedMap(new WeakHashMap<>());
//        new ConcurrentHashMap<>();
    private final DatabaseConfig config;
    private final Logger logger;

    public FieldsCache(final DatabaseConfig config, final Logger logger) {
        this.config = config;
        this.logger = logger;
    }

    public List<CacheInput> getCachesFor(final IndexReader reader, Set<String> fields) {
        final Object cacheKey = reader.getCoreCacheKey();
        final ReaderFieldsCache readerCache = cache.get(cacheKey);
        if (readerCache == null) {
            return null;
        }
        return readerCache.getCachesFor(fields);
    }

    public CacheInput getCacheFor(final IndexReader reader, final String field) {
        final Object cacheKey = reader.getCoreCacheKey();
        final ReaderFieldsCache readerCache = cache.get(cacheKey);
        if (readerCache == null) {
            return null;
        }
        return readerCache.getCacheFor(field);
    }

    public void loadCache(
        final IndexReader reader,
        final ActivityProvider activityProvider)
        throws IOException
    {
//        System.err.println("loadCache: " + reader);
        final IndexReader[] subReaders = getSubReaders(reader);

        for (int i = 0; i < subReaders.length; i++) {
            IndexReader subReader = subReaders[i];
            Object cacheKey = subReader.getCoreCacheKey();
            ReaderFieldsCache readerCache = cache.get(cacheKey);
            if (readerCache == null) {
                ReaderFieldsCache newCache =
                    new ReaderFieldsCache(activityProvider, subReader, config);
                readerCache =
                    cache.putIfAbsent(cacheKey, newCache);
                if (readerCache == null) {
                    readerCache = newCache;
//                    System.err.println("CACHING: " + subReader);
                    readerCache.load();
                }
            }
        }
    }

    public void reloadCache(
        final IndexReader reader,
        final Set<String> newPrefixes)
        throws IOException
    {
//        System.err.println("loadCache: " + reader);
        final IndexReader[] subReaders = getSubReaders(reader);

        for (int i = 0; i < subReaders.length; i++) {
            IndexReader subReader = subReaders[i];
            Object cacheKey = subReader.getCoreCacheKey();
            ReaderFieldsCache readerCache = cache.get(cacheKey);
            if (readerCache != null) {
                readerCache.reload(newPrefixes);
            }
        }
    }

    public void incRef(final IndexReader reader) {
//        System.err.println("incRef: " + reader);
        final IndexReader[] subReaders = getSubReaders(reader);
        for (int i = 0; i < subReaders.length; i++) {
//            System.err.println("incRefs: " + subReaders[i] + " / "
//                + subReaders[i].getCoreCacheKey());
            ReaderFieldsCache readerCache =
                cache.get(subReaders[i].getCoreCacheKey());
            if (readerCache != null) {
                int refs = readerCache.incRef();
//                System.err.println("incRefs: " + refs + " / " + subReaders[i]);
            }
        }
    }

    public void decRef(final IndexReader reader) {
//        System.err.println("decRef: " + reader);
        final IndexReader[] subReaders = getSubReaders(reader);
        for (int i = 0; i < subReaders.length; i++) {
            Object cacheKey = subReaders[i].getCoreCacheKey();
//            System.err.println("decRefs: " + subReaders[i] + " / " + cacheKey);
            ReaderFieldsCache readerCache = cache.get(cacheKey);
            if (readerCache != null) {
                int refs = readerCache.decRef();
//                System.err.println("Refs: " + refs + " / " + subReaders[i]);
                if (refs <= 0) {
                    cache.remove(cacheKey);
//                    System.err.println("UNCaching: " + subReaders[i]);
                    readerCache.close();
                }
            }
        }
    }

    private IndexReader[] getSubReaders(final IndexReader reader) {
        IndexReader[] subReaders = reader.getSequentialSubReaders();
        if (subReaders == null) {
            subReaders = new IndexReader[1];
            subReaders[0] = reader;
        }
        return subReaders;
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Iterator<Map.Entry<String, JavaAllocator>> iter =
            JavaAllocator.iterator();
        long totalSize = 0;
        long totalDataSize = 0;
        long totalIdxSize = 0;
        HashMap<String, long[]> perFieldSize = new HashMap<>();
        HashMap<String, long[]> perFieldIdxSize = new HashMap<>();
        while (iter.hasNext()) {
            Map.Entry<String, JavaAllocator> entry = iter.next();
            String name = entry.getKey();
            if (name.startsWith(RawFieldCache.FC_IDX)) {
                long size = entry.getValue().memorySize();
                name = name.substring(RawFieldCache.FC_IDX.length());
                totalIdxSize += size;
                totalSize += size;
                long[] fieldIdxSize =
                    perFieldIdxSize.computeIfAbsent(name, k -> new long[1]);
                fieldIdxSize[0] += size;
            } else if (name.startsWith(RawFieldCache.FC)) {
                long size = entry.getValue().memorySize();
                name = name.substring(RawFieldCache.FC.length());
                totalDataSize += size;
                totalSize += size;
                long[] fieldSize =
                    perFieldSize.computeIfAbsent(name, k -> new long[1]);
                fieldSize[0] += size;
            }
        }
        statsConsumer.stat("fieldscache_totalsize_ammm", totalSize);
        statsConsumer.stat("fieldscache_totalsize_axxx", totalSize);
        statsConsumer.stat("fieldscache_totalsize_ammx", totalSize);
        statsConsumer.stat("fieldscache_total_data_size_ammm", totalDataSize);
        statsConsumer.stat("fieldscache_total_data_size_axxx", totalDataSize);
        statsConsumer.stat("fieldscache_total_data_size_ammx", totalDataSize);
        statsConsumer.stat("fieldscache_total_index_size_ammm", totalIdxSize);
        statsConsumer.stat("fieldscache_total_index_size_axxx", totalIdxSize);
        statsConsumer.stat("fieldscache_total_index_size_ammx", totalIdxSize);
        for (Map.Entry<String, long[]> entry: perFieldSize.entrySet()) {
            String name = entry.getKey();
            long size = entry.getValue()[0];
            statsConsumer.stat("fieldscache_field_" + name + "_ammx", size);
            statsConsumer.stat("fieldscache_field_" + name + "_axxx", size);
        }
        for (Map.Entry<String, long[]> entry: perFieldIdxSize.entrySet()) {
            String name = entry.getKey();
            long size = entry.getValue()[0];
            statsConsumer.stat("fieldscache_idx_" + name + "_ammx", size);
            statsConsumer.stat("fieldscache_idx_" + name + "_axxx", size);
        }
    }
}
