package ru.yandex.msearch.fieldscache;

import java.io.Closeable;
import java.io.IOException;

import java.text.ParseException;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.RAMDirectory;
import org.apache.lucene.util.packed.PackedInts;

import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.collector.YaField;
import ru.yandex.msearch.collector.YaField.FieldType;
import ru.yandex.msearch.util.PackedIntsHashMap;

import ru.yandex.msearch.util.JavaAllocator;

public class RawFieldCache extends AbstractFieldCache {
    public static final String FC = "FC_";
    public static final String FC_IDX = "FC_idx_";

    private static final RAMDirectory dir = new RAMDirectory();

    private final String filename;
    private final String fieldname;
    private final String aliasname;
    private final int maxDoc;
    private final FieldType fieldType;
    private IndexInput input = null;
    private RawCacheOutput writer = null;
    private Index index = null;

    //CSOFF: ParameterNumber
    public RawFieldCache(
        final FieldConfig fieldConfig,
        final String fieldname,
        final String aliasname,
        final String readerKey,
        final int maxDoc)
    {
        super(fieldConfig);
        this.fieldType = fieldConfig.type();
        this.fieldname = fieldname;
        this.aliasname = aliasname;
        this.filename = aliasname + '@' + readerKey;
        this.maxDoc = maxDoc;
    }
    //CSON: ParameterNumber

    @Override
    public void free() {
        if (writer != null) {
            try {
                writer.close();
            } catch (IOException ign) {
                ign.printStackTrace();
            }
            writer = null;
        }
        if (input != null) {
            try {
                input.close();
                dir.deleteFile(filename);
            } catch (IOException e) {
            }
            input = null;
        }
        if (index != null) {
            try {
                index.close();
            } catch (IOException cannotBe) {
                cannotBe.printStackTrace();
            }
            index = null;
        } else {
            return;
        }
    }

    @Override
    public void finalize() {
        if (input != null) {
            System.err.println("Finalizing unfreed RawFieldCache: "
                + "field: " + fieldname
                + ", readerKey: " + filename);
        }
    }

    @Override
    public CacheOutput writer() throws IOException {
        if (writer == null) {
            writer = new RawCacheOutput(
                dir.createOutput(filename, JavaAllocator.get(FC + aliasname)));
        }
        return writer;
    }

    @Override
    public CacheOutput reopenWriter() throws IOException {
        writer.reopen();
        return writer;
    }

    @Override
    public CacheInput reader() throws IOException {
        return new RawCacheInput();
    }

    private class RawCacheOutput implements CacheOutput {
        private final IndexOutput out;
        private int[] index = new int[maxDoc];
        private long lastFilePosition = 0;
        private int currentDocId = -1;
        private int docs = 0;

        RawCacheOutput(final IndexOutput out) throws IOException {
            this.out = out;
            //offset data by one byte, so zero offset in index will be
            //evaluated later as missing value
            out.writeByte((byte) 0);
        }

        @Override
        public void newDoc(final int docId) {
            currentDocId = docId;
            lastFilePosition = out.getFilePointer();
            index[docId] = (int) lastFilePosition;
        }

        @Override
        public void shareDoc(final int docId) {
            index[docId] = (int) lastFilePosition;
            docs++;
        }

        @Override
        public void writeByte(final byte value) throws IOException {
//            System.err.println("writeByte: " + value);
            out.writeByte(value);
        }

        @Override
        public void writeInt(final int value) throws IOException {
            out.writeInt(value);
        }

        @Override
        public void writeVInt(final int value) throws IOException {
            out.writeVInt(value);
        }

        @Override
        public void writeLong(final long value) throws IOException {
            out.writeLong(value);
        }

        @Override
        public void writeVLong(final long value) throws IOException {
            out.writeVLong(value);
        }

        @Override
        public void write(final byte[] in, final int offset, final int length)
            throws IOException
        {
            out.writeBytes(in, offset, length);
        }

        @Override
        public void commit() throws IOException {
            docs++;
        }

        @Override
        public void rollback() throws IOException {
            index[currentDocId] = 0;
            out.seek(lastFilePosition);
        }

        @Override
        public void finish() throws IOException {
//            out.close();
            out.flush();
            final Index tempIndex;
            if (docs < maxDoc / 2) {
                tempIndex = new PackedHashMapIndex(
                    docs,
                    maxDoc,
                    lastFilePosition,
                    JavaAllocator.get(FC_IDX + aliasname));
                for (int i = 0; i < maxDoc; i++) {
                    int off = index[i];
                    if (off > 0) {
                        tempIndex.put(i, off);
                    }
                }
            } else {
                tempIndex = new PackedIndex(
                maxDoc,
                lastFilePosition,
                JavaAllocator.get(FC_IDX + aliasname));
                for (int i = 0; i < maxDoc; i++) {
                    tempIndex.put(i, index[i]);
                }
            }
            index = null;
            input = dir.openInput(filename);
//            if (oldInput != null) {
//                oldInput.file().incRef();
//                oldInput.close();
//            }
            Index oldIndex = RawFieldCache.this.index;
            RawFieldCache.this.index = tempIndex;
            if (oldIndex != null) {
                oldIndex.close();
            }
        }

        public void reopen() {
            index = new int[maxDoc];
            for (int i = 0; i < maxDoc; i++) {
                index[i] = (int) RawFieldCache.this.index.get(i);
            }
        }

        public void close() throws IOException {
            out.close();
        }
    }

    private class RawCacheInput implements CacheInput {
        private final IndexInput in;

        RawCacheInput() {
            in = (IndexInput) input.clone();
        }

        @Override
        public boolean seek(final int docId) throws IOException {
            int offset = (int) index.get(docId);
            if (offset == 0) {
                return false;
            } else {
                in.seek(offset);
                return true;
            }
        }

        @Override
        public YaField field() throws IOException {
            return fieldType.create(this);
        }

        @Override
        public byte readByte() throws IOException {
            return in.readByte();
        }

        @Override
        public int readInt() throws IOException {
            return in.readInt();
        }

        @Override
        public int readVInt() throws IOException {
            return in.readVInt();
        }

        @Override
        public long readLong() throws IOException {
            return in.readLong();
        }

        @Override
        public long readVLong() throws IOException {
            return in.readVLong();
        }

        @Override
        public void read(
            final byte[] out,
            final int offset,
            final int length)
            throws IOException
        {
            in.readBytes(out, offset, length);
        }

        @Override
        public String fieldname() {
            return fieldname;
        }
    }

    private static abstract class Index implements Closeable {
        public abstract long get(final int idx);

        public abstract void put(final int idx, final long value);
    }

    public static class PackedIndex extends Index {
        private PackedInts.Mutable index;

        PackedIndex(
            final int maxDoc,
            final long maxValue,
            final JavaAllocator allocator)
        {
            index = PackedInts.getMutable(
                maxDoc,
                PackedInts.bitsRequired(maxValue),
                allocator);
        }

        @Override
        public long get(final int idx) {
            return index.get(idx);
        }

        @Override
        public void put(final int idx, final long value) {
            index.set(idx, value);
        }

        @Override
        public void close() {
            index.close();
        }
    }

    public static class PackedHashMapIndex extends Index {
        private PackedIntsHashMap index;

        PackedHashMapIndex(
            final int size,
            final int maxDoc,
            final long maxValue,
            final JavaAllocator allocator)
        {
            index = new PackedIntsHashMap(
                size,
                maxDoc,
                maxValue,
                allocator);
        }

        @Override
        public long get(final int idx) {
            long value = index.get(idx);
            if (value == -1) {
                return 0;
            } else {
                return value;
            }
        }

        @Override
        public void put(final int idx, final long value) {
            index.put(idx, value);
        }

        @Override
        public void close() {
            index.close();
        }
    }
}
