package ru.yandex.msearch.fieldscache;

import java.io.Closeable;
import java.io.IOException;
import java.io.StringReader;

import java.lang.ref.WeakReference;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;

import org.apache.lucene.document.FieldSelectorResult;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;

import ru.yandex.analyzer.PaddingFilter;

import ru.yandex.msearch.ActivityProvider;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.collector.AbstractFieldVisitor;
import ru.yandex.msearch.collector.SingleFieldToIndex;
import ru.yandex.msearch.collector.YaField.FieldType;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.Prefix;

public class ReaderFieldsCache implements Closeable {
    private static final TokenStream DUMMY_STREAM =
        new KeywordAnalyzer().tokenStream(
            "DUMMY",
            new StringReader("YMMUD"));

    private final Map<String, FieldCache> perFieldCaches = new HashMap<>();
    private final DatabaseConfig config;
    private final WeakReference<IndexReader> reader;
    private final ActivityProvider activityProvider;
    private final AtomicInteger refs = new AtomicInteger(0);
    private boolean closed = false;

    public ReaderFieldsCache(
        final ActivityProvider activityProvider,
        final IndexReader reader,
        final DatabaseConfig config)
    {
        this.reader = new WeakReference<>(reader);
        this.config = config;
        this.activityProvider = activityProvider;
    }

    public List<CacheInput> getCachesFor(final Set<String> fields) {
        ArrayList<CacheInput> caches = new ArrayList(fields.size());
        for (String field: fields) {
            FieldCache cache = perFieldCaches.get(field);
            if (cache == null) {
//                System.err.println("NO CACHE FOR field: " + field + '@' + reader);
                return null;
            }
            try {
                caches.add(cache.reader());
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }
        return caches;
    }

    public CacheInput getCacheFor(final String field) {
        FieldCache cache = perFieldCaches.get(field);
        if (cache == null) {
            return null;
        }
        try {
            return cache.reader();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    private boolean loadFiltered(final FieldConfig config) {
        if (config.indexFilters().isEmpty() && config.indexFilters().isEmpty()) {
            return false;
        }

        // i don't really understand now for what reason PaddingFilter so special
        // but will keep it same old way for a while
        if ((config.indexFilters().size() == 1 && config.searchFilters().size() <= 1) ||
                (config.indexFilters().size() == 0 && config.searchFilters().size() == 1))
        {
            if (config.indexFilters().size() == 1) {
                TokenFilter indexFilter =
                        config.indexFilters().get(0).createFilter(DUMMY_STREAM);
                if (!(indexFilter instanceof PaddingFilter)) {
                    return true;
                }
            }

            if (config.searchFilters().size() == 1) {
                TokenFilter searchFilter =
                        config.searchFilters().get(0).createFilter(DUMMY_STREAM);
                if (!(searchFilter instanceof PaddingFilter)) {
                    return true;
                }
            }

            return false;
        }

        return true;
    }

    public synchronized void load() throws IOException {
        for(String field: config.knownFields()) {
            FieldConfig config = this.config.fieldConfig(field);
            FieldCache.Type cacheType = config.cache();

            if (cacheType != null
                && cacheType != FieldCache.Type.NO_CACHE)
            {
                String fieldName = field;
                String aliasFor = config.cacheAlias();
                if (aliasFor != null) {
                    fieldName = aliasFor;
                }
                if (perFieldCaches.get(fieldName) == null) {
                    loadFieldCache(fieldName, field, config, cacheType);
                }
            }
        }
    }

    public synchronized void reload(final Set<String> newPrefixes)
        throws IOException
    {
        IndexReader reader = this.reader.get();
        if (reader == null) {
            return;
        }
        for (FieldCache cache: perFieldCaches.values()) {
            if (cache.fieldConfig().prefixed()) {
                if (!loadFiltered(cache.fieldConfig())) {
                    updateCache(cache, newPrefixes, reader);
                } else {
                    updateCacheFiltered(cache, newPrefixes, reader);
                }
            }
        }
    }

    private void loadFieldCache(
        final String fieldname,
        final String aliasname,
        final FieldConfig config,
        final FieldCache.Type cacheType)
        throws IOException
    {
//        System.err.println("LOADING CACHE: " + fieldname  + '@' + reader);

        IndexReader reader = this.reader.get();
        if (reader == null) {
            return;
        }

        FieldCache cache =
            cacheType.createCache(
                reader,
                fieldname,
                aliasname,
                config);
        if (cache == null) {
//            System.err.println("CACHE NULL: " + fieldname);
            return;
        }
        try {
            cache = loadCache(cache);
            if (cache != null) {
                perFieldCaches.put(fieldname, cache);
            }
        } catch (Throwable e) {
            e.printStackTrace();
            cache.free();
        }
    }

    private FieldCache loadCache(final FieldCache cache) throws IOException {
        IndexReader reader = this.reader.get();
        if (reader == null) {
            return null;
        }
        final Terms terms = reader.terms(cache.aliasname());
        if (terms == null) {
            return null;
        }
        final TermsEnum te = terms.iterator(false);
        if (te == null) {
//            System.err.println("NOT TERMS ENUM FOR: " + cache.fieldname());
            return null;
        }

        final boolean prefixed = cache.fieldConfig().prefixed();
        final boolean onlyActive = cache.fieldConfig().cacheOnlyActive();
        if (!loadFiltered(cache.fieldConfig())) {
            if (prefixed && onlyActive) {
                return loadCacheByActivity(cache, te, reader);
            } else {
                return loadSequential(cache, te, reader);
            }
        } else {
            if (prefixed && onlyActive) {
                return loadCacheByActivityFiltered(cache, te, reader);
            } else {
                return loadSequentialFiltered(cache, te, reader);
            }
        }
    }

    private FieldCache loadSequential(
        final FieldCache cache,
        final TermsEnum te,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final boolean prefixed = cache.fieldConfig().prefixed();
        final BytesRef ref = new BytesRef(0);
        final CacheOutput out = cache.writer();
        final Bits deletedDocs = reader.getDeletedDocs();
        final BytesRef prefixRef = new BytesRef(0);
        boolean success = false;
        DocsEnum de = null;
        try {
            for (BytesRef term = te.next(); term != null; term = te.next()) {
                ref.bytes = term.bytes;
                ref.length = term.length;
//                System.err.println("term: " + term.utf8ToString());
                if (prefixed) {
                    int sep = indexOf(term, (byte) '#');
                    if (sep == -1) {
                        continue;
                    }
                    ref.offset = term.offset + sep + 1;
                    ref.length -= sep + 1;
//                    System.err.println("sep="+sep + ", off=" + term.offset + ", len=" + term.length);
//                    System.err.println("ref: " + ref.utf8ToString());
                } else {
                    ref.offset = term.offset;
                }
                de = te.docs(deletedDocs, de);
                boolean serialized = false;
                for (
                    int docId = de.nextDoc();
                    docId != DocIdSetIterator.NO_MORE_DOCS;
                    docId = de.nextDoc())
                {
                    if (!serialized) {
                        serialized = true;
                        out.newDoc(docId);
                        if (type.parseAndSerialize(ref, out)) {
                            out.commit();
                        } else {
                            out.rollback();
                            break;
                        }
                    } else {
                        out.shareDoc(docId);
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
        return cache;
    }

    private FieldCache loadSequentialFiltered(
        final FieldCache cache,
        final TermsEnum te,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final CacheOutput out = cache.writer();
        final Bits deletedDocs = reader.getDeletedDocs();
        boolean success = false;
        DocsEnum de = null;
        SingleFieldVisitor visitor = new SingleFieldVisitor(cache.fieldname());
        final BytesRef ref = visitor.ref();
        final BytesRef prev = new BytesRef(0);
        try {
            for (BytesRef term = te.next(); term != null; term = te.next()) {
                de = te.docs(deletedDocs, de);
                for (
                    int docId = de.nextDoc();
                    docId != DocIdSetIterator.NO_MORE_DOCS;
                    docId = de.nextDoc())
                {
                    visitor.reset();
                    reader.readDocument(docId, visitor);
                    if (!ref.equals(prev)) {
                        out.newDoc(docId);
                        if (type.parseAndSerialize(ref, out)) {
                            out.commit();
//                            System.err.println("Sq:" + cache.fieldname() + ": " + ref.toString());
                            prev.bytes = ref.bytes;
                            prev.offset = ref.offset;
                            prev.length = ref.length;
                        } else {
                            out.rollback();
                            break;
                        }
                    } else {
                        out.shareDoc(docId);
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
        return cache;
    }

    private FieldCache loadCacheByActivity(
        final FieldCache cache,
        final TermsEnum te,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final BytesRef ref = new BytesRef(0);
        final CacheOutput out = cache.writer();
        final Bits deletedDocs = reader.getDeletedDocs();
        final BytesRef prefixRef = new BytesRef(0);
        boolean success = false;
        DocsEnum de = null;
        Collection<String> prefixes = activityProvider.sortedPrefixesStrings();
        try {
            for (String prefixString: prefixes) {
                if (!activityProvider.activePrefix(prefixString)) {
                    continue;
                }
                System.err.println("Prefix: " + prefixString);
                BytesRef prefix = new BytesRef(prefixString);
                TermsEnum.SeekStatus ss = te.seek(prefix, false);
                if (ss == TermsEnum.SeekStatus.END) {
                    break;
                }
                for (
                    BytesRef term = te.term();
                    term != null && term.startsWith(prefix);
                    term = te.next())
                {
                    ref.bytes = term.bytes;
                    ref.length = term.length;
//                System.err.println("term: " + term.utf8ToString());
                    int sep = indexOf(term, (byte) '#');
                    if (sep == -1) {
                        continue;
                    }
                    ref.offset = term.offset + sep + 1;
                    ref.length -= sep + 1;
//                    System.err.println("sep="+sep + ", off=" + term.offset + ", len=" + term.length);
//                    System.err.println("ref: " + ref.utf8ToString());
                    de = te.docs(deletedDocs, de);
                    boolean serialized = false;
                    for (
                        int docId = de.nextDoc();
                        docId != DocIdSetIterator.NO_MORE_DOCS;
                        docId = de.nextDoc())
                    {
                        if (!serialized) {
                            serialized = true;
                            out.newDoc(docId);
                            if (type.parseAndSerialize(ref, out)) {
                                out.commit();
                            } else {
                                out.rollback();
                                break;
                            }
                        } else {
                            out.shareDoc(docId);
                        }
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
        return cache;
    }

    private FieldCache loadCacheByActivityFiltered(
        final FieldCache cache,
        final TermsEnum te,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final CacheOutput out = cache.writer();
        final Bits deletedDocs = reader.getDeletedDocs();
        final BytesRef prefixRef = new BytesRef(0);
        boolean success = false;
        DocsEnum de = null;
        SingleFieldVisitor visitor = new SingleFieldVisitor(cache.fieldname());
        final BytesRef valueRef = visitor.ref();
        final BytesRef prev = new BytesRef(0);
        Collection<String> prefixes = activityProvider.sortedPrefixesStrings();
        try {
            for (String prefixString: prefixes) {
                if (!activityProvider.activePrefix(prefixString)) {
                    continue;
                }
                System.err.println("Filtered Prefix: " + prefixString);
                BytesRef prefix = new BytesRef(prefixString);
                TermsEnum.SeekStatus ss = te.seek(prefix, false);
                if (ss == TermsEnum.SeekStatus.END) {
                    break;
                }
                for (
                    BytesRef term = te.term();
                    term != null && term.startsWith(prefix);
                    term = te.next())
                {
                    de = te.docs(deletedDocs, de);
                    for (
                        int docId = de.nextDoc();
                        docId != DocIdSetIterator.NO_MORE_DOCS;
                        docId = de.nextDoc())
                    {
                        visitor.reset();
                        reader.readDocument(docId, visitor);
                        if (!valueRef.equals(prev)) {
                            out.newDoc(docId);
                            if (type.parseAndSerialize(valueRef, out)) {
                                out.commit();
//                            System.err.println(cache.fieldname() + ": " + valueRef.toString());
                                prev.bytes = valueRef.bytes;
                                prev.offset = valueRef.offset;
                                prev.length = valueRef.length;
                            } else {
                                out.rollback();
                                break;
                            }
                        } else {
                            out.shareDoc(docId);
                        }
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
        return cache;
    }

    private void updateCache(
        final FieldCache cache,
        final Set<String> newPrefixes,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final Terms terms = reader.terms(cache.fieldname());
        if (terms == null) {
            return;
        }
        final TermsEnum te = terms.iterator(false);
        if (te == null) {
//            System.err.println("NOT TERMS ENUM FOR: " + cache.fieldname());
            return;
        }
        final BytesRef ref = new BytesRef(0);
        final CacheOutput out = cache.reopenWriter();
        final Bits deletedDocs = reader.getDeletedDocs();
        final BytesRef prefixRef = new BytesRef(0);
        boolean success = false;
        DocsEnum de = null;
        try {
            for (String prefixString: newPrefixes) {
                System.err.println("updateCache Prefix: " + prefixString);
                BytesRef prefix = new BytesRef(prefixString);
                TermsEnum.SeekStatus ss = te.seek(prefix, false);
                if (ss == TermsEnum.SeekStatus.END) {
                    break;
                }
                for (
                    BytesRef term = te.term();
                    term != null && term.startsWith(prefix);
                    term = te.next())
                {
                    ref.bytes = term.bytes;
                    ref.length = term.length;
//                System.err.println("term: " + term.utf8ToString());
                    int sep = indexOf(term, (byte) '#');
                    if (sep == -1) {
                        continue;
                    }
                    ref.offset = term.offset + sep + 1;
                    ref.length -= sep + 1;
//                    System.err.println("sep="+sep + ", off=" + term.offset + ", len=" + term.length);
//                    System.err.println("ref: " + ref.utf8ToString());
                    de = te.docs(deletedDocs, de);
                    boolean serialized = false;
                    for (
                        int docId = de.nextDoc();
                        docId != DocIdSetIterator.NO_MORE_DOCS;
                        docId = de.nextDoc())
                    {
                        if (!serialized) {
                            serialized = true;
                            out.newDoc(docId);
                            if (type.parseAndSerialize(ref, out)) {
                                out.commit();
                            } else {
                                out.rollback();
                                break;
                            }
                        } else {
                            out.shareDoc(docId);
                        }
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
    }

    private void updateCacheFiltered(
        final FieldCache cache,
        final Set<String> newPrefixes,
        final IndexReader reader)
        throws IOException
    {
        final FieldType type = cache.fieldConfig().type();
        final Terms terms = reader.terms(cache.fieldname());
        if (terms == null) {
            return;
        }
        final TermsEnum te = terms.iterator(false);
        if (te == null) {
//            System.err.println("NOT TERMS ENUM FOR: " + cache.fieldname());
            return;
        }
        final CacheOutput out = cache.reopenWriter();
        final Bits deletedDocs = reader.getDeletedDocs();
        final BytesRef prefixRef = new BytesRef(0);
        SingleFieldVisitor visitor = new SingleFieldVisitor(cache.fieldname());
        final BytesRef valueRef = visitor.ref();
        final BytesRef prev = new BytesRef(0);
        boolean success = false;
        DocsEnum de = null;
        try {
            for (String prefixString: newPrefixes) {
                System.err.println("updateCache Prefix: " + prefixString);
                BytesRef prefix = new BytesRef(prefixString);
                TermsEnum.SeekStatus ss = te.seek(prefix, false);
                if (ss == TermsEnum.SeekStatus.END) {
                    break;
                }
                for (
                    BytesRef term = te.term();
                    term != null && term.startsWith(prefix);
                    term = te.next())
                {
                    de = te.docs(deletedDocs, de);
                    for (
                        int docId = de.nextDoc();
                        docId != DocIdSetIterator.NO_MORE_DOCS;
                        docId = de.nextDoc())
                    {
                        visitor.reset();
                        reader.readDocument(docId, visitor);
                        if (!valueRef.equals(prev)) {
                            out.newDoc(docId);
                            if (type.parseAndSerialize(valueRef, out)) {
                                out.commit();
                                prev.bytes = valueRef.bytes;
                                prev.offset = valueRef.offset;
                                prev.length = valueRef.length;
                            } else {
                                out.rollback();
                                break;
                            }
                        } else {
                            out.shareDoc(docId);
                        }
                    }
                }
            }
            success = true;
        } finally {
            if (success) {
                out.finish();
            }
        }
    }

    public int incRef() {
        return refs.incrementAndGet();
    }

    public int decRef() {
        return refs.decrementAndGet();
    }

    @Override
    public void close() {
        if (closed) {
//            System.err.println("DOUBLE CLOSE: " + reader);
            return;
        }
        closed = true;
        for (FieldCache cache: perFieldCaches.values()) {
//            System.err.println("Freeing cache: " + cache.fieldname()
//                + '@' + reader);
            cache.free();
        }
    }

    private static int indexOf(final BytesRef ref, byte ch) {
        final int end = ref.offset + ref.length;
        final byte[] buf = ref.bytes;
        for (int i = ref.offset; i < end; i++) {
            if (ch == buf[i]) {
                return i - ref.offset;
            }
        }
        return -1;
    }

    private static final class SingleFieldVisitor extends AbstractFieldVisitor {
        private final BytesRef ref = new BytesRef(0);
        private final String field;

        SingleFieldVisitor(final String field) {
            super(new SingleFieldToIndex(field));
            this.field = field;
        }

        @Override
        public FieldSelectorResult fieldSelectorResult(final String fieldName) {
            if (fieldName.equals(field)) {
                return FieldSelectorResult.LOAD_AND_BREAK;
            } else {
                return FieldSelectorResult.NO_LOAD;
            }
        }

        @Override
        public void storeFieldValue(final int index, final byte[] value) {
            ref.bytes = value;
            ref.offset = 0;
            ref.length = value.length;
//            System.err.println("Store field value<" + index + ">: " + ref.toString());
        }

        @Override
        public void storeFieldSize(final int index, final int size) {
        }

        public BytesRef ref() {
            return ref;
        }

        public void reset() {
            ref.bytes = BytesRef.EMPTY_BYTES;
            ref.length = 0;
        }
    }
}
