package ru.yandex.msearch.index;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.server.sync.HttpEntityEnclosingRequestHandler;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.HttpIndexerServer;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.JournalableMessage;
import ru.yandex.msearch.Message;
import ru.yandex.msearch.MessageContext;
import ru.yandex.msearch.MessageQueueId;
import ru.yandex.msearch.QueueShard;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.util.SumLimiter;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.ScanningCgiParams;

public abstract class BaseIndexHandler
    implements HttpEntityEnclosingRequestHandler
{
    private final SumLimiter limiter;
    private final DatabaseManager dbManager;
    private final HttpIndexerServer server;

    public BaseIndexHandler(
        final DatabaseManager dbManager,
        final SumLimiter limiter,
        final HttpIndexerServer server)
    {
        this.limiter = limiter;
        this.dbManager = dbManager;
        this.server = server;
    }

    protected abstract JournalableMessage parse(
        MessageContext context,
        byte[] dump,
        Charset charset,
        int priority,
        final QueueShard shard,
        final MessageQueueId queueId,
        final DatabaseConfig config,
        final boolean journalable,
        final boolean orderIndependentUpdate)
        throws IOException, ParseException;

    public static MessageQueueId extractQueueId(final HttpRequest request)
        throws BadRequestException
    {
        HeadersParser headers = new HeadersParser(request);
        boolean weakQueueIdCheck = false;
        long longQueueId =
            headers.getLong(
                YandexHeaders.ZOO_QUEUE_ID,
                QueueShard.MAGIC_QUEUEID);
        if (longQueueId == QueueShard.MAGIC_QUEUEID) {
            longQueueId = headers.getLong(
                YandexHeaders.ZOO_QUEUE_ID_TO_CHECK,
                QueueShard.MAGIC_QUEUEID);
            if (longQueueId != QueueShard.MAGIC_QUEUEID) {
                weakQueueIdCheck = true;
            }
        }
        return new MessageQueueId(
            longQueueId,
            new ScanningCgiParams(request).getLong("zoo-queue-id", longQueueId),
            weakQueueIdCheck);
    }

    @Override
    public void handle(
        final HttpEntityEnclosingRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        try {
            boolean wait = true;
            //This is a temporary solution until update/delete race condition will not be fixed.
//                    new ScanningCgiParams(request).getBoolean("wait", true);
            final HttpEntity entity = request.getEntity();
            ContentType contentType = ContentType.get(entity);
            Charset charset = null;
            if (contentType != null) {
                charset = contentType.getCharset();
            }

            if (charset == null) {
                charset = StandardCharsets.UTF_8;
            }

            MessageQueueId queueId = extractQueueId(request);
            final HeadersParser headers = new HeadersParser(request);
            final String service =
                headers.getString(
                    YandexHeaders.ZOO_QUEUE,
                    QueueShard.DEFAULT_SERVICE);
            final int shardId =
                headers.getInt(
                    YandexHeaders.ZOO_SHARD_ID,
                    -1);
            final QueueShard shard = new QueueShard(service, shardId);
            long contentLength = entity.getContentLength();
            Logger logger = (Logger) context.getAttribute(HttpServer.LOGGER);
            if (contentLength == -1L) {
                limiter.waitIdleLimit();
            } else {
                logger.fine("Message size: " + contentLength);
                limiter.acquire(contentLength);
            }
            try {
                byte[] dump = CharsetUtils.toByteArray(entity);
                if (contentLength == -1L) {
                    logger.fine("Message size: " + dump.length);
                    contentLength = dump.length;
                    limiter.acquireAfterIdle(contentLength);
                }
                MessageContext messageContext = new MessageContext() {
                    @Override
                    public Logger logger() {
                        return logger;
                    }
                };
                ScanningCgiParams params = new ScanningCgiParams(request);
                Index index = dbManager.database(params, service);
                if (index == null) {
                    throw new BadRequestException("Can not determine database for indexation " + service + " " + dbManager.databases());
                }

                boolean journalable =
                    index.config().useJournal()
                        && params.getBoolean("journal", true);

                final boolean orderIndependentUpdate = params.getBoolean("order-independent-update",
                        index.config().orderIndependentUpdate());

                try (JournalableMessage message = parse(
                        messageContext,
                        dump,
                        charset,
                        wait ? Message.Priority.ONLINE : Message.Priority.OFFLINE,
                        shard,
                        queueId,
                        index.config(),
                        journalable,
                        orderIndependentUpdate))
                {
                    index.dispatcher(wait).dispatch(HttpIndexerServer.loggingMessage(message, logger));
                }
                server.storeLag(request);
            } finally {
                if (contentLength == -1L) {
                    limiter.releaseIdle();
                } else {
                    limiter.release(contentLength);
                }
            }
        } catch (Throwable t) {
            HttpIndexerServer.handleException(t, response);
        }
    }
}
