package ru.yandex.msearch.index;

import java.io.IOException;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.HttpIndexerServer;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.JsonDeleteMessage;
import ru.yandex.msearch.Message;
import ru.yandex.msearch.MessageContext;
import ru.yandex.msearch.MessageQueueId;
import ru.yandex.msearch.QueueShard;
import ru.yandex.msearch.util.SumLimiter;
import ru.yandex.parser.uri.ScanningCgiParams;

public class DeleteByQueryHandler implements HttpRequestHandler {
    private final SumLimiter limiter;
    private final DatabaseManager dbManager;
    private final HttpIndexerServer server;

    public DeleteByQueryHandler(
        final DatabaseManager dbManager,
        final SumLimiter sumLimiter,
        final HttpIndexerServer server)
    {
        this.dbManager = dbManager;
        this.limiter = sumLimiter;
        this.server = server;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        try {
            MessageQueueId queueId = BaseIndexHandler.extractQueueId(request);
            final HeadersParser headers = new HeadersParser(request);
            final String service =
                headers.getString(
                    YandexHeaders.ZOO_QUEUE,
                    QueueShard.DEFAULT_SERVICE);
            final int shardId =
                headers.getInt(
                    YandexHeaders.ZOO_SHARD_ID,
                    -1);
            final QueueShard shard = new QueueShard(service, shardId);
            final Logger logger = (Logger) context.getAttribute(HttpServer.LOGGER);
            MessageContext messageContext = new MessageContext() {
                @Override
                public Logger logger() {
                    return logger;
                }
            };
            ScanningCgiParams params = new ScanningCgiParams(request);
            Index index = dbManager.database(params, service);
            if (index == null) {
                throw new BadRequestException("Can not determine database for indexation");
            }

            boolean journalable =
                index.config().useJournal()
                    && params.getBoolean("journal", true);

            index.dispatcher(true).dispatch(
                HttpIndexerServer.loggingMessage(
                    new JsonDeleteMessage(
                        messageContext,
                        request.getRequestLine().getUri(),
                        Message.Priority.ONLINE,
                        shard,
                        queueId,
                        index.config(),
                        journalable),
                    logger));
            server.storeLag(request);
        } catch (Throwable t) {
            HttpIndexerServer.handleException(t, response);
        }
    }

    @Override
    public String toString() {
        return "https://wiki.yandex-team.ru/ps/Documentation/Lucene/"
            + "IndexHandlers#/delete";
    }
}
