package ru.yandex.msearch.indexprocessor;

import java.io.IOException;
import java.text.ParseException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.util.StringHelper;

import ru.yandex.msearch.AnalyzerProvider;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.FieldFunctionIOException;
import ru.yandex.msearch.HTMLDocument;
import ru.yandex.msearch.PrimaryKey;
import ru.yandex.msearch.PrimaryKeyPartBase;
import ru.yandex.msearch.PrimaryKeySearcher;
import ru.yandex.msearch.UpdateOnlyHTMLDocument;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.json.dp.join.JoinDocProcessorConfig;
import ru.yandex.search.json.fieldfunction.ConditionsAccessor;
import ru.yandex.search.json.fieldfunction.ConstantValueFieldFunction;
import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.util.string.StringUtils;

public class PrimaryKeyJoinDocProcessor implements IndexDocProcessor {
    private final JoinDocProcessorConfig docConfig;
    private final DatabaseConfig config;
    private final Logger logger;
    private final Set<String> getFields;

    public PrimaryKeyJoinDocProcessor(
        final JoinDocProcessorConfig docConfig,
        final DatabaseConfig config,
        final Logger logger)
        throws IOException
    {
        this.docConfig = docConfig;
        this.config = config;
        this.logger = logger;

        Set<String> verifiedFields =
            new LinkedHashSet<>(docConfig.getFields().size());
        for (String getField: docConfig.getFields()) {
            FieldConfig fieldConfig =
                config.fieldConfigFast(StringHelper.intern(getField));
            if (fieldConfig == null) {
                throw new IOException("Field not found in config " + getField);
            }

            verifiedFields.add(fieldConfig.field());
        }

        this.getFields = verifiedFields;
    }

    @Override
    public void init(final AnalyzerProvider analyzerProvider)
        throws IOException, ParseException
    {
    }

    @Override
    public void process(
        final PrimaryKeyPartBase part,
        final IndexSearcher indexSearcher,
        final ConditionsAccessor conditionsAccessor,
        final HTMLDocument document)
        throws IOException, ParseException
    {
        UpdateOnlyHTMLDocument updateDoc = null;
        String queryStr;
        try {
            if (document instanceof UpdateOnlyHTMLDocument) {
                updateDoc = (UpdateOnlyHTMLDocument) document;
                HTMLDocument converted =
                    new HTMLDocument(
                        updateDoc.updateFields(),
                        document.prefix(),
                        document.phantomQueueId(),
                        document.queueName(),
                        document.orderIndependentUpdate(),
                        config,
                        logger);
                converted.prepare(conditionsAccessor);
                queryStr =
                    docConfig.query()
                        .value(converted, conditionsAccessor)
                        .stringValue();
            } else {
                queryStr =
                    docConfig.query()
                        .value(document, conditionsAccessor).stringValue();
            }
        } catch (FieldFunctionException e) {
            throw new FieldFunctionIOException("Failed to parse subquery ", e);
        }

        String pkName = config.primaryKey().iterator().next();
        PrimaryKey primaryKey =
            PrimaryKey.create(
                Collections.singletonMap(pkName, queryStr),
                document.prefix(),
                config);

        PrimaryKeySearcher result =
            part.findPrimaryKey(primaryKey, indexSearcher, false);

        if (result != null) {
            Document doc;
            if (result.part() == null) {
                doc = result.doc();
            } else {
                doc = result.part().getDocumentByPrimaryKey(primaryKey);
            }

            StringBuilder logSb;
            if (logger.isLoggable(Level.FINE)) {
                logSb = new StringBuilder(getFields.size() * 15);
            } else {
                logSb = null;
            }
            if (updateDoc == null) {
                if (logSb != null) {
                    logSb.append("Doc id ");
                    logSb.append(doc.get(pkName));
                    logSb.append(" Modifying field: ");
                }

                for (String fieldName : getFields) {
                    Fieldable field = doc.getField(fieldName);
                    if (field != null) {
                        if (logSb != null) {
                            logSb.append(fieldName);
                            logSb.append('=');
                            logSb.append(field.stringValue());
                            logSb.append(' ');
                        }
                        document.setField(
                            field.name(),
                            field.stringValue());
                    } else {
                        logger.warning(
                            StringUtils.concat(
                                "Modifying field not found ", fieldName, ' ', String.valueOf(doc)));
                    }
                }
            } else {
                for (String fieldName : getFields) {
                    Fieldable field = doc.getField(fieldName);
                    if (field != null) {
                        if (logSb != null) {
                            logSb.append(fieldName);
                            logSb.append('=');
                            logSb.append(field.stringValue());
                            logSb.append(' ');
                        }
                        updateDoc.updateFields().put(
                            fieldName,
                            new ConstantValueFieldFunction(
                                field.stringValue()));
                    } else {
                        logger.warning(
                            StringUtils.concat(
                                "Updating field not found  ", fieldName, ' ', String.valueOf(doc)));
                    }
                }
            }

            if (logSb != null) {
                logger.fine(logSb.toString());
            }
        } else {
            logger.warning("No doc found for " + queryStr);
        }
    }
}
