package ru.yandex.msearch.jobs;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import ru.yandex.concurrent.NamedThreadFactory;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.IndexGetter;
import ru.yandex.msearch.parallel.ParaWork;
import ru.yandex.msearch.parallel.ParallelExec;

import ru.yandex.search.prefix.PrefixParser;

public class IndexCopyJob extends Job implements Runnable
{
    private static final int MAGIC_MOD = 65534;
    private static final int RETRY_INTERVAL = 10000;
    private final Index index;
    private final Logger logger;
    private final ShardStatus[] shardsStatus;
    private JSONArray shardsStatusJson;
    private int maxThreads;
    private String[] urls;
    private int nextUrl;
    private int userShardStart;
    private int userShardEnd;
    private int timeout;
    private int retryCount;
    private Set<String> shardingFields;
    private boolean lockIndex;
    private boolean checkDups;
    private boolean replace;
    private boolean disableIndexing;
    private ThreadPoolExecutor executor;
    private DownloadLimiter limiter = null;
//    private ParallelExec pexec;

    public IndexCopyJob(
        final File file,
        final JSONObject obj,
        final Index index)
        throws IOException
    {
        super(file, obj);
        this.index = index;
        this.logger = index.logger();
        shardsStatus = new ShardStatus[index.shardsCount()];
        for (int i = 0; i < index.shardsCount(); i++) {
            shardsStatus[i] = ShardStatus.PENDING;
        }
        try {
            loadValues();
        } catch (JSONException e) {
            throw new IOException("Can't load job paramters from file <"
                + file.getName() +">", e);
        }
        nextUrl = 0;
    }

    public void fakeAll() throws IOException {
        try {
            for (int i = 0; i < index.shardsCount(); i++) {
                shardsStatus[i] = ShardStatus.FINISHED;
                shardsStatusJson.put(i, ShardStatus.FINISHED.toString());
            }
            this.status = JobStatus.FINISHED;
//            save();
        } catch (JSONException e) {
            throw new IOException(e);
        }
    }

    public void setPending(final int shard) throws IOException {
        shardsStatus[shard] = ShardStatus.PENDING;
        try {
            shardsStatusJson.put(shard, ShardStatus.PENDING.toString());
        } catch (JSONException e) {
            throw new IOException(e);
        };
        save();
    }

    public int userShardStart() {
        return userShardStart;
    }

    public int userShardEnd() {
        return userShardEnd;
    }

    public void setLimiter(final DownloadLimiter limiter) {
        this.limiter = limiter;
    }

    public DownloadLimiter limiter() {
        return limiter;
    }

    private void loadValues() throws JSONException {
        maxThreads = maxThreads();
        if (json.has("shards_status")) {
            JSONArray arr = json.getJSONArray("shards_status");
            for (int i = 0; i < arr.length() && i < index.shardsCount(); i++) {
                shardsStatus[i] = ShardStatus.valueOf(arr.getString(i));
            }
            shardsStatusJson = arr;
        } else {
            shardsStatusJson = new JSONArray();
            for (int i = 0; i < index.shardsCount(); i++) {
                shardsStatusJson.put(i, shardsStatus[i]);
            }
            json.put("shards_status", shardsStatusJson);
        }
        JSONArray arr = json.getJSONArray("copy_urls");
        urls = new String[arr.length()];
        for (int i = 0; i < urls.length; i++) {
            urls[i] = arr.getString(i);
        }
        userShardStart = json.getInt("start_shard");
        userShardEnd = json.getInt("end_shard");
        timeout = jsonGetInt("timeout", 60 * 60);
        retryCount = jsonGetInt("retry-count", 5);
        shardingFields = new HashSet<String>();
        arr = json.optJSONArray("sharding-fields");
        if (arr == null) {
            shardingFields.add("__prefix");
            shardingFields.add("suid");
        } else {
            for (int i = 0; i < arr.length(); i++) {
                shardingFields.add(arr.getString(i));
            }
        }
        lockIndex = json.optBoolean("lock-index", false);
        checkDups = json.optBoolean("check-dups", false);
        replace = json.optBoolean("replace", false);
        disableIndexing = json.optBoolean("disable-indexing", false);
    }

    @Override
    public synchronized void run() {
	executor = new ThreadPoolExecutor(
	    maxThreads,
	    maxThreads,
	    1,
	    TimeUnit.MINUTES,
	    new LinkedBlockingQueue<Runnable>(),
	    new NamedThreadFactory("IndexCopyJob-"));
	List<Future<Void>> runningJobs = new LinkedList<>();
	List<ShardCopyJob> jobs = new LinkedList<>();
        for (int i = 0; i < index.shardsCount(); i++) {
            ShardCopyJob job = new ShardCopyJob(
                index,
                urls,
                i,
                userShardStart,
                userShardEnd,
                this,
                timeout,
                retryCount,
                shardingFields,
                lockIndex,
                checkDups,
                replace,
                disableIndexing);
            jobs.add(job);
            runningJobs.add(executor.submit(job));
        }
        while (runningJobs.size() > 0 && !stop) {
            Iterator<Future<Void>> iter = runningJobs.iterator();
            while (iter.hasNext()) {
                Future<Void> job = iter.next();
                if (job.isDone()) {
                    iter.remove();
                }
            }
            try {
                wait(1000);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if (stop) {
            for (Future<Void> job : runningJobs) {
                job.cancel(true);
            }
            for (ShardCopyJob job : jobs) {
                job.abort();
            }
            status = JobStatus.STOPPED;
        }
        executor.shutdown();
        executor = null;
        runningJobs = null;
        synchronized (shardsStatus) {
            if (!stop) {
                status = JobStatus.FINISHED;
                for (int i = 0; i < index.shardsCount(); i++) {
                    if (shardsStatus[i] != ShardStatus.FINISHED) {
                        status = JobStatus.FAILED;
                        break;
                    }
                }
            }
        }
    }

    @Override
    public synchronized void stop() {
        stop = true;
        notify();
        while (status == JobStatus.RUNNING) {
            try
            {
                wait(100);
            } catch( Exception e ){};
        }
    }

    private int jsonGetInt(String key, int defaultValue) throws JSONException {
        try {
            return json.getInt(key);
        } catch (JSONException e) {
            json.put(key, defaultValue);
            return defaultValue;
        }
    }

    private int maxThreads()
    {
	int threads = 1;
	try
	{
	    threads = json.getInt( "max_threads" );
	}
	catch( JSONException e )
	{
	    try
	    {
		json.put( "max_threads", threads );
	    }
	    catch( JSONException ign ){}
	}
	return threads;
    }

    public ShardStatus getStatus(final int shard) {
        synchronized (shardsStatus) {
            return shardsStatus[shard];
        }
    }

    public void setStatus(final int shard,
        final ShardStatus status)
        throws IOException
    {
        synchronized (shardsStatus) {
            shardsStatus[shard] = status;
        }
        switch (status) {
            default:
            case IN_PROGRESS:
            case FAILED:
                break;
            case FINISHED:
                synchronized (shardsStatus) {
                    try {
                        shardsStatusJson.put(shard, status.toString());
                        save();
                    } catch (JSONException e) {
                        throw new IOException(e);
                    };
                }
                break;
        }
    }

    private class ShardCopyJob implements Callable<Void> {
        private final Index index;
        private final String[] urls;
        private final int shard;
        private final int userShardStart;
        private final int userShardEnd;
        private final IndexCopyJob parent;
        private final int timeout;
        private final Set<String> shardingFields;
        private final boolean lockIndex;
        private final boolean checkDups;
        private final boolean replace;
        private final boolean disableIndexing;
        private IndexGetter getter = null;
        private boolean abort = false;
        private int retryCount;

        public ShardCopyJob(
            final Index index,
            final String[] urls,
            final int shard,
            final int userShardStart,
            final int userShardEnd,
            final IndexCopyJob parent,
            final int timeout,
            final int retryCount,
            final Set<String> shardingFields,
            final boolean lockIndex,
            final boolean checkDups,
            final boolean replace,
            final boolean disableIndexing)
        {
            this.index = index;
            this.urls = urls;
            this.shard = shard;
            this.userShardStart = userShardStart;
            this.userShardEnd = userShardEnd;
            this.parent = parent;
            this.timeout = timeout;
            this.retryCount = retryCount;
            this.shardingFields = shardingFields;
            this.lockIndex = lockIndex;
            this.checkDups = checkDups;
            this.replace = replace;
            this.disableIndexing = disableIndexing;
        }

        public void abort() {
            Thread.currentThread().interrupt();
            if (getter != null) {
                getter.abort();
            }
        }

        @Override
        public Void call() throws IOException {
            int url = 0;
            if (parent.getStatus(shard) == ShardStatus.FINISHED) {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("ShardCopyJob: skipping shard <" + shard + ">: "
                        + "already copied");
                }
                return null;
            }
            while (!Thread.currentThread().isInterrupted()
                && retryCount-- > 0)
            {
                boolean success = false;
                try {
                    boolean skipCopy = true;
                    if (index.shardsCount() % MAGIC_MOD == 0) {
                        for (int i = userShardStart; i <= userShardEnd; i++) {
                            if (i % index.shardsCount() == shard) {
                                skipCopy = false;
                                break;
                            }
                        }
                    } else {
                        skipCopy = false;
                    }
                    if (!skipCopy) {
                        getter = new IndexGetter(
                            index,
                            urls[url],
                            shard,
                            userShardStart,
                            userShardEnd,
                            timeout,
                            shardingFields,
                            lockIndex,
                            checkDups,
                            replace,
                            disableIndexing,
                            logger);
                        parent.setStatus(shard, ShardStatus.IN_PROGRESS);
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info(
                                "ShardCopyJob: Trying to copy shard <" + shard
                                + "> from <" + urls[url] + "> with timeout="
                                + timeout + ", retriesCount=" + retryCount
                                + ", sharding-fields=" + shardingFields.toString());
                        }
                        if (parent.limiter() != null) {
                            getter.setLimiter(parent.limiter());
                        }
                        getter.get();
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info("ShardCopyJob: Copying shard <" + shard
                                + "> from <"+ urls[url] +"> was succesfull");
                        }
                    } else {
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info(
                                "ShardCopyJob: lucene shard<" + shard
                                    + "> is not in range of ZK shards: "
                                    + userShardStart + "-" + userShardEnd);
                        }
                    }
                    parent.setStatus(shard, ShardStatus.FINISHED);
                    success = true;
                    break;
                } catch (org.apache.lucene.util.ThreadInterruptedException e) {
                    Thread.currentThread().interrupt();
                    parent.setStatus(shard, ShardStatus.FAILED);
                    logger.log(Level.SEVERE, "ShardCopyJob: copy aborted", e);
                } catch (IOException e) {
                    parent.setStatus(shard, ShardStatus.FAILED);
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.log(
                            Level.SEVERE,
                            "ShardCopyJob: Error copying shard <" + shard
                                + "> from <"+ urls[url] +">",
                            e);
                    }
                    url++;
                    if (url >= urls.length) {
                        url = 0;
                    }
                    try {
                        Thread.sleep(RETRY_INTERVAL);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                } catch (Exception e) {
                    parent.setStatus(shard, ShardStatus.FAILED);
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.log(
                            Level.SEVERE,
                            "ShardCopyJob: Error copying shard <" + shard
                                + "> from <"+ urls[url] +">",
                            e);
                        }
                    if (e instanceof InterruptedException) {
                        Thread.currentThread().interrupt();
                        logger.severe("ShardCopyJob: copy aborted");
                        break;
                    }
                    url++;
                    if (url >= urls.length) {
                        url = 0;
                    }
                    try {
                        Thread.sleep(RETRY_INTERVAL * 10);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                } finally {
                    if (success) {
                        parent.setStatus(shard, ShardStatus.FINISHED);
                    } else {
                        parent.setStatus(shard, ShardStatus.FAILED);
                    }
                }
            }
            if (Thread.currentThread().isInterrupted()) {
                logger.severe("ShardCopyJob: copy aborted while");
            }
            return null;
        }
    }

    static enum ShardStatus {
        PENDING {
            @Override
            public String toHtmlString() {
                return "_";
            }
        },
        FAILED {
            @Override
            public String toHtmlString() {
                return "<font color=red>X</font>";
            }
        },
        FINISHED {
            @Override
            public String toHtmlString() {
                return "<font color=green>#</font>";
            }
        },
        IN_PROGRESS {
            @Override
            public String toHtmlString() {
                return "<font color=blue>*</font>";
            }
        };
        public abstract String toHtmlString();
    }

    public ShardStatus shardStatus(final int shard) {
        synchronized (shardsStatus) {
            return shardsStatus[shard];
        }
    }

    @SuppressWarnings("cast")
    public void printStats(final PrintStream ps) {
        float copied = 0;
        synchronized (shardsStatus) {
            for (int i = 0; i < index.shardsCount(); i++) {
                if (shardsStatus[i] == ShardStatus.FINISHED) {
                    copied++;
                }
            }
        }
        ps.println("<html><body>");
        ps.println("progress: " + ((float)(copied / index.shardsCount()) * 100) + " %<br>");
        String status = this.status.toString().toLowerCase();
        ps.println("status: " + status + "<br>");
        ps.println("<table border=1>");
        int x = 0;
        synchronized (shardsStatus) {
            ps.println("<tr><td></td>");
            for (int i = 0; i < 50; i++) {
                ps.println("<td>" + i + "</td>");
            }
            ps.println("</tr>");
            for (int i = 0; i < index.shardsCount(); i++) {
                if (x == 0) {
                    ps.println("<tr><td>" + i + "</td>");
                }
                ps.println("<td>");
                ps.println(shardsStatus[i].toHtmlString());
                ps.println( "</td>" );
                x++;
                if (x >= 50) {
                    x = 0;
                    ps.println("<td>" + i + "</td></tr>");
                }
            }
        }
        ps.println("</table>");
        ps.println("</body></html>");
    }

    @Override
    public float progress() {
        float copied = 0;
        synchronized (shardsStatus) {
            for (int i = 0; i < index.shardsCount(); i++) {
                if (shardsStatus[i] == ShardStatus.FINISHED) {
                    copied++;
                }
            }
        }
        return ((float)(copied / index.shardsCount()) * 100.0F);
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof IndexCopyJob) {
            IndexCopyJob other = (IndexCopyJob)o;
            if (other.userShardStart == userShardStart
                &&  other.userShardEnd == userShardEnd)
            {
                return true;
            } else {
                return false;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return userShardStart | userShardEnd << 16;
    }

    @Override
    public String toString() {
        return "IndexCopy_" + userShardStart + "-" + userShardEnd;
    }

    public void update(Job o) {
        if (!(o instanceof IndexCopyJob)) {
            return;
        }
        IndexCopyJob other = (IndexCopyJob)o;
        maxThreads = other.maxThreads();
        urls = other.urls;
        shardingFields = other.shardingFields;
        System.err.println("IndexCopyJob.update: urls=" + urls
            + ", threads=" + maxThreads
            + ", shardingFields=" + shardingFields);
        try {
            json.put( "max_threads", maxThreads );
            JSONArray arr = new JSONArray();
            for( int i = 0; i < urls.length; i++ )
            {
                arr.put( i, urls[i] );
            }
            json.put( "copy_urls", arr );
            json.put("sharding-fields", shardingFields);
            save();
        } catch (Exception ign) {
        }
    }
}
