package ru.yandex.msearch.jobs;

import java.io.PrintStream;
import java.io.FileInputStream;
import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.OutputStream;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;

import java.util.Collections;
import java.util.Set;

import java.util.logging.Logger;

import org.json.JSONObject;
import org.json.JSONException;
import org.json.JSONArray;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.Index;

import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.search.prefix.PrefixParser;

public abstract class Job implements Runnable {
    protected File jobFile;
    protected JSONObject json;
    protected Thread t;
    protected boolean stop = false;
    protected volatile JobStatus status = JobStatus.PENDING;

    protected Job(File file, JSONObject obj) {
        jobFile = file;
        json = obj;
    }

    public static Job createFakeCopyJob(
        final int startShard,
        final int endShard,
        final Index index)
        throws IOException
    {
        try {
            JSONObject obj = new JSONObject();
            obj.put( "jobType", "IndexCopy" );
            obj.put( "start_shard", startShard );
            obj.put( "end_shard", endShard );
            obj.put( "max_threads", 1 );
            JSONArray arr = new JSONArray();
            arr.put( 0, "fakehost:80" );
            obj.put( "copy_urls", arr );
            obj.put( "timeout", 1 );
            obj.put( "retry-count", 1 );
            obj.put("sharding-fields", Collections.singleton("__prefix"));
            obj.put("lock-index", false);
            obj.put("check-dups", false);
            obj.put("replace", false);
            obj.put("disable-indexing", false);
            IndexCopyJob j =
                new IndexCopyJob(
                    null,
                    obj,
                    index);
            j.fakeAll();
            return j;
        } catch( JSONException e ) {
            throw new IOException(e);
        }
    }

    public static Job createIndexCopyJob(
        Index index,
        int startShard,
        int endShard,
        String[] urls,
        int maxThreads,
        int timeout,
        int retryCount,
        final Set<String> shardingFields,
        final boolean lockIndex,
        final boolean checkDups,
        final boolean replace,
        final boolean disableIndexing)
    {
	try
	{
	    JSONObject obj = new JSONObject();
	    obj.put( "jobType", "IndexCopy" );
	    obj.put( "start_shard", startShard );
	    obj.put( "end_shard", endShard );
	    obj.put( "max_threads", maxThreads );
	    JSONArray arr = new JSONArray();
	    for( int i = 0; i < urls.length; i++ )
	    {
	        arr.put( i, urls[i] );
	    }
	    obj.put( "copy_urls", arr );
	    obj.put( "timeout", timeout );
	    obj.put( "retry-count", retryCount );
	    obj.put("sharding-fields", shardingFields);
	    obj.put("lock-index", lockIndex);
	    obj.put("check-dups", checkDups);
	    obj.put("replace", replace);
	    obj.put("disable-indexing", disableIndexing);
	    Job j = new IndexCopyJob(null, obj, index);
	    return j;
	}
	catch( JSONException e )
	{
	    e.printStackTrace();
	}
	catch( IOException e )
	{
	    e.printStackTrace();
	}
	return null;
    }

    public static Job createShardsRemoverJob(
        Index index,
        int startShard,
        int endShard,
        int maxThreads,
        final Set<String> shardingFields)
    {
	try
	{
	    JSONObject obj = new JSONObject();
	    obj.put( "jobType", "ShardsRemover" );
	    obj.put( "start_shard", startShard );
	    obj.put( "end_shard", endShard );
	    obj.put( "max_threads", maxThreads );
	    obj.put("sharding-fields", shardingFields);
	    Job j = new ShardsRemoverJob(
	        null,
	        obj,
	        index);
	    return j;
	}
	catch( JSONException e )
	{
	    e.printStackTrace();
	}
	catch( IOException e )
	{
	    e.printStackTrace();
	}
	return null;
    }

	public static Job createJob(
		File file,
		JSONObject obj,
		Index index)
		throws IOException
	{
		String jobType;
		try {
			jobType = obj.getString("jobType");
		} catch (JSONException e) {
			throw new IOException("Invalid job file: can't determine jobType");
		}
		if (jobType.equals("IndexCopy")) {
			return new IndexCopyJob(file, obj, index);
		} else if (jobType.equals("ShardsRemover")) {
			return new ShardsRemoverJob(
				file,
				obj,
				index);
		} else {
			throw new IOException("Invalid job file: unknown job type: " + jobType);
		}
	}

	public static Job createJob(
		File file,
		Index index)
		throws IOException {
		String json = readFile(file);
		try {
			JSONObject obj = new JSONObject(json);
			Job job = createJob(file, obj, index);
			return job;
		} catch (JSONException e) {
			throw new IOException("Can't load json file: " + file.getName() + " : " + e.toString());
		}
	}

	private static String readFile(File file) throws IOException {
		InputStreamReader ir = null;
		try {
			final int size = (int) file.length();
			char[] fileContent = new char[size];
			ir = new InputStreamReader(
				new BufferedInputStream(new FileInputStream(file)),
				StandardCharsets.UTF_8);
			ir.read(fileContent);
			return new String(fileContent);
		} finally {
			if (ir != null) {
				ir.close();
			}
		}
	}

    public String jsonString() {
        return json.toString();
    }

    public void save() throws IOException {
        File tmpFile = new File(jobFile + ".tmp");
        try (OutputStream os =
                new BufferedOutputStream(new FileOutputStream(tmpFile)))
        {
            os.write(json.toString(4).getBytes(StandardCharsets.UTF_8));
        } catch(JSONException e) {
            throw new IOException("Can't save job file: " + tmpFile, e);
        }
        Files.move(
            tmpFile.toPath(),
            jobFile.toPath(),
            StandardCopyOption.REPLACE_EXISTING);
    }

    public void setFile( File file )
    {
	this.jobFile = file;
    }

    public synchronized void start()
    {
	if( status != JobStatus.RUNNING )
	{
	    stop = false;
	    status = JobStatus.RUNNING;
	    Thread t = new Thread( this, "Job<" + toString() +">" );
	    t.start();
	}
    }

    public synchronized void stop()
    {
	if( status == JobStatus.RUNNING )
	{
	    stop = true;
	}
    }

    public JobStatus status() {
        return status;
    }

    public void update(Job other) {
    }

    public abstract float progress();

    public File getFile()
    {
	return jobFile;
    }

    static enum JobStatus {
        PENDING,
        RUNNING,
        STOPPED,
        FAILED,
        FINISHED
    }

    public abstract int hashCode();
    public abstract String toString();
    public abstract void printStats( PrintStream ps );
    public abstract boolean equals( Object o );
}
