package ru.yandex.msearch.jobs;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.Index;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class JobsManager {
    private final Map<Job, Job> jobs = new HashMap<>();
    private final File jobsPath;
    private final Index index;
    private final Config config;
    private final DownloadLimiter rateLimiter;
    private final JobsMonitor jobsMonitor;

    public JobsManager(
            final File jobsPath,
            final Index index,
            final Config config,
            final PrefixedLogger logger)
        throws IOException
    {
        this.jobsPath = jobsPath;
        this.index = index;
        this.config = config;
        rateLimiter = new DownloadLimiter(index.config().indexCopyRateLimitMb());
        checkPath();
        jobsMonitor = new JobsMonitor(this, jobsPath.toPath(), logger);
        jobsMonitor.start();
    }

    public static double round(final double value, final int places) {
        if (places < 0) {
            throw new IllegalArgumentException();
        }
        BigDecimal bd = new BigDecimal(value);
        bd = bd.setScale(places, RoundingMode.HALF_UP);
        return bd.doubleValue();
    }

    public DownloadLimiter rateLimiter() {
        return rateLimiter;
    }

    public Stater stater() {
        return new JobsStater();
    }

    public void setPending(final int shard) throws IOException {
        synchronized (jobs) {
            for (Job j: jobs.keySet()) {
                if (j instanceof IndexCopyJob) {
                    ((IndexCopyJob) j).setPending(shard);
                }
            }
        }
    }

    public boolean shardInCopy(
        final int shard,
        final int outerShardStart,
        final int outerShardEnd)
    {
        boolean inCopy = false;
        synchronized (jobs) {
            for (Job job : jobs.keySet()) {
                if (job instanceof IndexCopyJob) {
                    final IndexCopyJob copyJob = (IndexCopyJob) job;
                    final int jobShardStart = copyJob.userShardStart();
                    final int jobShardEnd = copyJob.userShardEnd();
                    if (jobShardStart <= outerShardEnd
                        && jobShardEnd >= outerShardStart
                        && copyJob.shardStatus(shard)
                            != IndexCopyJob.ShardStatus.FINISHED)
                    {
                        inCopy = true;
                        break;
                    }
                }
            }
        }
        return inCopy;
    }

    public boolean shardsCopied(
        final int luceneShard,
        final int outerShardStart,
        final int outerShardEnd)
    {
        boolean copied = true;
        boolean matched = false;
        synchronized (jobs) {
            for (Job job : jobs.keySet()) {
                if (job instanceof IndexCopyJob) {
                    final IndexCopyJob copyJob = (IndexCopyJob) job;
                    final int jobShardStart = copyJob.userShardStart();
                    final int jobShardEnd = copyJob.userShardEnd();
                    if (jobShardStart <= outerShardEnd
                        && jobShardEnd >= outerShardStart)
                    {
                        matched = true;
                        if (copyJob.shardStatus(luceneShard)
                            != IndexCopyJob.ShardStatus.FINISHED)
                        {
                            copied = false;
                            break;
                        }
                    }
                }
            }
        }
        if (!matched) {
            copied = false;
        }
        return copied;
    }

    public boolean shardCopied(
        final int luceneShard,
        final int outerShard)
    {
        boolean copied = true;
        boolean matched = false;
//        System.err.println("check copied: " + luceneShard + ", out: "+ outerShard);
        synchronized (jobs) {
            for (Job job : jobs.keySet()) {
                if (job instanceof IndexCopyJob) {
                    final IndexCopyJob copyJob = (IndexCopyJob) job;
                    final int jobShardStart = copyJob.userShardStart();
                    final int jobShardEnd = copyJob.userShardEnd();
//                    System.err.println("Job: " + jobShardStart + "/" + jobShardEnd);
                    if (jobShardStart <= outerShard
                        && jobShardEnd >= outerShard)
                    {
                        matched = true;
                        if (copyJob.shardStatus(luceneShard)
                            != IndexCopyJob.ShardStatus.FINISHED)
                        {
                            //System.err.println("CopyJob not finished: " + copyJob.shardStatus(luceneShard));
                            copied = false;
                            break;
                        }
                    }
                }
            }
        }
        if (!matched) {
            System.err.println("Matched=" + matched);
            copied = false;
        }
        return copied;
    }

    public boolean indexCopied() {
        int copyFinished = 0;
        int copyJobsTotal = 0;
        synchronized (jobs) {
            for (Job job : jobs.keySet()) {
                if (job instanceof IndexCopyJob) {
                    final Job.JobStatus status = job.status();
                    copyJobsTotal++;
                    switch (status) {
                        case FINISHED:
                            copyFinished++;
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        return copyFinished == copyJobsTotal && copyJobsTotal > 0;
    }

    public List<Job> getJobs() {
        synchronized (jobs) {
            return new ArrayList<Job>(jobs.keySet());
        }
    }

    public void fakeCopy(final int startShard, final int endShard)
        throws IOException
    {
        Job job = Job.createFakeCopyJob(startShard, endShard, index);
        synchronized (jobs) {
            if (!jobs.containsKey(job)) {
                jobs.put(job, job);
                if (job.getFile() == null) {
                    job.setFile(new File(jobsPath, "fakecopy.jobik"));
                }
                job.save();
            }
        }
    }

    public void addJob(File file) {
        try {
            Job job = loadJob(file);
            addJob(job);
        } catch (Exception e) {
            e.printStackTrace();
            file.renameTo(new File(file.getAbsolutePath() + ".corrupted"));
        }
    }

    public Job addJob( Job job ) throws IOException
    {
        synchronized(jobs) {
            if( jobs.containsKey( job ) )
            {
                System.err.println( "Job with the same target allready exists. Will not add duplicate." );
                return jobs.get( job );
            }
            if (job instanceof IndexCopyJob) {
                ((IndexCopyJob) job).setLimiter(rateLimiter);
            }
            jobs.put( job, job );
            if( job.getFile() == null )
            {
                job.setFile( new File( jobsPath, job.toString() + ".jobik" ) );
            }
            job.save();
            return job;
        }
    }

    public Job getJob( Job job )
    {
        synchronized(jobs) {
            return jobs.get( job );
        }
    }

    public void removeJob( Job job ) throws IOException
    {
        synchronized(jobs) {
            job = getJob( job );
            if( job != null )
            {
                job.stop();
                job.getFile().renameTo( new File( job.getFile().getCanonicalPath() + ".removed" ) );
                job.setFile( null );
                jobs.remove( job );
            }
        }
    }

    public void runJobs() throws IOException {
        synchronized(jobs) {
            for (Job j: jobs.keySet()) {
                j.start();
            }
        }
    }

    public void stopJobs() throws IOException {
        synchronized(jobs) {
            for (Job j: jobs.keySet()) {
                j.stop();
            }
        }
    }

    private void checkPath() throws IOException {
        if (!jobsPath.exists()) {
            jobsPath.mkdirs();
        } else if (!jobsPath.isDirectory()) {
            throw new IOException(
                "Job path must be a directory: "
                + jobsPath.getCanonicalPath()
                + '/' + jobsPath.getName());
        }
    }

    private Job loadJob(final File file) throws IOException {
        return Job.createJob(file, index);
    }

    public boolean hasCopyJobs() {
        boolean hasCopyJobs = false;
        synchronized (jobs) {
            for (Job job : jobs.keySet()) {
                if (job instanceof IndexCopyJob) {
                    hasCopyJobs = true;
                    break;
                }
            }
        }
        return hasCopyJobs;
    }

    private String signalNameWithDb(final String name) {
        if (index.config().name() == DatabaseManager.DEFAULT_DATABASE) {
            return name;
        }

        return StringUtils.concat(index.config().name(), '-', name);
    }

    private class JobsStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            float minProgress = 100;
            int copyRunning = 0;
            int copyFinished = 0;
            int copyFailed = 0;
            synchronized (jobs) {
                for (Job job : jobs.keySet()) {
                    if (job instanceof IndexCopyJob) {
                        final float progress = job.progress();
                        if (minProgress > progress) {
                            minProgress = progress;
                        }
                        Job.JobStatus status = job.status();
                        switch (status) {
                            case RUNNING:
                                copyRunning++;
                                break;
                            case FINISHED:
                                copyFinished++;
                                break;
                            case FAILED:
                                copyFailed++;
                                break;
                        }
                    }
                }
            }
            minProgress = (float) round(minProgress, 2);
            int indexCopied = 0;
            int hasCopyJobs = 0;
            int emptyIndex = 1;
            int copyJobsTotal = copyFinished + copyRunning + copyFailed;
            if (copyJobsTotal > 0) {
                hasCopyJobs = 1;
            }
            if (copyFinished > 0 && copyRunning == 0 && copyFailed == 0) {
                indexCopied = 1;
                emptyIndex = 0;
            }
            statsConsumer.stat(
                signalNameWithDb("index-copy-progress_ammt"),
                round(minProgress / 100.0F, 2));
            statsConsumer.stat(
                signalNameWithDb("index-copy-progress_perc_annt"),
                round(minProgress, 2));
            statsConsumer.stat(signalNameWithDb("index-copy-running_ammt"), copyRunning);
            statsConsumer.stat(signalNameWithDb("index-copy-finished_ammt"), copyFinished);
            statsConsumer.stat(signalNameWithDb("index-copy-failed_ammt"), copyFailed);
            statsConsumer.stat(signalNameWithDb("index-copy-total_ammt"), copyJobsTotal);
            statsConsumer.stat(signalNameWithDb("index-copied_ammt"), indexCopied);
            statsConsumer.stat(signalNameWithDb("index-empty-total_ammt"), emptyIndex);
            statsConsumer.stat(signalNameWithDb("index-empty-max_axxt"), emptyIndex);
            statsConsumer.stat(signalNameWithDb("has-copy-jobs_ammt"), hasCopyJobs);
            statsConsumer.stat(signalNameWithDb("has-copy-jobs-min_annt"), hasCopyJobs);
            statsConsumer.stat(
                signalNameWithDb("index-copy-current-rate-mb_ammv"),
                round(rateLimiter.currentRate(), 2));
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

            GolovanChart chart = new GolovanChart(
                "index-copied",
                " backends with full index",
                false,
                false,
                0d);
            chart.addSplitSignal(
                config,
                statsPrefix + signalNameWithDb("index-copied_ammt"),
                0,
                false,
                false);
            group.addChart(chart);

            panel.addCharts(
                "index",
                null,
                group);
        }
    }
}
