package ru.yandex.msearch.jobs;

import java.io.File;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;

import ru.yandex.logger.PrefixedLogger;

/**
 * A background thread which waits for new jobik-s and adds them to {@link JobsManager} for execution.
 * On start schedules all jobs that are found in {@code jobsPath}.
 */
public class JobsMonitor {
    private static final Duration DEFAULT_CHECK_INTERVAL = Duration.ofSeconds(10);

    private final JobsManager jobsManager;
    private final Path jobsPath;
    private final Duration checkInterval;
    private final PrefixedLogger logger;
    private final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

    private long lastModificationTime = 0;

    public JobsMonitor(JobsManager jobsManager, Path jobsPath, PrefixedLogger logger) {
        this(jobsManager, jobsPath, DEFAULT_CHECK_INTERVAL, logger);
    }

    public JobsMonitor(JobsManager jobsManager, Path jobsPath, Duration checkInterval, PrefixedLogger logger) {
        this.jobsManager = jobsManager;
        this.jobsPath = jobsPath;
        this.checkInterval = checkInterval;
        this.logger = logger.addPrefix(this.getClass().getSimpleName());
    }

    public void start() {
        executorService.scheduleWithFixedDelay(
                this::scheduleNewJobs,
                0,
                checkInterval.toMillis(),
                TimeUnit.MILLISECONDS);
    }

    public void stop() {
        if (!executorService.isTerminated()) {
            executorService.shutdown();
            executorService.shutdownNow();
        }
    }

    public void scheduleNewJobs() {
        try {
            File[] jobs = jobsPath.toFile().listFiles();

            if (jobs == null) {
                logger.warning("Can't list files in directory " + jobsPath);
                return;
            }

            List<File> unprocessedJobs = Arrays.stream(jobs)
                    .filter(f -> f.getName().endsWith(".jobik"))
                    .filter(f -> f.lastModified() > lastModificationTime)
                    .sorted(Comparator.comparingLong(File::lastModified))
                    .collect(Collectors.toList());

            for (File unprocessedJob : unprocessedJobs) {
                jobsManager.addJob(unprocessedJob);
                lastModificationTime = unprocessedJob.lastModified();
            }
        } catch (Exception ex) {
            logger.log(Level.WARNING, "An exception occurred", ex);
        }
    }
}
