package ru.yandex.msearch.jobs;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import ru.yandex.concurrent.NamedThreadFactory;

import ru.yandex.msearch.Config;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.IndexGetter;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.parallel.ParaWork;
import ru.yandex.msearch.parallel.ParallelExec;

import ru.yandex.search.prefix.PrefixParser;

public abstract class PerShardJobBase extends Job implements Runnable {
    private static final int RETRY_INTERVAL = 10000;
    protected final Index index;
    private final ShardStatus[] shardsStatus;
    private JSONArray shardsStatusJson;
    private int maxThreads;
    protected int userShardStart;
    protected int userShardEnd;
    protected Set<String> shardingFields;
    private ThreadPoolExecutor executor;
    protected final Logger logger;

    public PerShardJobBase(
        final File file,
        final JSONObject obj,
        final Index index)
        throws IOException
    {
        super(file, obj);
        this.index = index;
        this.logger = index.logger();
        shardsStatus = new ShardStatus[index.shardsCount()];
        for (int i = 0; i < index.shardsCount(); i++) {
            shardsStatus[i] = ShardStatus.PENDING;
        }
        try {
            loadValues();
        } catch (JSONException e) {
            throw new IOException("Can't load job paramters from file <"
                + file.getName() +">", e);
        }
    }

    public Logger logger() {
        return logger;
    }

    private void loadValues() throws JSONException
    {
	maxThreads = maxThreads();
	if( json.has( "shards_status" ) )
	{
	    JSONArray arr = json.getJSONArray( "shards_status" );
	    for( int i = 0; i < arr.length() && i < index.shardsCount(); i++ )
	    {
	        shardsStatus[i] = ShardStatus.valueOf(arr.getString(i));
	    }
	    shardsStatusJson = arr;
	}
	else
	{
	    shardsStatusJson = new JSONArray();
	    for( int i = 0; i < index.shardsCount(); i++ )
	    {
	        shardsStatusJson.put( i, shardsStatus[i] );
	    }
	    json.put( "shards_status", shardsStatusJson );
	}
	userShardStart = json.getInt( "start_shard" );
	userShardEnd = json.getInt( "end_shard" );
	shardingFields = new HashSet<String>();
	JSONArray arr;
	arr = json.optJSONArray("sharding-fields");
	if (arr == null) {
	    shardingFields.add("__prefix");
	    shardingFields.add("suid");
	} else {
	    for (int i = 0; i < arr.length(); i++) {
	        shardingFields.add(arr.getString(i));
	    }
	}
	doLoadValues();
    }

    protected abstract void doLoadValues() throws JSONException;
    protected abstract SingleShardJobBase createJobForShard(final int shard);

    @Override
    public synchronized void run() {
	executor = new ThreadPoolExecutor(
	    maxThreads,
	    maxThreads,
	    1,
	    TimeUnit.MINUTES,
	    new LinkedBlockingQueue<Runnable>(),
	    new NamedThreadFactory(toString() + "-"));
	List<Future<Void>> runningJobs = new LinkedList<>();
	List<SingleShardJobBase> jobs = new LinkedList<>();
        for (int i = 0; i < index.shardsCount(); i++) {
            SingleShardJobBase job = createJobForShard(i);
            jobs.add(job);
            runningJobs.add(executor.submit(job));
        }
        while (runningJobs.size() > 0 && !stop) {
            Iterator<Future<Void>> iter = runningJobs.iterator();
            while (iter.hasNext()) {
                Future<Void> job = iter.next();
                if (job.isDone()) {
                    iter.remove();
                }
            }
            try {
                wait(1000);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if (stop) {
            for (Future<Void> job : runningJobs) {
                job.cancel(true);
            }
            for (SingleShardJobBase job : jobs) {
                job.abort();
            }
            status = JobStatus.STOPPED;
        }
        executor.shutdown();
        executor = null;
        runningJobs = null;
        synchronized (shardsStatus) {
            if (!stop) {
                status = JobStatus.FINISHED;
                for (int i = 0; i < index.shardsCount(); i++) {
                    if (shardsStatus[i] != ShardStatus.FINISHED) {
                        status = JobStatus.FAILED;
                        break;
                    }
                }
            }
        }
    }

    @Override
    public synchronized void stop() {
        stop = true;
        notify();
        while (status == JobStatus.RUNNING) {
            try
            {
                wait(100);
            } catch( Exception e ){};
        }
    }

    private int jsonGetInt(String key, int defaultValue) throws JSONException {
        try {
            return json.getInt(key);
        } catch (JSONException e) {
            json.put(key, defaultValue);
            return defaultValue;
        }
    }

    private int maxThreads()
    {
	int threads = 1;
	try
	{
	    threads = json.getInt( "max_threads" );
	}
	catch( JSONException e )
	{
	    try
	    {
		json.put( "max_threads", threads );
	    }
	    catch( JSONException ign ){}
	}
	return threads;
    }

    public ShardStatus getStatus(final int shard) {
        synchronized (shardsStatus) {
            return shardsStatus[shard];
        }
    }

    public void setStatus(final int shard,
        final ShardStatus status)
        throws IOException
    {
        synchronized (shardsStatus) {
            shardsStatus[shard] = status;
        }
        switch (status) {
            default:
            case IN_PROGRESS:
            case FAILED:
                break;
            case FINISHED:
                synchronized (shardsStatus) {
                    try {
                        shardsStatusJson.put(shard, status.toString());
                        save();
                    } catch (JSONException e) {
                        throw new IOException(e);
                    };
                }
                break;
        }
    }

    private static enum ShardStatus {
        PENDING {
            @Override
            public String toHtmlString() {
                return "_";
            }
        },
        FAILED {
            @Override
            public String toHtmlString() {
                return "<font color=red>X</font>";
            }
        },
        FINISHED {
            @Override
            public String toHtmlString() {
                return "<font color=green>#</font>";
            }
        },
        IN_PROGRESS {
            @Override
            public String toHtmlString() {
                return "<font color=blue>*</font>";
            }
        };
        public abstract String toHtmlString();
    }

    @SuppressWarnings("cast")
    public void printStats(final PrintStream ps) {
        float copied = 0;
        synchronized (shardsStatus) {
            for (int i = 0; i < index.shardsCount(); i++) {
                if (shardsStatus[i] == ShardStatus.FINISHED) {
                    copied++;
                }
            }
        }
        ps.println("<html><body>");
        ps.println("progress: " + ((float)(copied / index.shardsCount()) * 100) + " %<br>");
        String status = this.status.toString().toLowerCase();
        ps.println("status: " + status + "<br>");
        ps.println("<table border=1>");
        int x = 0;
        synchronized (shardsStatus) {
            ps.println("<tr><td></td>");
            for (int i = 0; i < 50; i++) {
                ps.println("<td>" + i + "</td>");
            }
            ps.println("</tr>");
            for (int i = 0; i < index.shardsCount(); i++) {
                if (x == 0) {
                    ps.println("<tr><td>" + i + "</td>");
                }
                ps.println("<td>");
                ps.println(shardsStatus[i].toHtmlString());
                ps.println( "</td>" );
                x++;
                if (x >= 50) {
                    x = 0;
                    ps.println("<td>" + i + "</td></tr>");
                }
            }
        }
        ps.println("</table>");
        ps.println("</body></html>");
    }

    @Override
    public float progress() {
        float copied = 0;
        synchronized (shardsStatus) {
            for (int i = 0; i < index.shardsCount(); i++) {
                if (shardsStatus[i] == ShardStatus.FINISHED) {
                    copied++;
                }
            }
        }
        return ((float)(copied / index.shardsCount()) * 100.0F);
    }

    @Override
    public abstract boolean equals(Object o);

    @Override
    public abstract int hashCode();

    @Override
    public abstract String toString();

    public void update(Job o) {
        if (!(o instanceof PerShardJobBase)) {
            return;
        }
        PerShardJobBase other = (PerShardJobBase)o;
        maxThreads = other.maxThreads();
        try {
            json.put( "max_threads", maxThreads );
            save();
        } catch (Exception ign) {
        }
    }

    static abstract class SingleShardJobBase implements Callable<Void> {
        protected Index index;
        protected DatabaseConfig config;
        protected int shardNo;
        protected int userShardStart;
        protected int userShardEnd;
        protected PerShardJobBase parent;
        protected Set<String> shardingFields;
        protected int retryCount;
        protected final Logger logger;

        public SingleShardJobBase(
            final Index index,
            final int shardNo,
            final int userShardStart,
            final int userShardEnd,
            final Set<String> shardingFields,
            final int retryCount,
            final PerShardJobBase parent)
        {
            this.index = index;
            this.config = index.config();
            this.shardNo = shardNo;
            this.userShardStart = userShardStart;
            this.userShardEnd = userShardEnd;
            this.shardingFields = shardingFields;
            this.retryCount = retryCount;
            this.parent = parent;
            logger = parent.logger();
        }

        public void abort() {
            Thread.currentThread().interrupt();
        }

        protected abstract void doJob() throws IOException;
        protected abstract void onFail();

        @Override
        public Void call() throws IOException {
            int url = 0;
            if (parent.getStatus(shardNo) == ShardStatus.FINISHED) {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Skipping task for shard:" + shardNo
                        + ": " + toString()
                        + ": task already finished");
                }
                return null;
            }
            while (!Thread.currentThread().isInterrupted()
                && retryCount-- > 0)
            {
                boolean success = false;
                try {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Starting task for shard:" + shardNo
                            + ": " + toString());
                    }
                    parent.setStatus(shardNo, ShardStatus.IN_PROGRESS);
                    doJob();
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Finished task for shard:" + shardNo
                            + ": " + toString());
                    }
                    parent.setStatus(shardNo, ShardStatus.FINISHED);
                    success = true;
                    break;
//                } catch (InterruptedException e) {
//                    parent.setStatus(shard, ShardStatus.FAILED);
//                    Logger.err("ShardCopyJob: copy aborted");
                } catch (Exception e) {
                    parent.setStatus(shardNo, ShardStatus.FAILED);
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.log(
                            Level.SEVERE,
                            "Task failed for shard:" + shardNo
                                + ": " + toString()
                                + ": with error",
                            e);
                    }
                    onFail();
                    parent.setStatus(shardNo, ShardStatus.IN_PROGRESS);
                    try {
                        Thread.sleep(RETRY_INTERVAL);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                } finally {
                    if (success) {
                        parent.setStatus(shardNo, ShardStatus.FINISHED);
                    } else {
                        parent.setStatus(shardNo, ShardStatus.FAILED);
                    }
                }
            }
            if (Thread.currentThread().isInterrupted()) {
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Task for shard:" + shardNo
                        + ": " + toString() + ": has been aborted");
                }
            }
            return null;
        }
    }
}
