package ru.yandex.msearch.printkeys;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldSelector;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.index.DocsAndPositionsEnum;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Weight;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;

import ru.yandex.collection.PatternMap;
import ru.yandex.http.server.sync.ContentProducerWriter;
import ru.yandex.http.server.sync.ContentWriter;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.msearch.Config;
import ru.yandex.msearch.DatabaseManager;
import ru.yandex.msearch.HttpRequestContext;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.QueueShard;
import ru.yandex.msearch.SearchHandler;
import ru.yandex.msearch.SearchHandlerBase;
import ru.yandex.msearch.Searcher;
import ru.yandex.msearch.collector.PruningCollector;
import ru.yandex.msearch.collector.YaDoc3;
import ru.yandex.msearch.collector.docprocessor.NullDocProcessor;
import ru.yandex.msearch.collector.postfilter.NullPostFilter;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.msearch.printkeys.PrintKeysParams;
import ru.yandex.msearch.printkeys.PrintKeysWriter;
import ru.yandex.msearch.util.Compress;
import ru.yandex.msearch.util.IOScheduler;
import ru.yandex.msearch.util.IOStater;
import ru.yandex.msearch.util.InvertBits;
import ru.yandex.msearch.util.OrNotBits;
import ru.yandex.msearch.util.SparseBitSet;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.queryParser.QueryParserFactory;
import ru.yandex.queryParser.YandexQueryParserFactory;

public class PrintKeysHandler
    extends SearchHandlerBase
    implements HttpRequestHandler
{
    private static final int BUFFER_THRESHOLD = 1048576;

    private final Map<String, QueryParserFactory> queryParserFactory;

    public PrintKeysHandler(
        final DatabaseManager dbManager,
        final Config config,
        final PatternMap<RequestInfo, IOStater> ioStaters)
    {
        super(dbManager, config, ioStaters);
        Map<String, QueryParserFactory> queryParserFactory = new LinkedHashMap<>();
        for (Map.Entry<String, DatabaseConfig> entry: config.databasesConfigs().entrySet()) {
            queryParserFactory.put(entry.getKey(), new YandexQueryParserFactory(entry.getValue()));
        }
        this.queryParserFactory = Collections.unmodifiableMap(queryParserFactory);
    }

    public Map<String, QueryParserFactory> queryParserFactory() {
        return queryParserFactory;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        CgiParams params = new CgiParams(request);
        Index index = dbManager.indexOrException(params, SearchHandler.BRE_GEN);

        PrintKeysParams printKeyParams =
            new PrintKeysParams(
                queryParserFactory,
                index,
                params,
                new HttpRequestContext(context));
        IOScheduler.setThreadReadPrio(
            params.getInt("IO_PRIO", IOScheduler.IOPRIO_SEARCH));
        IOStater ioStater = ioStaterFor(context);
        Compress.updateSsdCache(params.getBoolean("update-ssd-cache", true));
        EntityTemplate entity = new EntityTemplate(
            new ContentProducerWriter(
                new PrintKeysProducer(index, printKeyParams, ioStater),
                request));
        entity.setChunked(true);
        ContentType contentType;
        if (printKeyParams.jsonType == null) {
            contentType = ContentType.TEXT_PLAIN;
        } else {
            contentType = ContentType.APPLICATION_JSON;
        }
        entity.setContentType(contentType.withCharset(
            CharsetUtils.acceptedCharset(request)).toString());
        String service = params.getString("service", null);
        if (service != null && printKeyParams.user != null) {
            long queueId;
            try {
                queueId = index.queueId(
                    new QueueShard(service, printKeyParams.user),
                    params.getBoolean("check-copyness", false));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            }
            response.setHeader(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.toString(queueId));
        }
        response.setEntity(entity);
    }

    @Override
    public String toString() {
        return "Print index keys for specified field";
    }

    private static class PlainPrintKeysWriter implements PrintKeysWriter {
        private final StringBuilder sb = new StringBuilder();
        private final Writer writer;

        public PlainPrintKeysWriter(final Writer writer) {
            this.writer = writer;
        }

        @Override
        public void close() throws IOException {
            if (sb.length() > 0) {
                writer.append(sb);
                sb.setLength(0);
            }
            writer.close();
        }

        @Override
        public void startTerm(final String term) throws IOException {
            sb.append(term);
        }

        @Override
        public void endTerm() throws IOException {
            sb.append('\n');
            writer.append(sb);
            sb.setLength(0);
        }

        @Override
        public void termFreq(final int freq) throws IOException {
            sb.append(' ');
            sb.append('(');
            sb.append(' ');
            sb.append(freq);
            sb.append(' ');
            sb.append(')');
        }

        @Override
        public void termFreq(final int freq, final int totalFreq)
            throws IOException
        {
            sb.append(' ');
            sb.append('(');
            sb.append(' ');
            sb.append(freq);
            sb.append(' ');
            sb.append(totalFreq);
            sb.append(' ');
            sb.append(')');
        }

        @Override
        public void startDocs() throws IOException {
            sb.append(' ');
            sb.append(':');
        }

        @Override
        public void endDocs() throws IOException {
        }

        @Override
        public void startDoc(final int docId) throws IOException {
            sb.append(' ');
            sb.append(docId);
        }

        @Override
        public void endDoc() throws IOException {
            if (sb.length() > BUFFER_THRESHOLD) {
                writer.append(sb);
                sb.setLength(0);
            }
        }

        @Override
        public void startFields() throws IOException {
            sb.append(' ');
            sb.append('[');
            sb.append(' ');
        }

        @Override
        public void endFields() throws IOException {
            sb.append(']');
        }

        @Override
        public void field(final String field) throws IOException {
            sb.append(field);
            sb.append(' ');
        }

        @Override
        public void startPositions() throws IOException {
            sb.append(' ');
            sb.append('{');
            sb.append(' ');
        }

        @Override
        public void endPositions() throws IOException {
            sb.append('}');
        }

        @Override
        public void pos(final int pos) throws IOException {
            sb.append(pos);
            sb.append(' ');
        }
    }

    private static class JsonPrintKeysWriter implements PrintKeysWriter {
        private final JsonWriter writer;
        private boolean closed = false;

        public JsonPrintKeysWriter(final JsonWriter writer)
            throws IOException
        {
            this.writer = writer;
            writer.startObject();
        }

        @Override
        public void close() throws IOException {
            if (!closed) {
                closed = true;
                writer.endObject();
                writer.close();
            }
        }

        @Override
        public void startTerm(final String term) throws IOException {
            writer.key(term);
            writer.startObject();
        }

        @Override
        public void endTerm() throws IOException {
            writer.endObject();
        }

        @Override
        public void termFreq(final int freq) throws IOException {
            writer.key("freq");
            writer.value(freq);
        }

        @Override
        public void termFreq(final int freq, final int totalFreq)
            throws IOException
        {
            writer.key("freq");
            writer.value(freq);
            writer.key("total-freq");
            writer.value(totalFreq);
        }

        @Override
        public void startDocs() throws IOException {
            writer.key("docs");
            writer.startArray();
        }

        @Override
        public void endDocs() throws IOException {
            writer.endArray();
        }

        @Override
        public void startDoc(final int docId) throws IOException {
            writer.startObject();
            writer.key("docId");
            writer.value(docId);
        }

        @Override
        public void endDoc() throws IOException {
            writer.endObject();
        }

        @Override
        public void startFields() throws IOException {
            writer.key("fields");
            writer.startArray();
        }

        @Override
        public void endFields() throws IOException {
            writer.endArray();
        }

        @Override
        public void field(final String field) throws IOException {
            writer.value(field);
        }

        @Override
        public void startPositions() throws IOException {
            writer.key("positions");
            writer.startArray();
        }

        @Override
        public void endPositions() throws IOException {
            writer.endArray();
        }

        @Override
        public void pos(final int pos) throws IOException {
            writer.value(pos);
        }
    }

    private static class PrintKeysProducer implements ContentWriter {
        private final Index index;
        private final PrintKeysParams params;
        private final IOStater ioStater;
        private final FieldSelector fieldSelector;
        private IndexReader reader;
        private Bits skipDocs;

        public PrintKeysProducer(
            final Index index,
            final PrintKeysParams params,
            final IOStater ioStater)
            throws IOException
        {
            this.index = index;
            this.params = params;
            this.ioStater = ioStater;
            fieldSelector =
                new MapFieldSelector(
                    new ArrayList<>(params.getFieldsConfig.fields()));
        }

        private PrintKeysWriter createWriter(final Writer out)
            throws IOException
        {
            if (params.jsonType == null) {
                return new PlainPrintKeysWriter(out);
            } else {
                return new JsonPrintKeysWriter(params.jsonType.create(out));
            }
        }

        @Override
        public void writeTo(final Writer out) throws IOException {
            Compress.resetStats();
            Searcher searcher;
            if (params.reverse) {
                if (params.shard == null) {
                    searcher = index.getDiskSearcher(params.user);
                } else {
                    searcher = index.getDiskSearcher(params.shard);
                }
            } else {
                if (params.shard == null) {
                    searcher = index.getSearcher(
                        params.user,
                        params.syncSearcher);
                } else {
                    searcher = index.getSearcher(
                        params.shard,
                        params.syncSearcher);
                }
            }
            Query query = params.query;
            SparseBitSet queryDocs;
            if (query == null) {
                queryDocs = null;
            } else {
                queryDocs = new SparseBitSet();
                IndexSearcher indexSearcher = searcher.searcher();
                Weight weight = query.weight(indexSearcher);
                IndexReader.AtomicReaderContext[] leaves =
                    indexSearcher.getTopReaderContext().leaves();
                if (leaves != null) {
                    leaves = leaves.clone();
                    Arrays.sort(
                        leaves,
                        (x, y) -> Long.compare(x.docBase, y.docBase));
                    if (params.docProcessor == NullDocProcessor.INSTANCE
                        && params.postFilter == NullPostFilter.INSTANCE)
                    {
                        for (IndexReader.AtomicReaderContext context: leaves) {
                            DocIdSetIterator docIds = weight.scorer(
                                context,
                                Weight.ScorerContext.def());
                            if (docIds != null) {
                                long docBase = context.docBase;
                                for (int docId = docIds.nextDoc();
                                    docId != DocIdSetIterator.NO_MORE_DOCS;
                                    docId = docIds.nextDoc())
                                {
                                    queryDocs.append(docBase + docId);
                                }
                            }
                        }
                    } else {
                        YaDoc3 doc = new YaDoc3(params);
                        params.fieldVisitor.doc(doc);
                        for (IndexReader.AtomicReaderContext context: leaves) {
                            DocIdSetIterator docIds = weight.scorer(
                                context,
                                Weight.ScorerContext.def());
                            if (docIds != null) {
                                long docBase = context.docBase;
                                IndexReader reader = context.reader;
                                for (int docId = docIds.nextDoc();
                                    docId != DocIdSetIterator.NO_MORE_DOCS;
                                    docId = docIds.nextDoc())
                                {
                                    doc.clear();
                                    reader.readDocument(
                                        docId,
                                        params.fieldVisitor);
                                    if (params.docProcessor
                                        .processWithFilter(doc)
                                        && params.postFilter.test(doc))
                                    {
                                        queryDocs.append(docBase + docId);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            reader = searcher.reader();
            try {
                Fields fields = MultiFields.getFields(reader);
                if (fields != null) {
                    Terms terms = fields.terms(params.field);
                    if (terms != null) {
                        if (params.skipDeleted) {
                            skipDocs = MultiFields.getDeletedDocs(reader);
                            if (queryDocs != null) {
                                if (skipDocs == null) {
                                    skipDocs = new InvertBits(queryDocs);
                                } else {
                                    skipDocs =
                                        new OrNotBits(skipDocs, queryDocs);
                                }
                            }
                        } else {
                            // queryDocs != null => skipDeleted
                            skipDocs = null;
                        }

                        PrintKeysWriter writer = createWriter(out);
                        for (BytesRef prefix: params.prefixRefs) {
                            processPrefix(writer, terms, prefix);
                        }
                        writer.close();
                    }
                }
            } finally {
                searcher.free();
                params.docProcessor.after();
                accountStats(ioStater, params.ctx.logger());
            }
        }

        private void processPrefix(
            final PrintKeysWriter writer,
            final Terms terms,
            final BytesRef prefix)
            throws IOException
        {
            TermsEnum termsEnum;
            if (params.reverse) {
                termsEnum = terms.reverseIterator();
            } else {
                termsEnum = terms.iterator();
            }
            BytesRef term;
            if (prefix == null) {
                term = termsEnum.next();
            } else {
                if (params.reverse) {
                    BytesRef seekPrefix = new BytesRef(prefix);
                    seekPrefix.append(PruningCollector.REVERSE_SEEK_SUFFIX);
                    termsEnum.seek(seekPrefix);
                } else {
                    termsEnum.seek(prefix);
                }
                term = termsEnum.term();
            }
            long pos = 0;
            DocsEnum docsEnum = null;
            DocsAndPositionsEnum docsAndPosEnum = null;
            final int docsMax = params.docsOffset + params.docsLength;
            final int docsOffset = params.docsOffset;
            while (term != null
                && pos < params.offset + params.length
                && (prefix == null
                    || term.startsWith(prefix)))
            {
                if ((pos >= params.offset || skipDocs != null)
                    && (!params.exact || term.endsWith((byte) '"')))
                {
                    int freq;
                    int totalFreq = 0;
                    if (params.printTotalFreqs || skipDocs != null) {
                        // Check that at least on doc position is not deleted
                        // for this term, otherwise skip term
                        freq = 0;
                        docsEnum = termsEnum.docs(skipDocs, docsEnum);
                        for (int docId = docsEnum.nextDoc();
                            docId != DocIdSetIterator.NO_MORE_DOCS;
                            docId = docsEnum.nextDoc())
                        {
                            ++freq;
                            totalFreq += docsEnum.freq();
                            if (freq == params.maxFreq) {
                                break;
                            }
                        }
                        if (freq == 0) {
//                            ++pos;
                            term = termsEnum.next();
                            continue;
                        } else if (pos < params.offset) {
                            ++pos;
                            term = termsEnum.next();
                            continue;
                        }
                    } else if (params.printFreqs) {
                        freq = termsEnum.docFreq();
                    } else {
                        freq = 0;
                    }
                    writer.startTerm(term.utf8ToString());
                    if (params.printTotalFreqs) {
                        writer.termFreq(freq, totalFreq);
                    } else if (params.printFreqs) {
                        writer.termFreq(freq);
                    }
                    if (params.printDocs) {
                        writer.startDocs();
                        if (params.printPos) {
                            docsAndPosEnum = termsEnum.docsAndPositions(
                                skipDocs,
                                docsAndPosEnum);
                            int docsCount = 0;
                            for (int docId = docsAndPosEnum.nextDoc();
                                docId != DocIdSetIterator.NO_MORE_DOCS
                                    && docsCount < docsMax;
                                docId = docsAndPosEnum.nextDoc(),
                                    docsCount++)
                            {
                                if (docsCount < docsOffset) {
                                    continue;
                                }
                                writer.startDoc(docId);
                                if (!params.getFieldsConfig.isEmpty()) {
                                    writer.startFields();
                                    Document doc = reader.document(
                                        docId,
                                        fieldSelector);
                                    for (String field
                                        : params.getFieldsConfig.fields())
                                    {
                                        writer.field(doc.get(field));
                                    }
                                    writer.endFields();
                                }
                                freq = docsAndPosEnum.freq();
                                writer.startPositions();
                                for (int i = 0; i < freq; ++i) {
                                    writer.pos(
                                        docsAndPosEnum.nextPosition());
                                }
                                writer.endPositions();
                                writer.endDoc();
                            }
                        } else {
                            docsEnum = termsEnum.docs(skipDocs, docsEnum);
                            int docsCount = 0;
                            for (int docId = docsEnum.nextDoc();
                                docId != DocIdSetIterator.NO_MORE_DOCS
                                    && docsCount < docsMax;
                                docId = docsEnum.nextDoc(),
                                    docsCount++)
                            {
                                if (docsCount < docsOffset) {
                                    continue;
                                }
                                writer.startDoc(docId);
                                if (!params.getFieldsConfig.isEmpty()) {
                                    writer.startFields();
                                    Document doc = reader.document(
                                        docId,
                                        fieldSelector);
                                    for (String field
                                        : params.getFieldsConfig.fields())
                                    {
                                        writer.field(doc.get(field));
                                    }
                                    writer.endFields();
                                }
                                writer.endDoc();
                            }
                        }
                        writer.endDocs();
                    }
                    writer.endTerm();
                }
                ++pos;
                term = termsEnum.next();
            }
        }
    }
}

