package ru.yandex.msearch.printkeys;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.lucene.document.MapFieldSelector;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.msearch.FieldConfig;
import ru.yandex.msearch.GetFieldsConfig;
import ru.yandex.msearch.Index;
import ru.yandex.msearch.NewSearchRequest;
import ru.yandex.msearch.ProcessorRequestContext;
import ru.yandex.msearch.RequestContext;
import ru.yandex.msearch.collector.CollectingFieldToIndex;
import ru.yandex.msearch.collector.FieldToIndex;
import ru.yandex.msearch.collector.MergeFunc;
import ru.yandex.msearch.collector.SearchParams;
import ru.yandex.msearch.collector.YaDocFieldVisitor;
import ru.yandex.msearch.collector.docprocessor.DocProcessor;
import ru.yandex.msearch.collector.docprocessor.ModuleFieldsAggregator;
import ru.yandex.msearch.collector.docprocessor.NullDocProcessor;
import ru.yandex.msearch.collector.group.GroupFunc;
import ru.yandex.msearch.collector.group.NullGroupFunc;
import ru.yandex.msearch.collector.postfilter.NullPostFilter;
import ru.yandex.msearch.collector.postfilter.PostFilter;
import ru.yandex.msearch.collector.sort.SortFunc;
import ru.yandex.msearch.config.DatabaseConfig;
import ru.yandex.parser.string.IntegerParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.queryParser.QueryParserFactory;
import ru.yandex.search.prefix.Prefix;

public class PrintKeysParams implements SearchParams {
    final DatabaseConfig config;
    final Index index;
    final RequestContext ctx;
    final List<BytesRef> prefixRefs;
    final Integer shard;
    final Prefix user;
    final Query query;
    final String field;
    final long offset;
    final long length;
    final DocProcessor docProcessor;
    final GetFieldsConfig getFieldsConfig;
    final PostFilter postFilter;
    final FieldToIndex fieldToIndex;
    final YaDocFieldVisitor fieldVisitor;
    final JsonType jsonType;
    final boolean printPos;
    final boolean printDocs;
    final boolean printFreqs;
    final boolean printTotalFreqs;
    final boolean skipDeleted;
    final int docsOffset;
    final int docsLength;
    final boolean reverse;
    final int maxFreq;
    final boolean exact;
    final boolean syncSearcher;

    public PrintKeysParams(
        final Map<String, QueryParserFactory> queryParserFactory,
        final Index index,
        final CgiParams params,
        final RequestContext ctx)
        throws HttpException
    {
        this.config = index.config();
        this.index = index;
        this.ctx = ctx;
        boolean checkCopyness =
            params.getBoolean("check-copyness", config.checkCopyness());
        syncSearcher =
            params.getBoolean(
                "sync-searcher",
                config.syncSearcherDefault());
        shard = params.get("shard", null, IntegerParser.INSTANCE);
        user = params.get("user", null, index.config().prefixParser());
        Set<String> generatedFields = new HashSet<>();
        CollectingFieldToIndex fieldToIndex = new CollectingFieldToIndex();
        String text = params.getString("text", null);
        if (text == null) {
            query = null;
            docProcessor = NullDocProcessor.INSTANCE;
            getFieldsConfig = NewSearchRequest.extractGetFieldsConfig(
                config,
                Collections.emptySet(),
                params);
            postFilter = NullPostFilter.INSTANCE;
        } else {
            List<Prefix> prefixes;
            if (user == null) {
                prefixes = Collections.emptyList();
            } else {
                prefixes = Collections.singletonList(user);
            }
            ProcessorRequestContext processorRequestContext =
                new ProcessorRequestContext(
                    prefixes,
                    index,
                    ctx,
                    fieldToIndex,
                    params);
            query =
                NewSearchRequest.parseQueries(
                        config,
                        processorRequestContext,
                        params,
                        queryParserFactory.get(index.config().name()),
                        prefixes)
                    .get(0)
                    .query;

            docProcessor = NewSearchRequest.extractDocProcessor(
                config,
                generatedFields,
                params,
                processorRequestContext);

            getFieldsConfig = NewSearchRequest.extractGetFieldsConfig(
                config,
                generatedFields,
                params);

            postFilter = NewSearchRequest.extractPostFilter(
                config,
                generatedFields,
                params,
                fieldToIndex);
        }
        Set<String> readFields = new HashSet<>(getFieldsConfig.fields());
        for (String field : readFields) {
            fieldToIndex.indexFor(field);
        }
        for (String field : postFilter.loadFields()) {
            if (!generatedFields.contains(field)) {
                NewSearchRequest.checkStoredField(config, field);
                readFields.add(field);
                fieldToIndex.indexFor(field);
            }
        }

        ModuleFieldsAggregator fieldsAggregator =
            new ModuleFieldsAggregator();
        docProcessor.apply(fieldsAggregator);
        for (String field : fieldsAggregator.loadFields()) {
            FieldConfig fieldConfig = config.fieldConfig(field);
            if (fieldConfig != null && fieldConfig.store()) {
                readFields.add(field);
            }
        }
        for (String field : fieldsAggregator.outFields()) {
            FieldConfig fieldConfig = config.fieldConfig(field);
            if (fieldConfig == null || !fieldConfig.store()) {
                readFields.remove(field);
            }
        }

        this.fieldToIndex = fieldToIndex.compactFieldToIndex();
        FieldConfig[] loadFields =
            new FieldConfig[this.fieldToIndex.fieldsCount()];
        for (String field : readFields) {
            loadFields[this.fieldToIndex.indexFor(field)] =
                config.fieldConfigFast(field);
        }
        fieldVisitor = new YaDocFieldVisitor(
            new MapFieldSelector(readFields),
            this.fieldToIndex,
            loadFields);

        List<String> prefixes = params.getAll("prefix");
        prefixRefs = new ArrayList<>(prefixes.size());
        for (String prefix : prefixes) {
            prefixRefs.add(prefix == null ? null : new BytesRef(prefix));
        }
        if (prefixes.size() == 0) {
            prefixRefs.add(null);
        }
        field = StringHelper.intern(params.getString("field"));

        offset = params.get(
            "offset",
            0L,
            NonNegativeLongValidator.INSTANCE);
        length = params.get(
            "length",
            Long.MAX_VALUE,
            PositiveLongValidator.INSTANCE);
        jsonType = JsonTypeExtractor.NULL.extract(params);
        printPos = params.getBoolean("print-pos", false);
        printDocs = printPos
                        || !getFieldsConfig.isEmpty()
                        || params.getBoolean("print-docs", false);
        printFreqs = params.getBoolean("print-freqs", printDocs);
        printTotalFreqs = params.getBoolean("print-total-freqs", false);
        skipDeleted =
            query != null || params.getBoolean("skip-deleted", false);
        docsOffset = params.getInt("docs-offset", 0);
        docsLength = params.getInt("docs-length", Integer.MAX_VALUE);
        reverse = params.getBoolean("reverse", false);
        maxFreq = params.get(
            "max-freq",
            1,
            NonNegativeIntegerValidator.INSTANCE);
        exact = params.getBoolean("exact", false);
    }

    @Override
    public GroupFunc groupFunc() {
        return NullGroupFunc.INSTANCE;
    }

    @Override
    public SortFunc sortFunc() {
        return null;
    }

    @Override
    public FieldToIndex fieldToIndex() {
        return fieldToIndex;
    }

    @Override
    public Set<String> mergeFields() {
        return Collections.emptySet();
    }

    @Override
    public MergeFunc mergeFunc() {
        return null;
    }

    public DatabaseConfig config() {
        return config;
    }

    public Index index() {
        return index;
    }

    public RequestContext ctx() {
        return ctx;
    }

    public List<BytesRef> prefixRefs() {
        return prefixRefs;
    }

    public Integer shard() {
        return shard;
    }

    public Prefix user() {
        return user;
    }

    public Query query() {
        return query;
    }

    public String field() {
        return field;
    }

    public long offset() {
        return offset;
    }

    public long length() {
        return length;
    }

    public DocProcessor docProcessor() {
        return docProcessor;
    }

    public GetFieldsConfig getFieldsConfig() {
        return getFieldsConfig;
    }

    public PostFilter postFilter() {
        return postFilter;
    }

    public YaDocFieldVisitor fieldVisitor() {
        return fieldVisitor;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public boolean printPos() {
        return printPos;
    }

    public boolean printDocs() {
        return printDocs;
    }

    public boolean printFreqs() {
        return printFreqs;
    }

    public boolean printTotalFreqs() {
        return printTotalFreqs;
    }

    public boolean skipDeleted() {
        return skipDeleted;
    }

    public int docsOffset() {
        return docsOffset;
    }

    public int docsLength() {
        return docsLength;
    }

    public boolean reverse() {
        return reverse;
    }

    public int maxFreq() {
        return maxFreq;
    }

    public boolean exact() {
        return exact;
    }

    public boolean syncSearcher() {
        return syncSearcher;
    }
}
